/**
 * @file module.cpp 
 * @brief Implementation of members of Module and ModuleProvider
 * 
 * @author Vojtěch Dvořák    
 */


#include "headers/common.h"

#include "headers/module.h"

ModuleProvider::ModuleMap  ModuleProvider::modules_;


ModuleProvider::ModuleProvider() {
}


ModuleProvider::~ModuleProvider() {
    DEBUG_LOG("Deleting module provider...\n");
}


ModuleProvider &ModuleProvider::getInstance() {
    static ModuleProvider instance; ///< Singleton instance (it is destroyed automatically)
    return instance;
}


void ModuleProvider::addModule(const std::shared_ptr<Module> &module) {
    const std::string &module_name = module->getName();

    if(ModuleProvider::hasModule(module_name)) { // If there is already module with the same name, merge its attributes
        ModuleProvider::getModule(module_name)->merge(std::move(module));

        #ifdef USE_MODULE_CACHE
            cacheSymbol({}, getModule(module_name)->getData().get());
        #endif
    }
    else {
        #ifdef USE_MODULE_CACHE
            cacheSymbol({}, module->getData().get());
        #endif

        modules_[module_name] = std::move(module);
    }
}


bool ModuleProvider::hasModule(const std::string &name) const {
    return modules_.find(name) != modules_.end();
}


bool ModuleProvider::hasModule(std::string_view name) const {
    return modules_.find(name) != modules_.end();
}


std::shared_ptr<Module> ModuleProvider::getModule(const std::string &name) const {
    auto result_it = modules_.find(name);
    return result_it == modules_.end() ? nullptr : result_it->second;
}


std::shared_ptr<Module> ModuleProvider::getModule(std::string_view name) const {
    auto result_it = modules_.find(name);
    return result_it == modules_.end() ? nullptr : result_it->second;
}


void ModuleProvider::clearCache() {
    symbol_cache_.clear();
}


const json *ModuleProvider::findCached(std::string_view full_symbol_name) const {
    auto result_it = symbol_cache_.find(full_symbol_name);

    if(result_it == symbol_cache_.end()) {
        return nullptr;
    }
    else {
        return result_it->second;
    }
}


void ModuleProvider::cacheSymbol(const std::string &parent_namespace, const json *symbol_ctx) {
    const json *name = Module::findField(symbol_ctx, "name");
    if(!name) {
        return;
    }

    std::string full_name = parent_namespace;
    if(!parent_namespace.empty()) {
        full_name += "."; // Add separator if it is not the top level symbol
    }
    full_name += name->get<std::string>();
    
    symbol_cache_[full_name] = symbol_ctx; // Save the symbol to the module cache
    
    if(!symbol_ctx->count("kind") || !symbol_ctx->at("kind").is_string()) {
        return;
    }
    
    if(symbol_ctx->at("kind") == "struct") { // If the symbol is a structure, creates entry for its attributes
        const json *attributes = Module::findField(symbol_ctx, "attributes");
        if(!attributes || !attributes->is_array()) {
            return;
        }

        for(auto &attr: *attributes) {
            cacheSymbol(full_name, &attr);
        }
    }

    //DEBUG_LOG("Cached symbol '%s'\n", full_name.c_str());
}



// Module

Module::Module(const std::string &path) : path_(path) {
}


Module::Module(std::string_view path) : path_(path) {
}



const std::string &Module::getName() const {
    return name_;
}


void Module::load() {
    std::ifstream file(path_);

    try {
        data_ = std::make_unique<json>(json::parse(file));
    }
    catch(json::parse_error &) {
        throw YaramodErrorException("Malformed file '" + path_ + "'. Modules must be specified in valid JSON file.");
    }

    name_ = std::string(*findField(data_.get(), "name"));

    file.close();

    DEBUG_LOG("Module: '%s' loaded\n", name_.c_str());
}


void Module::merge(const std::shared_ptr<Module> &other) {
    json new_attr;

    try {
        new_attr = other->getData()->at("attributes"); // Get content of attribute field in foreign module
    }
    catch(nlohmann::detail::out_of_range const&) {
        throw YaramodErrorException("'attributes' field was not found in module " + other->getName());
    } 

    try {
        mergeJson(data_.get(), other->getData().get());
    }
    catch(nlohmann::detail::out_of_range const&) {
        throw YaramodErrorException("'attributes' field was not found in module " + getName());
    } 
}


void Module::mergeJson(json *dst, const json *src) {
    if(dst->is_object() && src->is_object()) {
        for(auto &field: src->items()) {
            if(dst->count(field.key())) { ///< If item is in the both files merge it recursively 
                mergeJson(&(dst->at(field.key())), &(src->at(field.key())));
            }
            else { ///< If it is not in dst object add it
                dst->push_back({ field.key(), field.value() });
            }
        }
    }
    else if(dst->is_array() && src->is_array()) { ///< JSON object is array type
        for(auto &obj: *src) {
            if(obj.is_object()) { ///< Only objects are processed (there are not directly nested arrays in modules)

                if(obj.count("name") && obj["name"].is_string()) {
                    
                    auto dst_obj_it = dst->begin();
                    for(; dst_obj_it != dst->end(); ++dst_obj_it) { ///< Ty to find corresponding object in dst
                        auto dst_obj = *dst_obj_it;
                        if(!dst_obj.count("name") || !dst_obj["name"].is_string()) {
                            continue;
                        }

                        if(dst_obj["name"] == obj["name"]) { 
                            mergeJson(&(*dst_obj_it), &(obj));
                            break;
                        }
                    }

                    if(dst_obj_it == dst->end()) {
                        dst->push_back(obj);
                    }
                }
                else if(!obj.count("name")) {
                    dst->push_back(obj);
                }


            }
        }


    }
}


const std::unique_ptr<json> &Module::getData() const {
    return data_;
} 


const json *Module::findInArray(const json *json, std::string_view obj_name) {
    for(auto &obj: *json) {
        if(!obj.count("name") || !obj.at("name").is_string()) {
            continue;
        }

        if(obj["name"] == obj_name) {
            return &obj;
        }
    }

    return nullptr;
}


const json *Module::findField(const json *json, std::string_view field_name) {
    if(!json->is_object()) {
        throw YaramodErrorException("Module error! JSON object expected!");
    }

    try {
        return &(json->at(field_name)); // Get content of attribute field in foreign module
    }
    catch(nlohmann::detail::out_of_range const&) {
        throw YaramodErrorException("Module error! Field '" + std::string(field_name) + "' was not found!");
    } 
}


Expression::Type Module::jsonToExprType(const json *json_type) {
    if(*json_type == "s") {
        return Expression::Type::String;
    }
    else if(*json_type == "i") {
        return Expression::Type::Int;
    }
    else if(*json_type == "f") {
        return Expression::Type::Float;
    }
    else if(*json_type == "b") {
        return Expression::Type::Bool;
    }
    else if(*json_type == "r") {
        return Expression::Type::Regexp;
    }
    else {
        return Expression::Type::Undefined;
    }
}


Expression::Type Module::getDataType(const json *ctx_obj) {
    std::string obj_name = findField(ctx_obj, "name")->dump();
    if(!ctx_obj->count("kind")) {
        throw YaramodErrorException("Module error! Missing 'kind' field in JSON object " + obj_name);
    }

    if(ctx_obj->at("kind") == "struct") {
        return Expression::Type::Object;
    }
    else if(ctx_obj->at("kind") == "value" || 
        ctx_obj->at("kind") == "dictionary") {
        if(!ctx_obj->count("type")) {
            return Expression::Type::Undefined;
        }
        else {
            return jsonToExprType(findField(ctx_obj, "type"));
        }
    }
    else if(ctx_obj->at("kind") == "array") {
        if(!ctx_obj->count("structure")) {
            return Expression::Type::Undefined;
        }
        else {
            return getDataType(findField(ctx_obj, "structure")); ///< Data type is stored in object in 'structure' field
        }
    }
    else if(ctx_obj->at("kind") == "function") {
        if(ctx_obj->count("return_type")) {
            return jsonToExprType(findField(ctx_obj, "return_type"));
        }
        else {
            return Expression::Type::Undefined;
        }
    }
    else {
        return Expression::Type::Undefined;
    }
}


const json *Module::qualifyObject(std::string_view qual_str, const json *obj) {
    if(!obj->count("attributes")) {
        throw YaramodErrorException("Module error! Bad reference!");
    }

    std::string_view top_lvl_id = qual_str.substr(0, qual_str.find(".")); ///< Get top level name
    
    auto attributes = findField(obj, "attributes");
    auto result = findInArray(attributes, top_lvl_id); ///< Try to find top level name in attributes
    
    if(qual_str.find(".") == std::string::npos) { ///< Top level is also the last id in qual_str
        if(result->count("structure")) {
            return findField(result, "structure");
        }

        return result;
    }
    else {
        std::string_view new_qualifier = qual_str.substr( ///< Update qualifier string
            qual_str.find(".") + 1, 
            qual_str.length() - qual_str.find(".")
        );

        return qualifyObject(new_qualifier, result);
    }
}


Expression::Type Module::getElementDataType(const json *ctx_obj) {
    if(!ctx_obj->count("structure")) {
        return Expression::Type::Undefined;
    }
    else {
        return getDataType(findField(ctx_obj, "structure"));
    }
}


Symbol::Type Module::getElementType(const json *ctx_obj) {
    if(!ctx_obj->count("structure")) {
        return Symbol::Type::Unknown;
    }
    else {
        if(!ctx_obj->at("structure").count("kind")) {
            return Symbol::Type::Unknown;
        }
        else {
            if(ctx_obj->at("structure").at("kind") == "struct") {
                return Symbol::Type::Struct;
            }
            else if(ctx_obj->at("structure").at("kind") == "value") {
                return Symbol::Type::Value;
            }
            else if(ctx_obj->at("structure").at("kind") == "array") {
                return Symbol::Type::Array;
            }
            else if(ctx_obj->at("structure").at("kind") == "dictionary") {
                return Symbol::Type::Dict;
            }
            else if(ctx_obj->at("structure").at("kind") == "function") {
                return Symbol::Type::Function;
            }
            else if(ctx_obj->at("structure").at("kind") == "reference") { ///< If element is reference kind, the type must be qualified
                std::string ref = *findField(&(ctx_obj->at("structure")), "type");

                std::string_view top_lvl = ref.substr(0, ref.find(".")); ///< Get top level id (module name)
                const ModuleProvider &mp = ModuleProvider::getInstance();

                auto mod = mp.getModule(top_lvl); ///< Try to find module with this top level id
                if(mod) {
                    std::string_view qualifiers = ref.substr(ref.find(".") + 1, ref.length());
                    auto refered_ctx = qualifyObject(qualifiers, mod->getData().get()); ///< Do qualification
                    return getElementType(refered_ctx);
                }
                else {
                    return Symbol::Type::Unknown;
                }
            }
            else {
                return Symbol::Type::Unknown;
            }
        }


    }
}


void Module::buildModuleContext(const std::shared_ptr<Symbol> &symbol, const json *ctx_obj) {
    std::string obj_name = findField(ctx_obj, "name")->dump();
    if(!ctx_obj->count("kind")) {
        throw YaramodErrorException("Module error! Missing 'kind' field in JSON object " + obj_name);
    }

    if(ctx_obj->at("kind") == "struct") {
        symbol->setType(Expression::Type::Object);
        symbol->setSymType(Symbol::Type::Struct);
        symbol->setContext(ctx_obj);
    }
    else if(ctx_obj->at("kind") == "value") {
        if(!ctx_obj->count("type")) {
            symbol->setType(Expression::Type::Undefined);
        }
    
        symbol->setSymType(Symbol::Type::Value);
        symbol->setType(jsonToExprType(findField(ctx_obj, "type")));
    }
    else if(ctx_obj->at("kind") == "dictionary") {
        if(!ctx_obj->count("type")) {
            symbol->setType(Expression::Type::Undefined);
        }

        symbol->setSymType(Symbol::Type::Dict);
        symbol->setType(jsonToExprType(findField(ctx_obj, "type")));
    }
    else if(ctx_obj->at("kind") == "array") {
        if(!ctx_obj->count("structure")) {
            symbol->setType(Expression::Type::Undefined);
        }
        else {
            symbol->setType(getDataType(findField(ctx_obj, "structure")));
        }

        symbol->setContext(ctx_obj);
        symbol->setSymType(Symbol::Type::Array);
    }
    else if(ctx_obj->at("kind") == "function") {
        if(ctx_obj->count("return_type")) {
            symbol->setType(jsonToExprType(findField(ctx_obj, "return_type")));
        }
        else {
            symbol->setType(Expression::Type::Undefined);
        }

        symbol->setSymType(Symbol::Type::Function);
        symbol->setContext(ctx_obj);
    }
    else if(ctx_obj->at("kind") == "reference") {
        if(!ctx_obj->count("type") || !findField(ctx_obj, "type")->is_string()) {
            symbol->setType(Expression::Type::Undefined);
        }
        else {
            std::string ref = *findField(ctx_obj, "type");

            const ModuleProvider &mp = ModuleProvider::getInstance();
            #ifdef USE_MODULE_CACHE
                if(auto cached = mp.findCached(ref)) {
                    buildModuleContext(symbol, cached);
                    return;
                }
            #endif

            std::string_view top_lvl = ref.substr(0, ref.find(".")); ///< Get top level id (the name of module)
            auto mod = mp.getModule(top_lvl); ///< Try to find corresponding module
            if(mod) {
                std::string_view qualifiers = ref.substr(ref.find(".") + 1, ref.length());
                auto referenced_ctx = qualifyObject(qualifiers, mod->getData().get()); ///< Qualify reference type
                buildModuleContext(symbol, referenced_ctx);
            }
        }
    }
}

