/**
 * @file language.cpp 
 * @brief Implementation of functions, that are associated with YARA
 * 
 * @author Vojtěch Dvořák    
 */



#include "headers/common.h"

#include "headers/language.h"



bool isYaraKeyword(std::string_view str) {
    static const std::unordered_set<std::string_view> keywords {
        "all", "and", "any", "ascii", "at", "base64", "base64wide", 
        "condition", "contains", "endswith", "entrypoint", "false", 
        "filesize", "for", "fullword", "global", "import", "icontains", 
        "iendswith", "iequals", "in", "include", "int16", "int16be", "int32", 
        "int32be", "int8", "int8be", "istartswith", "matches", "meta", 
        "nocase", "none", "not", "of", "or", "private", "rule", "startswith", 
        "strings", "them", "true", "uint16", "uint16be", "uint32", "uint32be", 
        "uint8", "uint8be", "wide", "xor", "defined", "variables"
    };

    return keywords.find(str) != keywords.end();
}


bool isValidEscapeSequence(std::string_view str) {
    static const std::unordered_set<std::string_view> escape_sequences {
        "\\\"", "\\n", "\\r", "\\t", "\\\\"
    };

    return escape_sequences.find(str) != escape_sequences.end() || 
        str.starts_with("\\x"); ///< The check, if there is valid hexadecimal number after it is done by parser
}


char escapeSequenceToChar(std::string_view str) {
    static const std::unordered_map<std::string_view, char> map = {
        {"\\\"", '\"'}, 
        {"\\n", '\n'}, 
        {"\\r", '\r'}, 
        {"\\t", '\t'}, 
        {"\\\\", '\\'},
    };

    if(str.starts_with("\\x")) {
        std::string_view num_sv = str.substr(2, 2); ///< The hexa number in esc. seq. starts at index 2 and has length 2

        u_char result = 0;
        auto [ptr, err] = std::from_chars(num_sv.begin(), num_sv.end(), result, 16);
        
        if(err == std::errc()) {
            return result;
        }
        else {
            throw InternalErrorException("Unable to convert escape sequence to character!"); ///< Convertibility should be ensured by the parser
        }
    }
    else {
        return map.at(str);
    }
}


const json *getBuiltinSymbol(std::string_view name) {
    static const std::unordered_map<std::string_view, json> builtins {
        //------ Int functions -------

        {"int16", json{{"name", "int16"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"int16be", json{{"name", "int16be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"int32", json{{"name", "int32"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"int32be", json{{"name", "int32be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"int8", json{{"name", "int8"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"int8be", json{{"name", "int8be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        //------ UInt functions -------

        {"uint16", json{{"name", "uint16"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"uint16be", json{{"name", "uint16be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"uint32", json{{"name", "uint32"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"uint32be", json{{"name", "uint32be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"uint8", json{{"name", "uint8"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

        {"uint8be", json{{"name", "uint8be"}, {"kind", "function"}, {"overloads", json::array({
            {{ "arguments", json::array({{{"type", "i"}}}) }}
        })}, {"return_type", "i"} }},

         //------ Builtin variables functions -------

        {"filesize", json{{"name", "filesize"}, {"kind", "value"}, {"type", "i"}}},
    };

    auto result_it = builtins.find(name);

    return ((result_it == builtins.end()) ? nullptr : &(result_it->second));
}

