/**
 * @file yaramod_config.h
 * @brief 
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once

#include "variable.h"
#include <string>
#include <unordered_map>


/**
 * @brief Objects of this class hold the user defined configuration of parser, 
 * that can be changed in runtime. 
 */
class YaramodConfig {
    public:
        /**
         * @brief The modes of error collectors
         */
        enum class ErrorMode {
            Tolerant, ///< Errors are collected to map (default)
            Strict, ///< When error is gouging to be added to map exception is thrown
        };

        /**
         * @brief The modes of parser
         */
        enum class ParsingMode {
            Auto, ///< Included files are parsed automatically (default)
            Manual, ///< Parsing of included files must be requested manually
        };

        /**
         * Class is container of unique_ptr, so copy ctor, assignment operator must be 
         * explicitly deleted (otherwise it causes problems with pybind11)
         */
        YaramodConfig() = default;

        YaramodConfig(const YaramodConfig &) = delete;

        YaramodConfig &operator= (const YaramodConfig &) = delete;


        /**
         * Sets the mode of error collectors
         */
        void setErrorMode(ErrorMode mode);

        /**
         * Gets the mode of error collectors
         */
        ErrorMode getErrorMode();


        /**
         * Sets the mode of parsing includes
         */
        void setParsingMode(ParsingMode mode);

        /**
         * Gets the mode of parsing includes
         */
        ParsingMode getParsingMode();


        /**
         * Sets the path of the base directory (path of directory for every
         * relative path that is used in include statements)
         */
        void setBaseDir(const std::string &path);

        /**
         * Gets the path of the base directory 
         */
        const std::string &getBaseDir();

        /**
         * Sets the path of directory with modules
         * @note In this directory should be located module descriptions in
         * JSON format. After changing of module directory, 
         * Yaramod::loadModules should be called to load JSONs located in the
         * new directory.
         */
        void setModuleDir(const std::string &path);

        /**
         * Gets the path of directory with modules 
         */
        const std::string &getModuleDir();

        /**
         * Adds external variable to the database, if variable with the same
         * name already exists, it is rewritten by the new value
         * @param var_name The name of variable
         * @param value Value stored in the variable
         */
        void addVar(const std::string &var_name, int64_t value);
        void addVar(const std::string &var_name, const std::string &value);

        /**
         * Adds boolean variable to variables
         * @param var_name The name of variable
         * @param value Value stored in the variable
         * @note It has the same functionality as YaramodConfig::addVar, but
         * this is not another overload, because it may conflict with other
         * var data types, that can be evaluated as bool
         */
        void addBoolVar(const std::string &var_name, bool value);

        /**
         * Removes variable from database of external variables
         * @note If variable does not exists, nothing happens
         * @param var_name Name of variable, that will be removed
         */
        void removeVar(const std::string &var_name);

        /**
         * Provides access to variable with specified name
         * @param var_name Name of variable, that will be returned 
         * @return Pointer to ExtVariable object with given name or nulltpr
         */
        ExtVariable *getVar(const std::string &var_name);

    private:
        ErrorMode err_mode_ = ErrorMode::Tolerant;

        ParsingMode parsing_mode_ = ParsingMode::Auto;

        /**
         * Directory, in which yaramod will look for included files (it parses 
         * string with YARA and path are not absolute)
         */
        std::string base_dir_ = ""; 

        /**
         * Directory, in which description of modules are stored ("modules" by default)
         */
        std::string module_dir_ = "modules";

        std::unordered_map<std::string, std::unique_ptr<ExtVariable>> ext_vars_;
};

