/**
 * @file yaramod.h 
 * @brief Header file for class representing the parser of the library
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once


#define MODULE_PATH_ENV_VAR "YARAMODV4_MODULES"

#include "yara_source.h"
#include "parser_adapter.h"
#include "tree_sitter/api.h"
#include <string>
#include <fstream>
#include <sstream>
#include <iostream>
#include <filesystem>

#include "forward.h"


/**
 * @brief The class, that controls parsing of Yara rulesets. 
 * 
 * The instance of this class is created by the user and user is also 
 * responsible for deleting of it.
 */
class Yaramod {
    public:
        Yaramod();
        Yaramod(
            const std::string &base_dir, 
            const std::string &module_dir = "modules",
            YaramodConfig::ParsingMode p_mode = YaramodConfig::ParsingMode::Auto, 
            YaramodConfig::ErrorMode e_mode = YaramodConfig::ErrorMode::Tolerant
        );

        ~Yaramod();
    
        /**
         * Parses yara file given by path (and included files)  
         * @param path Path of the entry file
         * @return Pointer to YaraSource object
         */
        YaraSourcePtr parseFile(std::string_view path) const;

        /**
         * Updates YaraSource object after edits (edit method must be called to
         * perform structural edit first - @see YaraSource).
         * @param path Path of the entry file (must be the same as was when was
         * YaraSource created)
         * @param old_src Instance of YaraSource should be updated
         * @return Reference to pointer to updated YaraSource (only to be 
         * consistent with Yaramod::parseFile)
         */
        void parseFile(std::string_view path, const YaraSourcePtr &old_src) const;
        void parseFile(std::string_view path, YaraSource *old_src) const;

        /**
         * Parses yara file given by C++ string. Files, that are included in
         * this string are also parsed.
         * @param string The content of the entry file
         * @return Pointer to YaraSource object
         */
        YaraSourcePtr parseString(const std::string &string) const;
        YaraSourcePtr parseString(const std::string &string, const std::string &entry_file_name) const;


        /**
         * Updates YaraSource object after edits (edit method must be called to
         * perform structural edit first - @see YaraSource).
         * @param string Content of the entry file (edited or original)
         * @param old_src Instance of YaraSource should be updated
         * @return Reference to pointer to updated YaraSource (only to be 
         * consistent with Yaramod::parseFile)
         */
        void parseString(const std::string &string, const YaraSourcePtr &old_src) const;
        void parseString(const std::string &string, YaraSource *old_src) const;

        void parseString(const std::string &string, const YaraSourcePtr &old_src, const std::string &entry_file_name) const;
        void parseString(const std::string &string, YaraSource *old_src, const std::string &entry_file_name) const;

        /**
         * Provides access to configuration object 
         */
        YaramodConfig &config();

        /**
         * Load module(s) located at given path
         * @note if path represents directory, it tries to load all modules in
         * it, if path is regular file, it is considered as standalone module  
         */
        void loadModulesFromPath(std::string_view path);

        /**
         * Loads modules from paths, that are in MODULE_PATH_ENV_VAR
         */
        void loadModulesFromEnvironment(); 

        /**
         * Loads (parses) the JSONs with module descriptions 
         */
        void loadModules();
        
    private:
        std::unique_ptr<IParserAdapter> yara_parser_ = nullptr; ///< Parser of yara rulesets
        std::shared_ptr<YaramodConfig> config_ = nullptr;
};
