/**
 * @file yara_source.h 
 * @brief Contains YaraSource class, that representing the set of
 * yara inputs (string or file with its dependencies)
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once


#include "types.h"
#include "common.h"
#include <map>
#include <memory>
#include <unordered_map>
#include <unordered_set>
#include <stdexcept>
#include <filesystem>

#include "forward.h"


/**
 * @brief Class representing the set of files that are connected by includes.
 * 
 * There is one entry file with N included files and in included files
 * there can be other includes. 
 */
class YaraSource : public Printable {
    public:
        using FileMap = std::unordered_map<std::string, std::shared_ptr<YaraFile>, string_hash_t, std::equal_to<>>;

        YaraSource() = default;
        ~YaraSource();

        /**
         * Returns pointer to entry YaraFile object 
         */
        const std::shared_ptr<YaraFile> &getEntryFile() const;

        /**
         * Returns pointer to YaraFile object with given path
         * @note if YaraFile with provided path does not exists exception is thrown 
         */
        std::shared_ptr<YaraFile> getFile(std::string_view path) const;

        /**
         * Check if there is YaraFile object in YaraSource
         * @return true if there is, otherwise false
         */
        bool hasFile(std::string_view path);

        /**
         * Adds YaraFile object to YaraSource
         */
        void addFile(const std::shared_ptr<YaraFile> &file_ptr);

        /**
         * Sets entry file of YaraSource
         * @note YaraFile object must be first added by YaraSource::addFile method
         */
        void setEntryFile(std::string_view path);

        /**
         * Removes YaraFile object with provided path from YaraSource 
         */
        void removeFile(const std::string &path);

        /**
         * Renames file in YaraSource 
         */
        bool renameFile(std::string_view old_path, std::string_view new_path);

        /**
         * Returns container with all files stored in YaraSource 
         */
        const FileMap &getFiles() const;

        /**
         * Removes all files, that have not set activity flag 
         */
        void clearInactiveFiles();
        
        /**
         * Unset the activity flag of all the files stored in YaraSource 
         */
        void deactivateFiles();

        /*
         * Edit management of YaraSource
         */

        /**
         * Reports an edit of entry file. Edits must be reported before 
         * reparsing for correct incremental parsing process.
         * 
         * @param offset Offset where edit starts (offset of first changed character)
         * @param ins_n Number of inserted characters
         * @param del_n Number of deleted characters
         * 
         * @warning After use of this overload of edit function, reparsing 
         * must be done, because there is now way how to update the 
         * content of internal YaraFile buffer! (because the inserted text is
         * not provided)
         */
        void edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n);

        /**
         * Reports an edit of file with given path. 
         * Edits must be reported before reparsing for correct incremental 
         * parsing process.
         * 
         * @param offset Offset where edit starts (offset of first changed character)
         * @param ins_n Number of inserted characters
         * @param del_n Number of deleted characters
         * @param path Path of edited yara file object
         * 
         * @warning After use of this overload of edit function, reparsing 
         * must be done, because there is now way how to update the 
         * content of internal YaraFile buffer! (because the inserted text is
         * not provided)
         */
        void edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n, const std::string &path);

        /**
         * Reports an edit of entry file. Edits must be reported before 
         * reparsing for correct incremental parsing process. It should
         * correspond to the way how are the edits reported in LSP.
         * 
         * @param edited_range Range where edit was done (start pt is location 
         * of the first edited character and end pt is location of the first
         * character that is not affected by the edit)
         * @param text Newly inserted text (if only deletion was done use 
         * empty string here)
         * 
         * @note By this overload of edit method, multiple edits can be 
         * reported before reparsing.
         */
        void edit(const range_t &edited_range, const std::string &text);

        /**
         * Reports an edit of file with given name. Edits must be reported 
         * before reparsing for correct incremental parsing process. It 
         * should correspond to the way how are the edits reported in LSP.
         * 
         * @param edited_range Range where edit was done (start pt is location 
         * of the first edited character and end pt is location of the first
         * character that is not affected by the edit)
         * @param text Newly inserted text (if only deletion was done use 
         * empty string here)
         * 
         * @note By this overload of edit method, multiple edits can be 
         * reported before reparsing.
         */
        void edit(const range_t &edited_range, const std::string &text, const std::string &path);

        /**
         * Remove all changes, that were reported by the YaraSource::edit and
         * YaraFile::edit calls since the last (re)parsing in the whole 
         * YaraSource object
         */
        void undo();

        /**
         * Returns stringstream with formatted content of all yara files
         * @note The order of yara files is preserved 
         */
        std::stringstream getTextFormatted() const;

        /**
         * Returns base directory path of yara source (under normal 
         * circumstances it is path to directory with the entry file)
         */
        const std::string &getBaseDir();
        void setBaseDir(std::string_view path);

        /*
         * Semantic bindings management 
         */

        /**
         * Notifies change of yara file element (child elements can be 
         * afterwards checked)
         */
        void notifyChange(YaraFileElement *element);

        /**
         * Creates semantic binding between two yara file elements 
         */
        void bind(YaraFileElement *parent, YaraFileElement *dependency);

        /**
         * Delete all bindings in which participate given element 
         */
        void deleteBindings(YaraFileElement *element);

        /**
         * Switch bindings of original parent to new parent element 
         */
        void moveBindings(YaraFileElement *original_parent, YaraFileElementBindable *new_parent);

        /**
         * Prints ranges of bindings from point of view parents 
         */
        void dumpBindChildren();

        /**
         * Prints ranges of bindings from point of view children 
         */
        void dumpBindParents();

        /**
         * Returns vector with pointers to all rules included in YaraSource 
         * object
         * @note This way is not effective (the new vector is allocated and 
         * files syntax rules are sequentially copied), so use it only if 
         * worse performance does not matter
         */
        std::vector<Rule *> getAllRules();

        /**
         * Returns vector with pointers to all syntax errors located in 
         * YaraSource object
         * @note This way is not effective (the new vector is allocated and 
         * files syntax rules are sequentially copied), so use it only if 
         * worse performance does not matter
         */
        std::vector<SyntaxError *> getAllSyntaxErrors(); 

        /**
         * Returns vector with pointers to all semantic errors located in 
         * YaraSource object
         * @note This way is not effective (the new vector is allocated and 
         * files syntax rules are sequentially copied), so use it only if 
         * worse performance does not matter
         */
        std::vector<SemanticError *> getAllSemanticErrors(); 

        /**
         * Find the first occurrence of import, that imports module with given
         * name. If stop_file is reached, searching is interrupted.
         * @return Pointer to Import object or nullptr
         */
        Import *getImport(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) const;
        
        /**
         * Find the first occurrence of rule with given id. If stop_file is, 
         * reached searching is interrupted.
         * @return Pointer to Import object or nullptr
         */
        Rule *getRule(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) const;

        bool hasPendingChanges() const;
    private:
        /**
         * Stores semantic dependencies of yara files elements, that are 
         * included in this yara source
         */
        std::unordered_map<YaraFileElement *, std::unordered_set<YaraFileElement *>> bindings_;
        
        /**
         * Reverse mapping of semantic bindings to easily destroy them (when
         * element is deleted) 
         */
        std::unordered_map<YaraFileElement *, std::unordered_set<YaraFileElement *>> bindings_rev_;

        std::string base_path_ = {};

        std::shared_ptr<YaraFile> entry_file_ = nullptr;

        FileMap files_; ///< The map for direct access to YaraFiles
};
