/**
 * @file yara_file_element.h 
 * @brief Contains class, that represents syntax structures of YaraFile
 * 
 * @author Vojtěch Dvořák 
 */

#pragma once

#include "types.h"
#include <vector>
#include <string>
#include <type_traits>
#include <memory>
#include <map>

#include "forward.h"

/**
 * @brief Class representing syntax structures of YaraFile (such as Rules, 
 * Imports, Includes - everything which has some characters in the source 
 * string)
 */
class YaraFileElement {
    public:
        YaraFileElement();
        YaraFileElement(const offset_t &offset, const size_t &len);

        /**
         * Returns start offset of yara file element 
         */
        virtual offset_t getOffset() const;

        /**
         * Sets the start offset of yara file element 
         */
        virtual void setOffset(const offset_t &offset);

        /**
         * Returns length of yara file element 
         */
        size_t getLen() const;

        /**
         * Sets the length of yara file element 
         */
        void setLen(const size_t &len);

        /**
         * Returns position of first character of the yara file element 
         */
        point_t getPosition() const;

        /**
         * Returns Range where is YaraFileElement located 
         */
        range_t getRange() const;

        /**
         * Sets parent file of element 
         */
        void setParentFile(const std::shared_ptr<YaraFile> &parent);

        /**
         * Gets weak_ptr to parent file
         */
        std::weak_ptr<YaraFile> getParentFile() const;

        /**
         * Checks if deletion of this YaraFileElement was requested
         */
        bool isDeletionRequested() const;

        /**
         * Request deletion of this YaraFileElement
         */
        void requestDeletion();

        /**
         * Check if YaraFileElement is fixed to other YaraFileElement
         */
        bool isFixed() const;

        /**
         * Sets YaraFileElement to this YaraFileElement as parent
         */
        void setParent(YaraFileElement *parent);
    
    protected:
        offset_t offset_;
        size_t len_;

        bool is_deletion_requested_ = false;

        std::weak_ptr<YaraFile> parent_file_;

        YaraFileElement *parent_element_ = nullptr;
};


/**
 * @brief Represents of elements of YaraFile, that can have some semantic 
 * binding with other elements
 */
class YaraFileElementBindable : public YaraFileElement {
    public:
        YaraFileElementBindable();
        YaraFileElementBindable(const offset_t &offset, const size_t &len);

        virtual ~YaraFileElementBindable();

        /**
         * Creates semantic binding between "this" and other element, that
         * is given as argument. This element is 
         */
        YaraFileElementBindable *bind(YaraFileElementBindable *dependency);

        /**
         * Creates semantic binding between "this" and other element
         */
        YaraFileElementBindable *bind_to(YaraFileElementBindable *parent);
};


/**
 * @brief Class for top level yara file elements 
 * 
 * Top level from point of view of semantic interface, so top level elements
 * are rules, imports, includes but also comments and errors 
 */
class TopLevelYaraFileElement : public YaraFileElementBindable {
    public:
        TopLevelYaraFileElement();
        TopLevelYaraFileElement(const offset_t &offset, const size_t &len);

        virtual ~TopLevelYaraFileElement();

        /**
         * Returns position of first character of the yara file element 
         */
        point_t getPosition();

        /**
         * Returns Range where is YaraFileElement located 
         */
        range_t getRange();

        /**
         * Invalidates cached range and position 
         */
        void invalidateCache();

        /**
         * Sets the range of rule, it is stored in range cache of the rule 
         */
        void setRangeCache(const range_t &range);

        void setPositionCache(const point_t &position);

        /**
         * Checks if cached values of position and range are valid
         */
        bool hasValidCache();
    
    protected:
        YaraFileElement *parent_element_ = nullptr; ///< Parent element that fixes elements on its place

        range_t cached_range_ = {{0, 0}, {0, 0}}; ///< Range cache to avoid unnecessary calls of offsetToPoint 
        bool is_range_cache_valid_ = false;
        bool is_cached_position_valid_ = false;
}; 
