/**
 * @file yara_file.h 
 * @brief Contains YaraFile class, that represents one continuous input
 * in YARA language
 * 
 * @author Vojtěch Dvořák
 */

#pragma once 

#include "yara_file_element.h"
#include "types.h"
#include "symtab.h"
#include "yara_source.h"
#include "error_collector.h"
#include "file_context.h"
#include <iostream>
#include <memory>
#include <string>
#include <set>
#include <unordered_set>
#include <unordered_map>
#include <vector>

#include "forward.h"


/**
 * @brief Represents textual edit of YaraFile 
 */
struct edit_t {
    edit_t(
        const offset_t &edit_offset, 
        const size_t &edit_ins_n, 
        const size_t &edit_del_n
    ) : offset(edit_offset), ins_n(edit_ins_n), del_n(edit_del_n) {};

    edit_t(
        const offset_t &edit_offset, 
        const size_t &edit_ins_n, 
        const size_t &edit_del_n,
        const std::string &text
    ) : offset(edit_offset), ins_n(edit_ins_n), del_n(edit_del_n), new_text(text) {};

    offset_t offset; ///< Offset where the edit begins (the first affected character)
    size_t ins_n; ///< Number of inserted characters
    size_t del_n; ///< Number of deleted characters
    std::string new_text = {}; ///< The inserted text (optional, recommended)

    TSInputEdit ts_edit;

    bool fast_edit = false; ///< Flag if edit is specified only by numeric values (new text is not needed) - not recommended
};


/**
 * @brief Represents on continuous sequence of yara rules, file includes and 
 * imports.
 * 
 * Also works as stateful object, that holds all necessary data for reparsing. 
 */
class YaraFile : public std::enable_shared_from_this<YaraFile>, public Printable {
    public:
        YaraFile(YaraSource *parent_src);
        ~YaraFile();

        /**
         * Returns the name of YaraFile (it is absolute path to file) 
         */
        const std::string &getName() const;

        /**
         * Sets the name of file 
         */
        void setName(const std::string &new_name);
        void setName(std::string_view new_name);

        // Error management

        /**
         * Adds error object to the specified error collector  
         * @note Helper template methods to avoid redundancies 
         */
        template <class T, class E>
        E *addError(ErrorCollector<T> &error_collector, std::unique_ptr<E> &&err);

        /**
         * Provides access to tab with syntax errors 
         */
        const SymTabOffset<SyntaxError> &getSyntaxErrors() const;

        /**
         * Clears tab with syntax errors in the current file 
         */
        void clearSyntaxErrors();

        /**
         * Adds SyntaxError to the yara file object
         * @return Raw pointer to the newly added SyntaxError object (to be additionally modified) 
         */
        template <class E>
        E *addSyntaxError(std::unique_ptr<E> &&err);
        
        template <class E>
        E *addSyntaxError(const std::string &desc, const offset_t &offset, size_t len);

        template <class E>
        E *addSyntaxError(const std::string &desc, const offset_t &offset, size_t len, const range_t &range);

        /**
         * Remove syntax errors specified by given lambda function (errors are
         * deleted if lambda is evaluated for them as true) 
         */
        template<typename Fn>
        uint32_t removeSyntaxErrors(const Fn &f);
        
        /**
         * Updates all syntax errro by given lambda function 
         */
        template<typename Fn>
        void updateSyntaxErrors(const Fn &f);

        /**
         * Prints formatted all syntax errors in the current file
         */
        void dumpSyntaxErrors();

        /**
         * Provides access to tab with syntax errors 
         */
        const SymTabOffset<SemanticError> &getSemanticErrors() const;

        /**
         * Clears tab with semantic errors in the current file 
         */
        void clearSemanticErrors();

        /**
         * Adds SemanticError to the yara file object
         * @return Raw pointer to the newly added semantic error 
         */
        template <class E>
        E *addSemanticError(std::unique_ptr<E> &&err);
        
        template <class E>
        E *addSemanticError(std::string_view desc, const offset_t &offset, size_t len);

        template <class E>
        E *addSemanticError(std::string_view desc, const offset_t &offset, size_t len, const range_t &range);

        /**
         * Remove semantic errors specified by given lambda function (errors are
         * deleted if lambda is evaluated for them as true) 
         */
        template<typename Fn>
        uint32_t removeSemanticErrors(const Fn &f);

        template<typename Fn>
        void updateSemanticErrors(const Fn &f);
        void dumpSemanticErrors() const;

        // Rule management

        /**
         * Makes lookup in this file and in the parent YaraSource object  
         */
        Rule *ruleLookUp(const std::string &rule_id, std::vector<YaraFileElementBindable *> &includes);

        /**
         * Create correct duplicated rule message with the correct location 
         */
        RuleError *createRuleError(const std::string &err_msg, Rule *wrong_rule, const std::vector<YaraFileElementBindable *> &includes);

        /**
         * Adds Rule object to the symbol tab
         */
        Rule *addRule(RulePtr &&rule);

        /**
         * Clears symbol tab with local rules 
         */
        void clearLocalRules();

        /**
         * Returns vector with the pointer to the rules in this YaraFile. 
         * Vector is sorted by offset of the rules. If rule is added, removed
         * or all rules are cleared returned, vector may not be valid. Caller
         * must not delete any of rules from returned list.  
         */
        const SymTab<Rule> &getLocalRules() const;

        /**
         * Get vector with pointers to the all rules located in this yara file
         * and in included yara files 
         */
        std::vector<Rule *> getRules() const;

        /**
         * Get vector with pointers to the all rules located in this yara file
         * and in included yara files, that have given name 
         */
        std::vector<Rule *> getRules(const std::string &name) const;

        /**
         * Check if the rule with given name is accessible in the whole context 
         * of this YaraFile 
         */
        bool hasRule(std::string_view name) const;

        /**
         * Check if the rule with given name is accessible in the context of
         * this YaraFile before given offset
         */
        bool hasRule(std::string_view name, offset_t before) const;

        /**
         * Returns pointer to rule with given name, that is located in this
         * yara file or in any of included files
         * @return Pointer to the rule or nullptr (if it was not found)
         */
        Rule *getRule(std::string_view name) const;
        Rule *getRule(std::string_view name, offset_t before) const;

        /**
         * Returns pointer to rule with given name, that is located in this
         * yara file or in any of included files. Optionally the searching
         * can be done from the start of YaraSource. Also returns the
         * dependent file includes in third (output) parameter (dependent
         * file includes mean file includes that are necessary to have
         * access to the rule in this yara_file).
         * @return Pointer to the rule or nullptr
         */
        Rule *getRule(
            std::string_view name,
            offset_t before, 
            std::vector<YaraFileElementBindable *> &includes, 
            bool make_lookup_from_top = false
        ) const;

        /**
         * Searches for the rule with given name in this file and in included 
         * files until the stop_file is reached. Also returns the
         * dependent file includes in third (output) parameter.
         * @return Pointer to the rule or nullptr
         */
        Rule *getRule(
            std::string_view name, 
            const YaraFile *stop_file, 
            std::vector<YaraFileElementBindable *> &includes
        );

        /**
         * Tries to find the rule by name in only in local context of this file
         * (included files are not searched).
         */
        const RulePtr &getLocalRule(const std::string &name) const;
        Rule *getLocalRule(uint32_t index) const;

        /**
         * Returns rule that overlaps given row, if there is no such rule,
         * nullptr is returned 
         */
        Rule *getLocalRuleByRow(uint32_t row);

        /**
         * Removes the all rules, for that is evaluated given boolean function
         * as true 
         */
        template <typename Fn>
        uint32_t removeLocalRules(const Fn &f);

        /**
         * Updates local by given lambda function
         */
        template <typename Fn>
        void updateLocalRules(const Fn &f);

        /**
         * Checks if this and other yara file have any rule with the same ids
         */
        void checkDuplicatedRules(const YaraFilePtr &other, FileInclude *include);

        //File include management

        /**
         * Adds FileInclude object to includes of yara file
         */
        FileInclude *addFileInclude(std::unique_ptr<FileInclude> include);
        
        /**
         * Clears symtab with local file includes
         */
        void clearLocalFileIncludes();
        
        /**
         * Get local file includes by index
         */
        FileInclude *getLocalFileInclude(uint32_t index) const;
        
        /**
         * Returns reference to symtab with file includes 
         */
        const SymTab<FileInclude> &getLocalFileIncludes() const;

        /**
         * Returns all FileIncludes in file included included FileIncludes
         */
        std::vector<FileInclude *> getFileIncludes() const;

        /**
         * Removes all local file includes for which is evaluated given 
         * function as true
         */
        template <typename Fn>
        uint32_t removeLocalFileIncludes(const Fn &f);

        /**
         * Updates all local file includes by given lambda function
         */
        template<typename Fn>
        void updateLocalFileIncludes(const Fn &f);

        // Import management

        /**
         * Searches for import that imports module with given name. Searching
         * is done in this file and in included files.
         * @return Pointer to found module or nullptr if it was not found
         */
        Import *getImport(std::string_view name) const;

        /**
         * Searches for Import object that imports module with given name. 
         * Import must occurs before given offset.
         * Searching is done in this file and in included files.
         * @return Pointer to found module or nullptr if it was not found
         */
        Import *getImport(std::string_view name, offset_t before) const;

        /**
         * Searches for Import object that imports module with given name.
         * Import must occurs before given offset.
         * Searching is done in this file and in included files and optionally 
         * in the whole parent YaraSource. Returns the dependent includes
         * as the third argument.
         * @return Pointer to found module or nullptr if it was not found
         */
        Import *getImport(
            std::string_view name, 
            offset_t before, 
            std::vector<YaraFileElementBindable *> &includes, 
            bool make_lookup_from_top = false
        ) const;

        /**
         * Searches for Import object that imports module with given name until
         * the stop file is found. Returns the dependent includes
         * as the third argument.
         */
        Import *getImport(
            std::string_view name, 
            const YaraFile *stop_file, 
            std::vector<YaraFileElementBindable *> &includes
        );

        /**
         * Adds Import object to the tab with imports  
         */
        Import *addImport(std::unique_ptr<Import> import);

        /**
         * Clear tab with local imports 
         */
        void clearLocalImports();

        /**
         * Returns local Import object specified by the current index 
         */
        Import *getLocalImport(uint32_t index) const;

        /**
         * Returns the vector with pointers to all import statements, that 
         * can be accessed from this file (sorted by offset)
         */
        std::vector<Import *> getImports() const;

        /**
         * Get map with imports in this file indexed by the offset of imports
         */
        const SymTabDuplId<Import> &getLocalImports() const;

        /**
         * Similar to YaraFile::removeLocalRules 
         */
        template<typename Fn>
        uint32_t removeLocalImports(const Fn &f);

        /**
         * Similar to YaraFile::updateLocalRules 
         */
        template<typename Fn>
        void updateLocalImports(const Fn &f);

        // Comment management

        /**
         * Adds comment to the tab of the comments 
         */
        void addComment(std::unique_ptr<Comment> comment);

        /**
         * Clears the tab with comments 
         */
        void clearComments();

        /**
         * Returns local comment at specified index (the first comment that 
         * occurs has index 0) 
         */
        const std::unique_ptr<Comment> &getCommentByIndex(uint32_t index) const;

        /**
         * PRovides access to tab with comments 
         */
        const SymTabOffset<Comment> &getComments() const;

        /**
         * Similar to YaraFile::removeComments 
         */
        template<typename Fn>
        uint32_t removeComments(const Fn &f);

        /**
         * Similar to YaraFile::updateComments 
         */
        template<typename Fn>
        void updateComments(const Fn &f);

        //---

        /**
         * Adds element to tab with file elements
         */
        void addWrong(std::unique_ptr<TopLevelYaraFileElement> &&element);
        
        /**
         * Clears tab with wrong file elements 
         */
        void clearWrong();

        /**
         * Clears all symbol tables and errors of yara file 
         */
        void clearAll();

        // Edit management

        /**
         * Adds new edit to the edit buffer 
         */
        void addEdit(const edit_t &edit);

        /**
         * Check if there is fast edit (edit specified only by numerical values) 
         */
        bool hasFastEdit();

        /**
         * Performs structural edit above this YaraFile (its tree structure).
         * This method cannot be called twice without reparsing (that restore
         * consistency of internal buffer).
         * @param offset The index of the byte where was file edited 
         * @param ins_n Number of inserted characters by the edit
         * @param del_n Number of deleted characters by the edit
         */
        void edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n);

        /**
         * Performs structural edit above this YaraFile (its tree structure).
         * This method CAN be called multiple times without reparsing, because
         * thanks to provided text it is possible to maintain internal buffer.
         * @param edited_range The index of the byte where was file edited 
         * @param text Inserted text
         * @note Parameters have same meaning as parameters of 
         * DidChangeTextDocument in LSP
         */
        void edit(const range_t &edited_range, const std::string &text);

        /**
         * Clears edit buffer and temporary string buffer 
         */
        void undo();

        /**
         * Clears edit buffer 
         */
        void clearEdits();

        /**
         * Provides access to edit buffer 
         */
        std::vector<edit_t> &getEdits();

        /**
         * Check if there is any pending edit in the edit buffer
         */
        bool wasEdited();

        /**
         * Checks if there is any range, that should be checked (because 
         * semantic context has changed)
         */
        bool mustBeReparsed();

        //--

        /**
         * Removes all errors with global flag 
         */
        void removeGlobalErrors();

        /**
         * Removes all top level elements of YaraFile (and errors and 
         * comments) in specified range 
         */
        offset_range_t removeElementsInRange(const offset_t &start, const offset_t &end);

        /**
         * Shifts all top level elements (their offset) of YaraFile by 
         * specified delta value (it can be also negative) 
         */
        void shiftElements(const offset_t &start, const uint32_t &start_row, const int32_t &delta, const int32_t &row_delta, const int32_t &col_delta);

        /**
         * Finds all offset ranges, that were modified by edits (important for 
         * deletes) or TS explicitly marks them as modified (it is important 
         * for contextual changes) 
         */
        std::vector<offset_edit_range_t> modify(TSTree *new_tree);

        /**
         * Sets the state of activity flag 
         */
        void setActivity(bool state = true);

        /**
         * Returns the state of activity flag 
         */
        bool isActive();


        /**
         * Sets the error mode of the yara file object
         * @note This function should be called by user, user should modify
         * error mode by changing error mode of the parser 
         */
        void setErrorMode(YaramodConfig::ErrorMode new_mode);


        /**
         * Notifies change of YaraFileElement (dependencies will be afterwards 
         * also reparsed)
         */
        void notifyChange(YaraFileElement *element);
        
        /**
         * Creates binding between parent element and dependent element
         */
        void bind(YaraFileElement *parent, YaraFileElement *dependency);
        
        /**
         * Creates binding between all elements in vector (they acts as parents)
         * and dependency YaraFileObject
         */
        void bind(const std::vector<YaraFileElementBindable *> &parents, YaraFileElement *dependency);
        
        /**
         * Removes all bindings in which participates given YaraFileElement
         * object 
         */
        void deleteBindings(YaraFileElement *element);

        void moveBindings(YaraFileElement *original_parent, YaraFileElementBindable *new_parent);

        /**
         * Adds new range to to_be_checked_ vector
         */
        void checkRange(const offset_range_t &range_to_be_checked); 

        /**
         * Clears to_be_checked_ vector
         */
        void clearToBeChecked(); 


        /**
         * Provides access to parsing context of the yara file object
         */
        FileContext &ctx();

        /**
         * Checks if YaraFile object is isolated (isolated flag is set when 
         * include, that references this YaraFile object is deleted) 
         */
        bool isIsolated() const;

        /**
         * Sets the sate of isolation flag 
         */
        void setIsolated(bool new_state = true);

        /**
         * Add missing symbol to the map of the missing symbols 
         */
        void addMissingSymbol(const std::string &missing, const offset_range_t &range);

        /**
         * Removes all missing symbols under with specified identifier 
         */
        void removeMissingSymbol(const std::string &missing);

        /**
         * Converts ranges that belongs to specific missing symbols to
         * ranges that will be checked during reparsing 
         */
        void checkMissing(const std::string &new_symbol);

        /**
         * Checks missing symbols with specified id in the all included files
         */
        void notifyAddedSymbol(const std::string &id);


        std::stringstream getTextFormatted() const override;
    
    private:
        std::vector<offset_range_t> to_be_checked_; ///< Ranges affected by the change of element that have semantic bindings to other elements

        FileContext ctx_; ///< Hold parsing context of Yara File object

        /**
         * Merges offset ranges that overlaps
         */
        static void mergeOverlappingRanges_(std::vector<offset_edit_range_t> &ranges);

        std::string name_ = {}; ///< The name of YaraFile (path or empty string if YaraFile was given by string)

        YaramodConfig::ErrorMode err_mode_; ///< Error mode of YaraFile (it is given by the Yaramod instance, that parses/reparses the YaraFile)

        SymTab<Rule> rules_tab_; ///< Symbol tab with rules
        SymTab<FileInclude> includes_tab_; ///< Symbol tab with file includes

        SymTabDuplId<Import> imports_; ///< Symbol tab with imports

        SymTabOffset<Comment> comments_; ///< Symbol tab with comments

        SymTabOffset<TopLevelYaraFileElement> wrong_elements_; ///< Symbol tab with wrong elements
        
        ErrorCollector<SyntaxError> syntax_; ///< Syntax errors located in this yara file object
        ErrorCollector<SemanticError> semantic_; ///< Semantic errors located in this yara file object

        std::vector<edit_t> edits_;

        bool active_ = true; ///< Flag for recognizing files, that were deleted from yara source

        YaraSource *parent_src_ = nullptr; ///< Pointer to parent YaraSource

        bool is_isolated_ = false;

        std::unordered_multimap<std::string, offset_range_t> missing_symbols_; ///< MMap with all symbols, that are missing in this file
};


// Template methods implementation

template <class T, class E>
E *YaraFile::addError(ErrorCollector<T> &error_collector, std::unique_ptr<E> &&err) {
    E *raw_ptr = err.get();
    error_collector.add(std::move(err), err_mode_);
    return raw_ptr;
}


template <class E>
E *YaraFile::addSyntaxError(std::unique_ptr<E> &&err) {
    err->setParentFile(shared_from_this());
    return addError(syntax_, std::move(err));
}


template <class E>
E *YaraFile::addSyntaxError(const std::string &desc, const offset_t &offset, size_t len) {
    auto new_err = std::make_unique<E>(desc, offset, len);
    new_err->setParentFile(shared_from_this());
    return addError(syntax_, std::move(new_err));
}

template <class E>
E *YaraFile::addSyntaxError(const std::string &desc, const offset_t &offset, size_t len, const range_t &range) {
    auto new_err = std::make_unique<E>(desc, offset, len);
    new_err->setParentFile(shared_from_this());
    new_err->setRangeCache(range);
    return addError(syntax_, std::move(new_err));
}


template <class E>
E *YaraFile::addSemanticError(std::unique_ptr<E> &&err) {
    err->setParentFile(shared_from_this());
    return addError(semantic_, std::move(err));
}


template <class E>
E *YaraFile::addSemanticError(std::string_view desc, const offset_t &offset, size_t len) {
    auto new_err = std::make_unique<E>(std::string(desc), offset, len);
    new_err->setParentFile(shared_from_this());
    return addError(semantic_, std::move(new_err));
}


template <class E>
E *YaraFile::addSemanticError(std::string_view desc, const offset_t &offset, size_t len, const range_t &range) {
    auto new_err = std::make_unique<E>(std::string(desc), offset, len);
    new_err->setParentFile(shared_from_this());
    new_err->setRangeCache(range);
    return addError(semantic_, std::move(new_err));
}