/**
 * @file visitor.h 
 * @brief Contains declaration of Visitor class
 *
 * @author Vojtěch Dvořák 
 */

#pragma once

#include "expression_types.h" 


/**
 * @brief Virtual base class for concrete visitor classes
 * 
 * If it is used as base class for concrete visitor, it ensures, that all 
 * possible types of expressions will be handled during traversal of 
 * expressions.
 */
class Visitor {
    public:
        virtual void visitRange(Range *r) = 0;
        virtual void visitEnum(Enum *e) = 0;
        virtual void visitParentheses(ParenthesesExpression *e) = 0;
        
        virtual void visitIn(InExpression *e) = 0;
        virtual void visitMatches(MatchesExpression *e) = 0;
        virtual void visitAt(AtExpression *e) = 0;

        // Arithmetics

        virtual void visitAdd(AddExpression *e) = 0;
        virtual void visitSub(SubExpression *e) = 0;
        virtual void visitMul(MulExpression *e) = 0;
        virtual void visitRemainder(RemainderExpression *e) = 0;
        virtual void visitDiv(DivExpression *e) = 0;

        // Bitwise operators

        virtual void visitLeftShift(LeftShiftExpression *e) = 0;
        virtual void visitRightShift(RightShiftExpression *e) = 0;
        virtual void visitBitwiseOr(BitwiseOrExpression *e) = 0;
        virtual void visitBitwiseAnd(BitwiseAndExpression *e) = 0;
        virtual void visitBitwiseXor(BitwiseXorExpression *e) = 0;

        // Relational operators 

        virtual void visitLt(LtExpression *e) = 0;
        virtual void visitLte(LteExpression *e) = 0;
        virtual void visitGt(GtExpression *e) = 0;
        virtual void visitGte(GteExpression *e) = 0;
        virtual void visitEq(EqExpression *e) = 0;
        virtual void visitNeq(NeqExpression *e) = 0;

        // String relational operators

        virtual void visitContains(ContainsExpression *e) = 0;
        virtual void visitIContains(IContainsExpression *e) = 0;
        virtual void visitStartsWith(StartsWithExpression *e) = 0;
        virtual void visitIStartsWith(IStartsWithExpression *e) = 0;
        virtual void visitEndsWith(EndsWithExpression *e) = 0;
        virtual void visitIEndsWith(IEndsWithExpression *e) = 0;
        virtual void visitIEq(IEqExpression *e) = 0;

        // Bool operators

        virtual void visitAnd(AndExpression *e) = 0;
        virtual void visitOr(OrExpression *e) = 0;

        // Unary operators

        virtual void visitDefined(DefinedExpression *e) = 0;
        virtual void visitUnaryMinus(UnaryMinusExpression *e) = 0;
        virtual void visitBitwiseNot(BitwiseNotExpression *e) = 0;
        virtual void visitNot(NotExpression *e) = 0;

        // Terminals

        virtual void visitLiteral(LiteralExpression *e) = 0;
        virtual void visitSize(SizeExpression *e) = 0;
        virtual void visitRegexp(RegexpExpression *e) = 0;
        virtual void visitString(StringExpression *e) = 0;
        virtual void visitStringWildcard(StringWildcardExpression *e) = 0;
        virtual void visitStringCount(StringCountExpression *e) = 0;
        virtual void visitStringOffset(StringOffsetExpression *e) = 0;
        virtual void visitStringMatchLength(StringMatchLengthExpression *e) = 0;

        // Symbols

        virtual void visitPlainSymbol(PlainSymbol *symbol) = 0;
        virtual void visitArray(ArrayExpression *e) = 0;
        virtual void visitStruct(StructExpression *e) = 0;
        virtual void visitFunctionCall(FunctionCallExpression *e) = 0;
        virtual void visitVariable(VariableExpression *e) = 0;
        virtual void visitRuleWildCard(RuleWildcardExpression *e) = 0;

        // Quantifiers

        virtual void visitNone(NoneExpression *e) = 0;
        virtual void visitAny(AnyExpression *e) = 0;
        virtual void visitAll(AllExpression *e) = 0;
        virtual void visitPercentQuantifier(PercentQuantifierExpression *e) = 0;
        virtual void visitThem(ThemExpression *e) = 0;

        // Sets

        virtual void visitStringSet(StringSetExpression *e) = 0;
        virtual void visitRuleSet(RuleSetExpression *e) = 0;
        virtual void visitExpressionSet(ExpressionSetExpression *e) = 0;

        // Of operator

        virtual void visitOf(OfExpression *e) = 0;

        // For operators

        virtual void visitFor(ForExpression *e) = 0;
        virtual void visitForInt(ForIntExpression *e) = 0;
        virtual void visitVarList(VarListExpression *e) = 0;
};
