/**
 * @file variable.h 
 * @brief Declaration of Variable class and its two specializations
 * IntVariable (for internal variables, Avast feature of YARA) and ExtVariables
 * (for external variables)
 *
 * @author Vojtěch Dvořák 
 */

#pragma once

#include "literal.h"
#include "rule_element.h"
#include <variant>
#include <memory>

#include "forward.h"


/**
 * @brief Base class for variables
 */
class Variable {
    public:
        Variable();
        Variable(const std::string &id);

        /**
         * Returns id of variable 
         */
        const std::string &getId() const;

        /**
         * Sets the id of variable 
         */
        void setId(std::string &&id);
        void setId(const std::string &id);

    protected:
        std::string id_ = std::string({});
};


/**
 * @brief Class, that represents internal variables
 * @note Internal variables are Avast feature
 */
class IntVariable : public Variable, public RuleElement, public Printable {
    public:
        IntVariable();
        IntVariable(const std::string &id);

        /**
         * Returns Expression, that represnts value of internal variable 
         */
        const std::shared_ptr<Expression> &getValue() const;

        /**
         * Sets the value of internal variable 
         */
        void setValue(std::shared_ptr<Expression> &&expr); 

        std::stringstream getTextFormatted() const override;

    private:
        std::shared_ptr<Expression> value_;
};


/**
 * @brief Class that represents external variables
 */
class ExtVariable : public Variable {
    public:
        ExtVariable();
        ExtVariable(const std::string &id, const std::string &value);
        ExtVariable(const std::string &id, int64_t value);
        ExtVariable(const std::string &id, bool value);

        /**
         * Returns Literal object, that represents value of external variable 
         */
        const std::unique_ptr<Literal> &getValue() const;

        /**
         * Sets the value of external variable 
         */
        void setValue(std::unique_ptr<Literal> &&literal);

    private:
        std::unique_ptr<Literal> value_;
};
