/**
 * @file types.h 
 * @brief Declaration of structures and simple data types, that are used
 * across the project
 * 
 * @author Vojtěch Dvořák 
 */

#pragma once

#include "json/json.hpp"

#include <cstdint>
#include <memory>

#include "forward.h"


using json = nlohmann::json;

using offset_t = uint32_t;


using RulePtr = std::unique_ptr<Rule>;
using ExpressionPtr = std::shared_ptr<Expression>;
using YaraFilePtr = std::shared_ptr<YaraFile>;
using YaraSourcePtr = std::unique_ptr<YaraSource>;

/**
 * @brief Type for specifying position in input string
 */
struct point_t {
    uint32_t row; ///< The row of input document
    uint32_t col; ///< The column of input document

    point_t() = default;

    point_t(uint32_t row_pos, uint32_t col_pos) : row(row_pos), col(col_pos) {};

    bool operator<(const point_t& r_point) {
        return row < r_point.row || (row == r_point.row && col < r_point.col);
    }

    bool operator==(const point_t& r_point) {
        return row == r_point.row && col == r_point.col;
    }

    point_t operator+(const point_t& r_point) {
        point_t point;
        point.row = row + r_point.row;
        point.col = col;
        return point;
    }

    point_t operator-(const point_t& r_point) {
        point_t point;
        point.row = row - r_point.row;
        point.col = col;
        return point;
    }
};


/**
 * @brief Range in input document specified by start point and end point 
 */
struct range_t {
    point_t start; ///< Start point of range (included in range)
    point_t end; ///< End point of range (NOT included in range)

    range_t() = default;

    range_t(point_t start_pt, point_t end_pt) : start(start_pt), end(end_pt) {};

    range_t(
        uint32_t start_row, 
        uint32_t start_col, 
        uint32_t end_row, 
        uint32_t end_col
    ) : start(point_t(start_row, start_col)), end(point_t(end_row, end_col)) {};

    range_t operator+(const point_t& r_point) {
        range_t range;
        range = {start + r_point, end + r_point };
        return range;
    }

    range_t operator-(const point_t& r_point) {
        range_t range;
        range = {start - r_point, end - r_point };
        return range;
    }
};


/**
 * @brief Hash struct for SymTab, provides heterogeneous comparison lookup
 * 
 * Based on: https://en.cppreference.com/w/cpp/container/unordered_map/find
 * Thanks to this, lookup in this SymTab can be done by std::string as 
 * well as with std::string_view (so, it is possible to avoid unnecessary allocation)
 */
struct string_hash_t {
    using hash_type = std::hash<std::string_view>;
    using is_transparent = void;

    std::size_t operator()(std::string_view sv) const {
        return hash_type{}(sv);
    }

    std::size_t operator()(const std::string &str) const {
        return hash_type{}(str);
    }
};
// End of part based on https://en.cppreference.com/w/cpp/container/unordered_map/find


/**
 * @brief Offset range for specifying the part of file, that was for example 
 * modified, deleted etc. 
 */
struct offset_range_t {
    offset_t start;
    offset_t end;

    bool operator<(const offset_range_t& right_range) {
        bool start_before = start < right_range.start; // "This" range starts before right range 
        bool same_start_end_after = start == right_range.start; // "This" range starts at the same offset as right range
        bool end_after = end > right_range.end; // "This" ends after the right range

        return start_before || (same_start_end_after && end_after);
    }

};


/**
 * @brief Structure for lossless storing data about atomic edits in unified 
 * form 
 */
struct offset_edit_range_t {

    /**
     * @brief Type of edit 
     */
    enum class EditType {
        MOD, ///< Unspecified modification
        DEL, ///< Deletion
        INS, ///< Insertion
    };

    offset_t start;
    offset_t end;
    EditType type = EditType::MOD;

    offset_edit_range_t(const offset_range_t &offset_range, EditType edit_type = EditType::MOD) {
        start = offset_range.start;
        end = offset_range.end;
        type = edit_type;
    }

    offset_edit_range_t(const offset_t &start_offset, const offset_t &end_offset, EditType edit_type = EditType::MOD) {
        start = start_offset;
        end = end_offset;
        type = edit_type;
    }

    /**
     * Computes the length of range
     * @return the length of the range 
     */
    size_t len() {
        return end - start;
    }

    bool operator<(const offset_edit_range_t& right_range) {
        bool start_before = start < right_range.start;
        bool same_start_end_after = start == right_range.start;
        bool end_after = end > right_range.end;

        return start_before || (same_start_end_after && end_after);
    }

};
