/**
 * @file symtab.h 
 * @brief Contains template classes for creating symbol tables
 * 
 * @author Vojtěch Dvořák
 */

#pragma once

#include "types.h"
#include "common.h"
#include <memory>
#include <map>
#include <vector>
#include <unordered_map>


/**
 * @brief Tab for symbols with double indexing. 
 * 
 * Symbols are indexed by their unique id and by their offset.
 */
template<class V>
class SymTab {
    public:
        using OffsetMap = std::multimap<offset_t, V*>;
        using IdMap = std::unordered_map<std::string, std::unique_ptr<V>, string_hash_t, std::equal_to<>>; ///< Holds the ownership of entries

        SymTab() = default;

        SymTab(const SymTab &) = delete;

        SymTab &operator= (const SymTab &) = delete;

        /**
         * @brief SymTab iterator (just a wrapper above 
         * OffsetMap::const_iterator)
         */
        class iterator {
            public:
                friend class SymTab;

                using iterator_category = std::forward_iterator_tag;
                using value_type = V;
                using difference_type = V;
                using pointer = V*;
                using reference = V&;

                explicit iterator(const OffsetMap::const_iterator &it);

                V* operator*() const;
                V* operator->() const;

                iterator& operator++();
                iterator operator++(int);

                friend bool operator==(const iterator &l, const iterator &r) {
                    return l.entry_it_ == r.entry_it_;
                }

                friend bool operator!=(const iterator &l, const iterator &r) {
                    return l.entry_it_ != r.entry_it_;
                }

            private:
                OffsetMap::const_iterator entry_it_; ///< Iterator of offset index of map
        };


        iterator begin() const;
        iterator end() const;

        size_t size() const;
        bool empty() const;

        const std::unique_ptr<V> &at(const std::string &id) const;

        /**
         * Inserts the new entry to the symtab 
         */
        void insert(std::unique_ptr<V> &&value);    

        /**
         * Returns pointer to entry with given key
         */
        const std::unique_ptr<V> &search(const std::string &key) const;

        const std::unique_ptr<V> &search(std::string_view key) const;

        /*
         * Checks if entry with given key exists in symtab or not
         */
        bool exists(const std::string &key) const;
        bool exists(std::string_view key) const;

        /**
         * Removes entry with given id 
         */
        void remove(const std::string &id);

        /**
         * Removes elements with offset from start to end, that meet
         * the condition specified by the lambda function (first arg)
         * @return Number of removed elements 
         */
        template<typename Fn>
        uint32_t remove(const Fn &f, offset_t start, offset_t end);

        /**
         * Removes elements in reversed order until the given lambda function
         * returns true 
         * @return Number of removed elements
         */
        template<typename Fn>
        uint32_t revRemoveUntil(const Fn &f, offset_t rstart);

        /**
         * Updates the entries with offset from start to end by given lambda.
         * Keys = offset and string id, can be also changed. 
         */
        template<typename Fn>
        void update(const Fn &f, offset_t start, offset_t end);

        /**
         * Clears the symbol tab 
         */
        void clear();

        /**
         * Gets the data indexed by string id 
         */
        const IdMap &dataById() const;

        /**
         * Gets the data indexed by offset 
         */
        const OffsetMap &dataByOffset() const;


        V *operator[](size_t index) const;
        V *operator[](const std::string &id) const;

    private:
        OffsetMap offset_map_;
        IdMap id_map_;
};

//----------------------------------- Iterator --------------------------------

template<class V>
SymTab<V>::iterator::iterator(const OffsetMap::const_iterator &it) {
    entry_it_ = it;
}

template<class V>
V* SymTab<V>::iterator::operator*() const {
    return entry_it_->second;
}

template<class V>
V* SymTab<V>::iterator::operator->() const {
    return entry_it_->second;
}

template<class V>
SymTab<V>::iterator& SymTab<V>::iterator::operator++() {
    ++entry_it_;
    return *this;
}

template<class V>
SymTab<V>::iterator SymTab<V>::iterator::operator++(int) {
    SymTab<V>::iterator result = *this;
    ++(*this);
    return result;
}

// ---------------------------------- SymTab ----------------------------------

template<class V>
SymTab<V>::iterator SymTab<V>::begin() const {
    return iterator(offset_map_.begin());
}


template<class V>
SymTab<V>::iterator SymTab<V>::end() const {
    return iterator(offset_map_.end());
}


template<class V>
size_t SymTab<V>::size() const {
    return id_map_.size();
}


template<class V>
bool SymTab<V>::empty() const {
    return id_map_.empty();
}


template<class V>
const std::unique_ptr<V> &SymTab<V>::at(const std::string &id) const {
    return id_map_.at(id);
}


template<class V>
void SymTab<V>::insert(std::unique_ptr<V> &&value) {
    offset_map_.emplace(value->getOffset(), value.get()); ///< Inserting entry to offset index
    id_map_.emplace(value->getId(), std::move(value)); ///< Moving entry to id index (which owns the entry)
}


template<class V>
bool SymTab<V>::exists(const std::string &key) const {
    return id_map_.find(key) != id_map_.end();
}


template<class V>
bool SymTab<V>::exists(std::string_view key) const {
    return id_map_.find(key) != id_map_.end();
}


template<class V>
const std::unique_ptr<V> &SymTab<V>::search(const std::string &key) const {
    return search(static_cast<std::string_view>(key));
}


template<class V>
const std::unique_ptr<V> &SymTab<V>::search(std::string_view key) const {
    auto result_it = id_map_.find(key);

    if(result_it == id_map_.end()) {
        throw InternalErrorException("Item does not exists in symbol tab!");
    }

    return result_it->second; 
}


template<class V>
void SymTab<V>::remove(const std::string &id) {
    if(!exists(id)) {
        return;
    }

    const auto &to_be_deleted = search(id);

    offset_t offset = to_be_deleted->getOffset();
    auto offset_entry = offset_map_.find(offset);

    // Erase the corresponding element in offset map
    for(auto it = offset_entry; 
        it != offset_map_.end() && it->second->getOffset() == offset; 
        ++it) {

        if(it->second == to_be_deleted.get()) { ///< Pointers are the same
            offset_map_.erase(it);
            break;
        }
    }
    
    id_map_.erase(id);
}


template<class V>
template<typename Fn>
uint32_t SymTab<V>::remove(const Fn &f, offset_t start, offset_t end) {
    auto start_it = offset_map_.lower_bound(start);
    auto end_it = offset_map_.upper_bound(end);

    uint32_t deleted = 0;

    for(auto it = start_it; it != end_it;) {
        if(f(*(it->second))) { ///< Check if condition is met
            id_map_.erase(it->second->getId());
            it = offset_map_.erase(it);
            deleted++;
        }
        else {
            ++it;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
uint32_t SymTab<V>::revRemoveUntil(const Fn &f, offset_t rstart) {
    auto start_it = offset_map_.upper_bound(rstart);

    uint32_t deleted = 0;

    for(auto it = std::make_reverse_iterator(start_it); it != offset_map_.rend();) {
        if(f(*(it->second))) { ///< Check if the condition is met
            id_map_.erase(it->second->getId());

            auto fw_it = std::next(it).base();
            it = std::make_reverse_iterator(offset_map_.erase(fw_it));
            deleted++;
        }
        else {
            break;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
void SymTab<V>::update(const Fn &f, offset_t start, offset_t end) {
    auto start_it = offset_map_.lower_bound(start);
    auto end_it = offset_map_.upper_bound(end);

    std::vector<std::pair<offset_t, V*>> to_be_updated;
    for(auto it = start_it; it != end_it; ++it) {
        to_be_updated.emplace_back(it->first, it->second);
    }

    for(auto &offset_pair: to_be_updated) {
        // Extract the node (key can be changed)
        auto eq_range = offset_map_.equal_range(offset_pair.first);
        auto id_entry_node = id_map_.extract(offset_pair.second->getId());

        for(auto it = eq_range.first; it != eq_range.second; ++it) {
            if(it->second == offset_pair.second) {
                auto offset_entry_node = offset_map_.extract(it);

               f(id_entry_node.mapped()); ///< Perform the update

                 // Put the remapped nodes back
                offset_entry_node.key() = id_entry_node.mapped()->getOffset();
                id_entry_node.key() = id_entry_node.mapped()->getId();

                offset_map_.insert(std::move(offset_entry_node));
                id_map_.insert(std::move(id_entry_node));

                break;
            }
        }
    }
}



template<class V>
void SymTab<V>::clear() {
    id_map_.clear();
    offset_map_.clear();
}


template<class V>
const SymTab<V>::IdMap &SymTab<V>::dataById() const {
    return id_map_;
}


template<class V>
const std::multimap<offset_t, V*> &SymTab<V>::dataByOffset() const {
    return offset_map_;
}


template<class V>
V *SymTab<V>::operator[](size_t index) const {
    if(index >= offset_map_.size()) {
        throw std::out_of_range("Index out of range!");
    }

    auto it = offset_map_.begin();
    std::advance(it, index);

    return it->second;
}


template<class V>
V *SymTab<V>::operator[](const std::string &id) const {
    try {
        return search(static_cast<std::string_view>(id)).get();
    }
    catch(const std::exception &) {
        throw std::out_of_range("Id does not exists!"); // Rethrow exception to be consistent with numeric index overload of this method
    }
}



/**
 * @brief Almost same type of symbol table as the SymTab, but in this case 
 * entries are allowed to have duplicated ids. 
 */
template<class V>
class SymTabDuplId {
    public:
        SymTabDuplId() = default;

        SymTabDuplId(const SymTabDuplId &) = delete;

        SymTabDuplId &operator= (const SymTabDuplId &) = delete;

        /**
         * @brief Pool that owns all elements with the same id 
         */
        class Pool {
            public:
                using DataContainer = std::vector<std::unique_ptr<V>>;

                Pool() = default;
                Pool(const Pool &) = delete;
                Pool &operator= (const Pool &) = delete;

                /**
                 * @brief Pool iterator 
                 */
                class iterator {
                    public:
                        friend class Pool;

                        using iterator_category = std::forward_iterator_tag;
                        using value_type = V;
                        using difference_type = V;
                        using pointer = V*;
                        using reference = V&;

                        explicit iterator(const DataContainer::const_iterator &it);

                        V* operator*() const;
                        V* operator->() const;

                        iterator& operator++();
                        iterator operator++(int);

                        friend bool operator==(const iterator &l, const iterator &r) {
                            return l.it_ == r.it_;
                        }

                        friend bool operator!=(const iterator &l, const iterator &r) {
                            return l.it_ != r.it_;
                        }

                    private:
                        DataContainer::const_iterator it_; ///< Iterator of offset index of map
                };

                V *first() const;
                size_t size() const;
                bool empty() const;

                V* operator[](size_t i) const;

                V* at(size_t i) const;

                iterator begin() const;
                iterator end() const;

                DataContainer &data();
                const DataContainer &rd_data() const;

                void erase(V* item); 

            private:
                DataContainer data_;
        };

        using IdMap = std::unordered_map<std::string, Pool, string_hash_t, std::equal_to<>>;
        using OffsetMap = std::multimap<offset_t, V*>;

        /**
         * @brief SymTabDuplId iterator
         */
        class iterator {
            public:
                friend class SymTabDuplId;

                using iterator_category = std::forward_iterator_tag;
                using value_type = V;
                using difference_type = V;
                using pointer = V*;
                using reference = V&;

                explicit iterator(const OffsetMap::const_iterator &it);

                V* operator*() const;
                V* operator->() const ;

                iterator& operator++();
                iterator operator++(int);

                friend bool operator==(const iterator &l, const iterator &r) {
                    return l.entry_it_ == r.entry_it_;
                }

                friend bool operator!=(const iterator &l, const iterator &r) {
                    return l.entry_it_ != r.entry_it_;
                }

            private:
                OffsetMap::const_iterator entry_it_;
        };

        iterator begin() const;
        iterator end() const;

        size_t size() const;

        bool empty() const;

        /**
         * Returns the vector of values stored under given string id 
         */
        const SymTabDuplId<V>::Pool &at(const std::string &id) const;

        /**
         * Inserts the new entry to the tab. 
         */
        void insert(std::unique_ptr<V> &&new_entry);

        /**
         * Check whether entry with given id exists or not. 
         */
        bool exists(const std::string &id) const;
        bool exists(std::string_view id) const;

        /**
         * Gets the vector of values stored under given string id 
         */
        const SymTabDuplId<V>::Pool &getAll(const std::string &id) const;
        const SymTabDuplId<V>::Pool &getAll(std::string_view id) const;

        V *get(const std::string &id) const;
        V *get(std::string_view id) const;

        /**
         * Clears the symbol table 
         */
        void clear();

        /**
         * Removes all entries with given id 
         */
        void remove(const std::string &id);

        /**
         * Removes all elements that meets the condition given as the first
         * parameter (lambda returning bool)
         */
        template<typename Fn>
        uint32_t remove(const Fn &f, offset_t start, offset_t end);

        template<typename Fn>
        uint32_t revRemoveUntil(const Fn &f, offset_t rstart);

        /**
         * Works similar as Symtab::update
         * @warning Offset can be changed, but id of element cannot! 
         */
        template<typename Fn>
        void update(const Fn &f, offset_t start, offset_t end);

        const IdMap &dataById() const;

        const OffsetMap &dataByOffset() const;

        V *operator[](size_t index) const;
        const SymTabDuplId<V>::Pool &operator[](const std::string &id) const;

    private: 
        IdMap map_; ///< Map with ids (has the ownership)
        OffsetMap offset_map_; ///< Map with offsets 
};


//---------------------------------- Pool iterator ----------------------------

template<class V>
SymTabDuplId<V>::Pool::iterator::iterator(const DataContainer::const_iterator &it) {
    it_ = it;
}


template<class V>
V* SymTabDuplId<V>::Pool::iterator::operator*() const {
    return it_->get();
}


template<class V>
V* SymTabDuplId<V>::Pool::iterator::operator->() const {
    return it_->get();
}


template<class V>
SymTabDuplId<V>::Pool::iterator& SymTabDuplId<V>::Pool::iterator::operator++() {
    ++it_;
    return *this;
}


template<class V>
SymTabDuplId<V>::Pool::iterator SymTabDuplId<V>::Pool::iterator::operator++(int) {
    Pool::iterator result = *this;
    ++(*this);
    return result;
}


//-------------------------------------- Pool ---------------------------------

template<class V>
V *SymTabDuplId<V>::Pool::first() const {
    return data_.begin()->get();
}

template<class V>
size_t SymTabDuplId<V>::Pool::size() const {
    return data_.size();
}

template<class V>
bool SymTabDuplId<V>::Pool::empty() const {
    return data_.empty();
}

template<class V>
V* SymTabDuplId<V>::Pool::operator[](size_t i) const {
    return data_[i].get();
}

template<class V>
V* SymTabDuplId<V>::Pool::at(size_t i) const {
    return data_.at(i).get();
}

template<class V>
SymTabDuplId<V>::Pool::iterator SymTabDuplId<V>::Pool::begin() const {
    return iterator(data_.begin());
}

template<class V>
SymTabDuplId<V>::Pool::iterator SymTabDuplId<V>::Pool::end() const {
    return iterator(data_.end());
} 

template<class V>
std::vector<std::unique_ptr<V>> &SymTabDuplId<V>::Pool::data() {
    return data_;
}

template<class V>
const std::vector<std::unique_ptr<V>> &SymTabDuplId<V>::Pool::rd_data() const {
    return data_;
}

template<class V>
void SymTabDuplId<V>::Pool::erase(V* item) {
    for(auto pool_data_it = data_.begin(); 
        pool_data_it != data_.end(); 
        ++pool_data_it) {

        if(pool_data_it->get() == item) { ///< Related entry is found
            data_.erase(pool_data_it);
            break;
        }
    }
}


//----------------------------------- Iterator --------------------------------

template<class V>
SymTabDuplId<V>::iterator::iterator(const OffsetMap::const_iterator &it) {
    entry_it_ = it;
}

template<class V>
V* SymTabDuplId<V>::iterator::operator*() const {
    return entry_it_->second;
}

template<class V>
V* SymTabDuplId<V>::iterator::operator->() const {
    return entry_it_->second;
}

template<class V>
SymTabDuplId<V>::iterator& SymTabDuplId<V>::iterator::operator++() {
    ++entry_it_;
    return *this;
}

template<class V>
SymTabDuplId<V>::iterator SymTabDuplId<V>::iterator::operator++(int) {
    SymTabDuplId<V>::iterator result = *this;
    ++(*this);
    return result;
}


//------------------------------SymTabDuplId-----------------------------------

template<class V>
SymTabDuplId<V>::iterator SymTabDuplId<V>::begin() const {
    return iterator(offset_map_.begin());
}

template<class V>
SymTabDuplId<V>::iterator SymTabDuplId<V>::end() const {
    return iterator(offset_map_.end());
}


template<class V>
size_t SymTabDuplId<V>::size() const {
    return offset_map_.size();
}


template<class V>
bool SymTabDuplId<V>::empty() const {
    return map_.empty();
}


template<class V>
const SymTabDuplId<V>::Pool &SymTabDuplId<V>::at(const std::string &id) const {
    return map_.at(id);
}


template<class V>
void SymTabDuplId<V>::insert(std::unique_ptr<V> &&new_entry) {
    offset_map_.emplace(new_entry->getOffset(), new_entry.get());
    map_[new_entry->getId()].data().emplace_back(std::move(new_entry));
}


template<class V>
bool SymTabDuplId<V>::exists(const std::string &id) const {
    return map_.find(id) != map_.end();
}


template<class V>
bool SymTabDuplId<V>::exists(std::string_view id) const {
    return map_.find(id) != map_.end();
}



template<class V>
const SymTabDuplId<V>::Pool &SymTabDuplId<V>::getAll(const std::string &id) const {
    auto result_it = map_.find(id);

    if(result_it == map_.end()) {
        throw InternalErrorException("Item does not exists in symbol tab!");
    }

    return result_it->second;
}


template<class V>
const SymTabDuplId<V>::Pool &SymTabDuplId<V>::getAll(std::string_view id) const {
    auto result_it = map_.find(id);

    if(result_it == map_.end()) {
        throw InternalErrorException("Item does not exists in symbol tab!");
    }

    return result_it->second;
}


template<class V>
V *SymTabDuplId<V>::get(const std::string &id) const {
    return getAll(id).first();
}


template<class V>
V *SymTabDuplId<V>::get(std::string_view id) const {
    return getAll(id).first();
}


template<class V>
void SymTabDuplId<V>::clear() {
    map_.clear();
    offset_map_.clear();
}


template<class V>
void SymTabDuplId<V>::remove(const std::string &id) {
    for(auto it = offset_map_.begin(); it != offset_map_.end();) {
        if(it->second->getId() == id) {
            it = offset_map_.erase(it);
        }
        else {
            ++it;
        }
    }

    map_.erase(id);
}


template<class V>
template<typename Fn>
uint32_t SymTabDuplId<V>::remove(const Fn &f, offset_t start, offset_t end) {
    uint32_t deleted = 0;
    auto start_it = offset_map_.lower_bound(start);
    auto end_it = offset_map_.upper_bound(end);

    for(auto it = start_it; it != end_it;) {
        if(f(*(it->second))) { ///< Check if condition given by user is met
            const auto &pool_it = map_.find(it->second->getId());

            // If there is pool indexed by given id (it should be), find the corresponding entry 
            if(pool_it != map_.end()) { 
                pool_it->second.erase(it->second);

                if(pool_it->second.empty()) { ///< Erase the pool
                    map_.erase(pool_it);
                }
            }

            it = offset_map_.erase(it);
        }
        else {
            ++it;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
uint32_t SymTabDuplId<V>::revRemoveUntil(const Fn &f, offset_t rstart) {
    uint32_t deleted = 0;
    auto start_it = offset_map_.upper_bound(rstart);

    for(auto it = std::make_reverse_iterator(start_it); it != offset_map_.rend();) {
        if(f(*(it->second))) { ///< Check if condition given by user is met
            const auto &pool_it = map_.find(it->second->getId());

            // If there is pool indexed by given id (it should be), find the corresponding entry 
            if(pool_it != map_.end()) { 
                pool_it->second.erase(it->second);

                if(pool_it->second.empty()) { ///< Erase the empty pool
                    map_.erase(pool_it);
                }
            }

            auto fw_it = std::next(it).base();
            it = std::make_reverse_iterator(offset_map_.erase(fw_it));
        }
        else {
            break;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
void SymTabDuplId<V>::update(const Fn &f, offset_t start, offset_t end) {
    auto start_it = offset_map_.lower_bound(start);
    auto end_it = offset_map_.upper_bound(end);

    std::vector<std::pair<offset_t, V*>> to_be_updated;
    for(auto it = start_it; it != end_it; ++it) {
        to_be_updated.emplace_back(it->first, it->second);
    }

    for(auto &offset_pair: to_be_updated) {
        // Extract the node (key can be changed)
        auto eq_range = offset_map_.equal_range(offset_pair.first);

        for(auto it = eq_range.first; it != eq_range.second; ++it) {
            if(it->second == offset_pair.second) {
                auto entry_node = offset_map_.extract(it);

                f(entry_node.mapped()); // Update entry

                entry_node.key() = entry_node.mapped()->getOffset(); ///< Update the key
                offset_map_.insert(std::move(entry_node)); // Move the node back
            
                break;
            }
        }
    }

}


template<class V>
const SymTabDuplId<V>::IdMap &SymTabDuplId<V>::dataById() const {
    return map_;
}


template<class V>
const SymTabDuplId<V>::OffsetMap &SymTabDuplId<V>::dataByOffset() const {
    return offset_map_;
}


template<class V>
V *SymTabDuplId<V>::operator[](size_t index) const {
    if(index >= offset_map_.size()) {
        throw std::out_of_range("Index out of range!");
    }

    auto it = offset_map_.begin();
    std::advance(it, index);

    return it->second;
}


template<class V>
const SymTabDuplId<V>::Pool &SymTabDuplId<V>::operator[](const std::string &id) const {
    try {
        return getAll(id);
    }
    catch(const std::exception &) {
        throw std::out_of_range("Id does not exists in symtab!");
    }
}


/**
 * @brief Symbol tab for symbols that are indexed only by offsets (which 
 * can be duplicated)
 */
template<class V>
class SymTabOffset {
    public:
        using OffsetMap = std::multimap<offset_t, std::unique_ptr<V>>;

        SymTabOffset() = default;

        SymTabOffset(const SymTabOffset &) = delete;

        SymTabOffset &operator= (const SymTabOffset &) = delete;

        /**
         * @brief SymTabOffset iterator
         */
        class iterator {
            public:
                friend class SymTabOffset;

                using iterator_category = std::forward_iterator_tag;
                using value_type = std::unique_ptr<V>;
                using difference_type = std::unique_ptr<V>;
                using pointer = std::unique_ptr<V>*;
                using reference = std::unique_ptr<V>&;

                explicit iterator(const OffsetMap::const_iterator &it);

                V* operator*() const;
                V* operator->() const;

                iterator& operator++();
                iterator operator++(int);

                friend bool operator==(const iterator &l, const iterator &r) {
                    return l.entry_it_ == r.entry_it_;
                }

                friend bool operator!=(const iterator &l, const iterator &r) {
                    return !(l == r);
                }

            private:
                OffsetMap::const_iterator entry_it_;
        };

        iterator begin() const;
        iterator end() const;

        size_t size() const;

        bool empty() const;

        /**
         * Inserts the new entry  
         */
        void insert(std::unique_ptr<V> &&new_entry);

        /**
         * Checks if there is any entry on given offset 
         */
        bool exists(const offset_t &offset) const;

        const std::unique_ptr<V> &get(const offset_t &offset) const;

        void clear();

        void remove(const offset_t &offset);
        void remove(const SymTabOffset<V>::OffsetMap::const_iterator &it);

        template<typename Fn>
        uint32_t remove(const Fn &f);

        /**
         * Removes the elements, that meet the condition given by lambda (first
         * argument) between start and end offset.
         */
        template<typename Fn>
        uint32_t remove(const Fn &f, offset_t start, offset_t end);

        template<typename Fn>
        uint32_t revRemoveUntil(const Fn &f, offset_t rstart);

        /**
         * Update all entries by given lambda function. 
         */
        template<typename Fn>
        void update(const Fn &f);

        /**
         * Update all entries between start and end offsets.
         */
        template<typename Fn>
        void update(const Fn &f, offset_t start, offset_t end);

        const OffsetMap &data() const;

        V *operator[](size_t index) const;

    private:
        OffsetMap map_;
};


//--------------------------------- Iterator ----------------------------------

template <class V>
SymTabOffset<V>::iterator::iterator(const OffsetMap::const_iterator &it) {
    entry_it_ = it;
}

template <class V>
V* SymTabOffset<V>::iterator::operator*() const {
    return entry_it_->second.get();
}

template <class V>
V* SymTabOffset<V>::iterator::operator->() const {
    return entry_it_->second.get();
}

template <class V>
SymTabOffset<V>::iterator& SymTabOffset<V>::iterator::operator++() {
    ++entry_it_;
    return *this;
}

template <class V>
SymTabOffset<V>::iterator SymTabOffset<V>::iterator::operator++(int) {
    SymTabOffset<V>::iterator result = *this;
    ++(*this);
    return result;
}



//---------------------------------SymTabOffset--------------------------------

template<class V>
SymTabOffset<V>::iterator SymTabOffset<V>::begin() const {
    return iterator(map_.begin());
}


template<class V>
SymTabOffset<V>::iterator SymTabOffset<V>::end() const {
    return iterator(map_.end());
}


template<class V>
size_t SymTabOffset<V>::size() const {
    return map_.size();
}


template<class V>
bool SymTabOffset<V>::empty() const {
    return map_.empty();
}


template<class V>
void SymTabOffset<V>::insert(std::unique_ptr<V> &&new_entry) {
    map_.emplace(new_entry->getOffset(), std::move(new_entry));
}


template<class V>
bool SymTabOffset<V>::exists(const offset_t &offset) const {
    return map_.find(offset) != map_.end();
}


template<class V>
const std::unique_ptr<V> &SymTabOffset<V>::get(const offset_t &offset) const {
    auto result_it = map_.find(offset);

    if(result_it == map_.end()) {
        throw InternalErrorException("Item does not exists in symbol offset tab!");
    }

    return result_it->second;
}


template<class V>
void SymTabOffset<V>::clear() {
    map_.clear();
}


template<class V>
void SymTabOffset<V>::remove(const offset_t &offset) {
    map_.erase(offset);
}


template<class V>
void SymTabOffset<V>::remove(const SymTabOffset<V>::OffsetMap::const_iterator &it) {
    map_.erase(it);
}


template<class V>
template<typename Fn>
uint32_t SymTabOffset<V>::remove(const Fn &f) {
    return remove(f, 0, static_cast<offset_t>(-1));
}


template<class V>
template<typename Fn>
uint32_t SymTabOffset<V>::remove(const Fn &f, offset_t start, offset_t end) {
    uint32_t deleted = 0;

    //Find the boundaries
    auto start_it = map_.lower_bound(start);
    auto end_it = map_.upper_bound(end);

    for(auto it = start_it; it != end_it;) {
        if(f(*(it->second))) { ///< Erase the elements that meet given condition
            it = map_.erase(it);
            deleted++;
        }
        else {
            ++it;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
uint32_t SymTabOffset<V>::revRemoveUntil(const Fn &f, offset_t rstart) {
    uint32_t deleted = 0;

    auto start_it = map_.upper_bound(rstart);

    for(auto it = std::make_reverse_iterator(start_it); it != map_.rend();) {
        if(f(*(it->second))) { ///< Erase the elements that meet given condition
            auto fw_it = std::next(it).base();
            it = std::make_reverse_iterator(map_.erase(fw_it));
            deleted++;
        }
        else {
            break;
        }
    }

    return deleted;
}


template<class V>
template<typename Fn>
void SymTabOffset<V>::update(const Fn &f) {
    update(f, 0, static_cast<offset_t>(-1));
}


template<class V>
template<typename Fn>
void SymTabOffset<V>::update(const Fn &f, offset_t start, offset_t end) {
    auto start_it = map_.lower_bound(start);
    auto end_it = map_.upper_bound(end);

    std::vector<std::pair<offset_t, V*>> to_be_updated;
    for(auto it = start_it; it != end_it; ++it) {
        to_be_updated.emplace_back(it->first, it->second.get());
    }

    for(auto &offset_pair: to_be_updated) {
        // Extract the node (key can be changed)
        auto eq_range = map_.equal_range(offset_pair.first);

        for(auto it = eq_range.first; it != eq_range.second; ++it) {
            if(it->second.get() == offset_pair.second) {
                auto entry_node = map_.extract(it);

                f(entry_node.mapped()); // Update entry

                entry_node.key() = entry_node.mapped()->getOffset(); ///< Update the key
                map_.insert(std::move(entry_node)); // Move the node back

                break;
            }
        }
    }
}


template<class V>
const SymTabOffset<V>::OffsetMap &SymTabOffset<V>::data() const {
    return map_;
}


template<class V>
V *SymTabOffset<V>::operator[](size_t index) const {
    if(index >= map_.size()) {
        throw std::out_of_range("Index out of range!");
    }

    auto it = map_.begin();
    std::advance(it, index);

    return it->second.get();
}
