/**
 * @file string.h 
 * @brief Declaration of classes String, StringModifier and 
 * StringModifierContainer
 * 
 * @author Vojtěch Dvořák 
 */


#pragma once

#include "types.h"
#include "common.h"
#include "rule_element.h"
#include <string>
#include <memory>
#include <variant>
#include <unordered_set>
#include <unordered_map>

/**
 * The length of base64 alphabet argument 
 */
#define BASE64_ALPHA_LEN 64

/**
 * Maximum value, that can apper in argument of xor modifier 
 */
#define XOR_UPPER_RANGE_BOUND 255

/**
 * Maximum value of repeat interval in Regular expressions 
 */
#define REGEXP_MAXIMUM_QUANTIFIER_NUM 32767

class YaraFile;
class Literal;
class String;
class StringModifier;


/**
 * @brief Represents modifier of yara string 
 */
class StringModifier : public RuleElement, public Printable {
    public:
    
        /**
         * @brief All valid types of yara strings modifiers 
         */
        enum class Type {
            NoCase,
            Wide,
            Ascii,
            Xor,
            Base64,
            Base64Wide,
            Fullword, 
            Private
        };

        using Range = std::pair<uint8_t, uint8_t>; 
        using Key = uint8_t;
        using Alphabet = std::array<char, BASE64_ALPHA_LEN>;
        using Arg = std::variant<std::monostate, Range, Alphabet, Key>;

        StringModifier(StringModifier::Type type);

        /**
         * Returns type of string modifier
         */
        StringModifier::Type getType() const;

        /**
         * Sets the argument of modifier
         */
        void setArg(const Key &key);
        void setArg(const Range &range);
        void setArg(const Alphabet &alphabet);
        void setArg(const std::string &alphabet_str);

        /**
         * Checks if modifier has any argument
         */
        bool hasAnyArg() const;

        /**
         * Checks if modifier has argument of type T
         */
        template <typename T>
        bool hasArg() const;

        /**
         * Returns reference to argument of modifier
         */
        const Arg &getArg() const;

        /**
         * Converts string view to related StringModifier::Type
         */
        static StringModifier::Type stringToType(std::string_view string);
        
        /**
         * Converts StringModifier::Type to readable string view
         */
        static std::string_view typeToString(StringModifier::Type type);

        std::stringstream getTextFormatted() const override;

    private:
        Arg arg_ = std::monostate();

        const StringModifier::Type type_; ///< Type of rule modifier

        static const std::unordered_map<StringModifier::Type, std::string_view> type_to_string_; ///< Provides mapping between StringModifiery::Type value and readable string
        static const std::unordered_map<std::string_view, StringModifier::Type> string_to_type_; ///< Provides reverse mapping for type_to_string_
};


template <typename T>
bool StringModifier::hasArg() const {
    return std::holds_alternative<T>(arg_);
}




/**
 * @brief Iterable container for string modifier 
 */
class StringModifierContainer {
    public:

        /**
         * @brief Iterator for StringModifierContainer
         */
        class iterator {
            public:
                friend class StringModifierContainer;

                using iterator_category = std::forward_iterator_tag;
                using value_type = StringModifier;
                using difference_type = StringModifier;
                using pointer = StringModifier*;
                using reference = StringModifier&;

                explicit iterator(const StringModifierContainer *container);

                StringModifier* operator*() const;
                StringModifier* operator->() const;

                iterator& operator++();
                iterator operator++(int);

                friend bool operator==(const iterator &l, const iterator &r) {
                    if(l.container_ == nullptr || r.container_ == nullptr) {
                        return l.container_ == r.container_; // Both cntnr ptrs must be null
                    }
                    else {
                        return l.container_ == r.container_ && l.it_ == r.it_;
                    }
                }

                friend bool operator!=(const iterator &l, const iterator &r) {
                    if(l.container_ == nullptr || r.container_ == nullptr) {
                        return l.container_ != r.container_;
                    }
                    else {
                        return l.container_ != r.container_ || l.it_ != r.it_;
                    }
                }

            private:
                const StringModifierContainer *container_ = nullptr;
                std::vector<std::unique_ptr<StringModifier>>::const_iterator it_;
        };
    

        StringModifierContainer() = default;
        StringModifierContainer(const StringModifierContainer &) = delete;
        StringModifierContainer &operator= (const StringModifierContainer &) = delete;
     
        /**
         * Returns number of elements in string modifier container 
         */
        size_t size() const;

        /**
         * Checks if there is at least one string modifier in container 
         */
        bool empty() const;

        /**
         * Checks if there is string modifier with given type in container
         */
        bool has(StringModifier::Type type) const;
      
        /**
         * Returns pointer to string modifier in container with given type
         * @return ptr to string modifier 
         */
        StringModifier *get(StringModifier::Type type) const;

        /**
         * Returns begin iterator of container
         * @return iterator of container
         */
        StringModifierContainer::iterator begin() const;
        
        /**
         * Returns end iterator of container 
         * @return iterator of container
         */
        StringModifierContainer::iterator end() const;

        /**
         * Adds modifier to container
         * @note If there is modifier with the same type it is erased
         */
        void add(std::unique_ptr<StringModifier> &&new_mod);

        /**
         * Erases the modifier with given type
         * @note If there is no such element, nothing is done
         */
        void erase(StringModifier::Type type);

    private:
        std::vector<std::unique_ptr<StringModifier>> modifiers_;
};


/**
 * @brief Represents yara string 
 */
class String : public RuleElement, public Printable {
    public:

        /**
         * @brief All valid types of yara strings 
         */
        enum class Type {
            Plain, ///< Plain text string
            Hex, ///< Hex string
            Regexp, ///< Regular expression string
        };

        using ValidModsTab = std::unordered_map<String::Type, std::unordered_set<StringModifier::Type>>;

        String() = default;
        String(String::Type type);

        String(const String &) = delete;
        String &operator= (const String &) = delete;

        /**
         * Sets id of string ('$' must be omitted) 
         */
        void setId(const std::string &id);

        /**
         * Returns reference to string with id of string
         * @param ignore_internal_id If it is set to true, id of anonymous string is ignored and empty string is returned (false by default)  
         */
        const std::string &getId(bool ignore_internal_id = false) const;

        /**
         * Returns type of string
         */
        String::Type getType() const;

        /**
         * Sets the content of string
         * @note Content is represented only as string
         */
        void setContent(const std::string &content);
        
        /**
         * Returns reference to string content of the string
         */
        const std::string &getContent() const;

        /**
         * Adds new modifier to modifier container
         */
        void addModifier(std::unique_ptr<StringModifier> &&new_mod);
        
        /**
         * Returns reference to container with string modifier
         */
        const StringModifierContainer &getModifiers() const;

        /**
         * Returns value of anonymous flag of the string
         * @note Anonymous flag determines, whether id of string was only defined as '$'
         */
        bool isAnonymous() const;

        /**
         * Sets the value of anonymous flag of string
         */
        void setAnonymous(bool is_anonymous);

        /**
         * Converts String::Type to readable string view
         */
        static std::string_view typeToString(String::Type type);
        

        std::stringstream getTextFormatted() const override;

    private:
        static const ValidModsTab valid_mods_; ///< Table of valid combinations of String::Types and Modifier::Types

        std::string id_; ///< Id of string (even anonymous string has this attribute set to internal id value)
        std::string content_; ///< Content of string

        const String::Type type_;

        StringModifierContainer mods_;

        bool is_anonymous_ = false;

        bool isModValid_(const std::unique_ptr<StringModifier> &mod);
};
