/**
 * @file rule.h 
 * @brief Contains declaration of class, that represents yara rule
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once


#include "types.h"
#include "yara_file_element.h"
#include "symtab.h"
#include "common.h"
#include "expression.h"
#include <string>
#include <unordered_map>
#include <unordered_set>
#include <iostream>

#include "forward.h"



/**
 * @brief Class that represents rule modifiers 
 */
class RuleModifier : public YaraFileElement {
    public:

        /**
         * @brief Type of rule the rule modifier 
         */
        enum class Type {
            Private, ///< Private rule
            Global ///< Global rule
        };

        RuleModifier(RuleModifier::Type type);

        /**
         * Provides conversion from string view to RuleModifier::Type 
         */
        static Type stringToType(std::string_view str);

        /**
         * Provides conversion RuleModifier::Type to string view 
         */
        static std::string_view typeToString(Type type);

        /**
         * Returns type of the rule modifier 
         */
        RuleModifier::Type getType();

    private:
        const RuleModifier::Type type_;

};


/**
 * Container for rule modifiers of rule. Because in YARA rule can have only
 * two modifiers (global and private), this container should be sufficient.  
 */
class RuleModifierContainer {
    public:

        /**
         * @brief Iterator for RuleModifierContainer
         */
        class iterator {
            public:
                friend class RuleModifierContainer;

                using iterator_category = std::forward_iterator_tag;
                using value_type = RuleModifier;
                using difference_type = RuleModifier;
                using pointer = RuleModifier*;
                using reference = RuleModifier&;

                explicit iterator(const RuleModifierContainer *container);

                RuleModifier* operator*() const;
                RuleModifier* operator->() const;

                iterator& operator++();
                iterator operator++(int);

                friend bool operator==(const iterator &l, const iterator &r) {
                    if(l.container_ == nullptr || r.container_ == nullptr) {
                        return l.container_ == r.container_; // Both cntnr ptrs must be null
                    }
                    else {
                        return l.container_ == r.container_ && l.type_ == r.type_;
                    }
                }

                friend bool operator!=(const iterator &l, const iterator &r) {
                    if(l.container_ == nullptr || r.container_ == nullptr) {
                        return l.container_ != r.container_;
                    }
                    else {
                        return l.container_ != r.container_ || l.type_ != r.type_;
                    }
                }

            private:
                const RuleModifierContainer *container_ = nullptr;
                RuleModifier::Type type_;
        };

        RuleModifierContainer() = default;
        RuleModifierContainer(const RuleModifierContainer &) = delete;
        RuleModifierContainer &operator= (const RuleModifierContainer &) = delete;

        /**
         * Returns the number of rule modifiers 
         */
        size_t size() const;
        
        /**
         * Checks whether there is at least one modifier in container 
         */
        bool empty() const;

        /**
         * Checks if container contains modifier with given type 
         * @return true if container contains modifier with given type
         */
        bool has(RuleModifier::Type type) const;

        /**
         * Checks if container contains global modifier
         */
        bool hasGlobal() const;

        /**
         * Checks if container contains private modifier
         */
        bool hasPrivate() const;

        /**
         * Returns pointer to global modifier
         */
        RuleModifier *getGlobal() const;

        /**
         * Returns pointer to private modifier
         */
        RuleModifier *getPrivate() const;

        /**
         * Returns begin iterator of container 
         */
        RuleModifierContainer::iterator begin() const;

        /**
         * Returns end iterator of container 
         */
        RuleModifierContainer::iterator end() const;

        /**
         * Adds modifier to the container 
         */
        void add(std::unique_ptr<RuleModifier> &&new_mod);

        /**
         * Erases modifier with given type from the container
         * @note It does nothing, if there is no such modifier
         */
        void erase(RuleModifier::Type type);

    private:
        std::unique_ptr<RuleModifier> private_ = nullptr;
        std::unique_ptr<RuleModifier> global_ = nullptr;
};


/**
 * @brief Class that represents yara rule 
 * 
 * Represents the whole sequence of tokens, that starts
 * with rule modifier and ends with '}'
 */
class Rule : public TopLevelYaraFileElement, public Printable {
    public:
        Rule() = default;
        ~Rule();

        Rule(const Rule &) = delete;
        Rule &operator= (const Rule &) = delete;

        /**
         * Returns rule id 
         * @return const reference to string with rule id 
         */
        const std::string &getId() const;

        /**
         * Sets the id of the rule 
         */
        void setId(const std::string &id);

        /**
         * Get all tags of the rule
         * @return unordered set with tags
         */
        const std::vector<std::string> &getTags() const;

        /**
         * Adds tag to the rule tags 
         */
        void addTag(const std::string &new_tag);

        /**
         * Checks whether the rule has given tag 
         */
        bool hasTag(const std::string &tag);

        /**
         * Removes tag from the rule tags 
         */
        void removeTag(const std::string &tag);

        /**
         * Returns all rule modifiers 
         */
        const RuleModifierContainer &getModifiers() const;

        /**
         * Adds modifier to the rule
         * @note If rule already has the modifier with the same type exception is thrown 
         */
        void addModifier(std::unique_ptr<RuleModifier> &&new_mod);

        /**
         * Removes modifier with given type from the rule modifiers 
         */
        void removeModifier(RuleModifier::Type type);

        /**
         * Check whether the rule has global modifier 
         */
        bool isGlobal();

        /**
         * Check whether the rule has private modifier 
         */
        bool isPrivate();

        /**
         * Returns all metas (symtab, that contains them) of the rule
         */
        const SymTabDuplId<Meta> &getMetas() const;

        /**
         * Returns pool with metas, that have the same id, that is given as
         * parameter of this method
         */
        const SymTabDuplId<Meta>::Pool &getMetasById(const std::string &id) const;

        /**
         * Adds meta to symbol tab of this rule 
         */
        void addMeta(std::unique_ptr<Meta> &&new_meta);

        /**
         * Removes all metas with given id from the symbol tab
         */
        void removeMeta(const std::string &id);

        /**
         * Returns reference to symtab with all strings of the rule
         */
        const SymTab<String> &getStrings() const;

        /**
         * Returns rule string, that has given id 
         */
        const std::unique_ptr<String> &getStringById(const std::string &id) const;

        /**
         * Adds string to the rule 
         */
        void addString(std::unique_ptr<String> &&new_string);

        /**
         * Removes string from the symtab of rule strings 
         */
        void removeString(const std::string &id);

        /**
         * Returns symbol tab with all internal variables of the rule
         * @note internal variables are Avast feature of YARA 
         */
        const SymTab<IntVariable> &getVars() const;

        /**
         * Returns reference to specific internal variable
         */
        const std::unique_ptr<IntVariable> &getVarById(const std::string &id) const;

        /**
         * Adds the new internal variable
         */
        void addVar(std::unique_ptr<IntVariable> &&new_var);

        /**
         * Removes internal variable with given id from the symtab
         */
        void removeVar(const std::string &id);

        /**
         * Sets the condition expression 
         */
        void setCondition(ExpressionPtr &&condition_expr);
        void setCondition(const ExpressionPtr &condition_expr);

        /**
         * Returns reference to the condition expression 
         */
        const ExpressionPtr &getCondition() const;

        /**
         * Returns global offset of rule id 
         */
        offset_t getIdOffset() const;

        /**
         * Sets the global offset of the rule id 
         */
        void setIdOffset(const offset_t &offset);


        std::stringstream getTextFormatted() const;

    private:
        offset_t id_start_ = 0;
        std::string id_ = std::string({});

        std::vector<std::string> tags_; ///< Tags of the rule

        RuleModifierContainer mods_; ///< Modifiers of the rule

        ExpressionPtr condition_; ///< Condition of the rule

        SymTabDuplId<Meta> metas_; ///< Metas of the rule

        SymTab<String> strings_; ///< Symbol tab fro strings
        SymTab<IntVariable> vars_; ///< Symbol tab fro variables

        size_t anonym_str_cnt = 0; ///< Counter of anonymous strings for generating unique internal IDs for them
};
