/**
 * @file other.h 
 * @brief Classes, that represents less complex syntax structures (such as
 * includes, imports, comments...) 
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once

#include "types.h"
#include <string>
#include <filesystem>
#include <json/json.hpp>

#include "forward.h"

/**
 * @brief Class representing comments (all valid types in YARA language) 
 */
class Comment : public TopLevelYaraFileElement, public Printable {
    public:
        Comment(const std::string &text, const offset_t &pos);
        ~Comment();

        const std::string &getText() const;

        std::stringstream getTextFormatted() const override;
    private:
        std::string text_;
};


/**
 * @brief Class representing "include <path>" structure in file 
 */
class FileInclude : public TopLevelYaraFileElement, public Printable {
    public:
        ~FileInclude();

        /**
         * Returns id of file include - the id is in this case same as path 
         */
        const std::string &getId() const;

        void setPath(const std::string &path, const std::string &base_path);

        /**
         * Returns path of included file 
         */
        const std::string &getPath() const;

        void setFile(const std::shared_ptr<YaraFile> &yara_file);

        /**
         * Returns pointer to included YaraFile 
         */
        std::shared_ptr<YaraFile> getFile() const;

        std::stringstream getTextFormatted() const override;
    private:
        std::string path_;

        std::shared_ptr<YaraFile> yara_file_; ///< Pointer to object, that represents included file
};

/**
 * @brief Class representing "import <module>" structure in file 
 */
class Import : public TopLevelYaraFileElement, public Printable {
    public:
        Import();
        ~Import();

        /**
         * Returns id of import - the id is in this case same as if of 
         * imported module
         */
        const std::string &getId() const;

        void setModuleName(const std::string &module_name);
        const std::string &getModuleName() const;

        void setModule(const std::shared_ptr<Module> &module);
        std::shared_ptr<Module> getModule() const;

        std::stringstream getTextFormatted() const override;
    private:
        std::string mod_name_; ///< Module name

        std::shared_ptr<Module> module_; ///< Pointer to object, that represents imported module
};

