/**
 * @file module.h 
 * @brief Classes for handling modules (their description in JSON) 
 * and their parsing
 * 
 * @author Vojtěch Dvořák    
 */

#pragma once


#include "json/json.hpp"
#include "expression.h"
#include <unordered_map>
#include <fstream>
#include <iostream>

#include "forward.h"


#define USE_MODULE_CACHE ///< Comment to disable using of module cache


/**
 * @brief Singleton class that provides access to loaded modules. Modules 
 * are loaded, when Yaramod class is instantiated.
 */
class ModuleProvider {
    public:
        using ModuleMap = std::unordered_map<std::string, std::shared_ptr<Module>, string_hash_t, std::equal_to<>>;
        using SymbolCache = std::unordered_map<std::string, const json *, string_hash_t, std::equal_to<>>;

        /**
         * Returns singleton instance of ModuleProvider 
         */
        static ModuleProvider &getInstance();

        /**
         * Checks whether module with given name was loaded to module provider 
         * or not 
         */
        bool hasModule(const std::string &name) const;
        bool hasModule(const std::string_view name) const;

        /**
         * Adds module to ModuleProvider (this module can be accessed later by 
         * getModule) 
         */
        void addModule(const std::shared_ptr<Module> &module);

        /**
         * Returns pointer to module with given name, that was stored to 
         * module provider. Module, that was returned by this function must 
         * not be deleted! (ModuleProvider deletes it automatically)
         */
        std::shared_ptr<Module> getModule(const std::string &name) const;
        std::shared_ptr<Module> getModule(std::string_view name) const;

        /**
         * Clears the symbol cache 
         */
        void clearCache();

        /**
         * Find cached symbol context (by full symbol name) 
         */
        const json *findCached(std::string_view full_symbol_name) const;

        /**
         * Creates entry in cache for the given symbol json ctx 
         */
        void cacheSymbol(const std::string &parent_namespace, const json *symbol_ctx);

    private:
        ModuleProvider(); ///< Hidden constructor
        ~ModuleProvider();

        ModuleProvider(ModuleProvider &module_provider) = delete; ///< Disable cloning
        void operator=(const ModuleProvider &) = delete; ///< Disable assigning
    
        static ModuleMap modules_; ///< Storage for modules

        SymbolCache symbol_cache_;
};


/**
 *@brief Class representing module (its description) 
 */
class Module {
    public:
        Module(const std::string &path);
        Module(std::string_view path);

        const std::string &getName() const;
        
        /**
         * Parses file in JSON format with module description to JSON object 
         * to be easily traversed 
         */
        void load();

        /**
         * Returns pointer to module data (JSON object) 
         */
        const std::unique_ptr<json> &getData() const;

        /**
         * Merges two modules (their data) 
         */
        void merge(const std::shared_ptr<Module> &other);

        /**
         * Recursively merges two JSON objects
         */
        static void mergeJson(json *dst, const json *src);

        /**
         * Converts string in JSON to Expression::Type 
         */
        static Expression::Type jsonToExprType(const json *json_type);

        /**
         * Finds object with given name in JSON array
         * @return pointer to object with given name or nullptr 
         */
        static const json *findInArray(const json *json, std::string_view obj_name);

        /**
         * Finds field with given name in JSON object
         * @return pointer to field value with given name or nullptr 
         */
        static const json *findField(const json *json, std::string_view field_name);

        /**
         * Gets data type of given JSON object
         * @return Data type represented as Expression::Type
         */
        static Expression::Type getDataType(const json *ctx_obj);

        /**
         * Tries to find JSON object that is specified by given string
         * @return JSON object with context that corresponds to the given string
         */
        static const json *qualifyObject(std::string_view qual_str, const json *obj);

        /**
         * Find data type of element of array in given JSON object
         * @note Wrapper above Module::getDataType
         */
        static Expression::Type getElementDataType(const json *ctx_obj);

        /**
         * Finds type of element of array in given JSON object
         */
        static Symbol::Type getElementType(const json *ctx_obj);

        /**
         * Builds semantic context of given Symbol object from given
         * JSON object
         */
        static void buildModuleContext(const std::shared_ptr<Symbol> &identifier, const json *ctx_obj);

    private:
        const std::string path_; ///< Path to the module

        std::string name_; ///< The name of top level object in json description

        std::unique_ptr<json> data_; ///< Parsed json data
};