/**
 * @file literal.h 
 * @brief Contains declaration of Literal class, that represents literals
 * in YaraFile
 *
 * @author Vojtěch Dvořák 
 */

#pragma once

#include "common.h"
#include <variant>
#include <sstream>
#include <string>
#include <iomanip>

using LiteralValue = std::variant<std::monostate, std::string, int64_t, float, bool>;


/**
 * @brief Object that represents literal, that may appear inside a rule 
 * definition 
 */
class Literal : public Printable {
    public:
        Literal();

        template <typename T>
        Literal(T new_value);

        /**
         * Sets the value of literal 
         */
        template <typename T>
        void set(T new_value);

        /**
         * Returns the new value of literals (std::get can be used to get 
         * concrete value) 
         */
        const LiteralValue &getRaw() const;

        /**
         * Returns the concrete value of Literal 
         */
        template <typename T>   
        const T &get() const;

        /**
         * Checks if Literal contains the value of given type
         * @return true if Literal contains value of provided data type 
         */
        template <typename T>
        bool isType() const;

        /**
         * Checks if Literal has undefined value (same as getType<std::monostate>)
         */
        bool isUndefined() const;

        bool isString() const;
        bool isInt() const;
        bool isFloat() const;
        bool isBool() const;
 
        std::stringstream getTextFormatted() const;

    private:
        LiteralValue value_;
};


template <typename T>
Literal::Literal(T new_value) : value_(new_value) {
}


template <typename T>
void Literal::set(T new_value) {
    value_ = new_value;
}


template <typename T>
bool Literal::isType() const {
    return std::holds_alternative<T>(value_);
}


template <typename T>
const T &Literal::get() const {
    return std::get<T>(value_);
}

