/**
 * @file file_context.h 
 * @brief Contains declaration of FileContext class
 * 
 * @author Vojtěch Dvořák 
 */


#include "types.h"
#include "common.h"
#include "tree_sitter/api.h"
#include <string>
#include <memory>


/**
 * @brief Wrapper for all necessary data to perform incremental 
 * reparsing (they are concrete enough)
 */
class FileContext {
    public:
        FileContext();
        ~FileContext();

        /**
         * @brief Entry in stored in cache
         */
        struct cache_entry_t {
            offset_t offset;
            point_t point;
        };

        /**
         * @brief Cache of FileContext for offset_t <-> point_t conversion
         */
        struct cache_t {
            FileContext::cache_entry_t entry;
            bool is_valid = false; 
        };

        /**
         * Returns reference to internal buffer of parsing context
         */
        const std::string &getString() const;

        /**
         * Returns reference to internal buffer of parsing context with temporary (edited) content of the file
         */
        const std::string &getTempString() const;

        /**
         * Sets the new content of the internal buffer
         */
        void setString(const std::string &string);

        /**
         * Partly updates temporary internal string buffer 
         */
        void stringEdit(const offset_t &offset, const size_t &size, const std::string &new_text);

        /**
         * Performs update of internal buffer, all edits since last 
         * FileContext::stringUpdate call are saved into internal buffer
         */
        void stringUpdate();

        /**
         * Discards string edits, that were done since last FileContext::stringUpdate call
         */
        void undoEdits();

        /**
         * Returns internal pointer to structure with syntax tree
         */
        TSTree *getTree() const;

        /**
         * Sets the internal ptr to internal structure with syntax tree to 
         * the new ptr
         */
        void setTree(TSTree *tree); 

        /**
         * Clears the internal structure with syntax tree
         */
        void clearTree(); 

        /**
         * Compares stored syntax tree and new syntax tree (specified 
         * by the first argument), modified ranges found by tree-sitter
         * are appended to the provided vector (second argument)  
         */
        void diffTree(TSTree *new_tree, std::vector<offset_edit_range_t> &diff_ranges);

        /**
         * Performs edit above the TS tree structure 
         */
        void treeEdit(const TSInputEdit &ts_edit);

        /**
         * Performs edit of parsing context
         */
        TSInputEdit toTSEdit(const offset_t &offset, const size_t &ins_n, const size_t &del_n);
        TSInputEdit toTSEdit(const offset_t &offset, const size_t &ins_n, const size_t &del_n, const range_t &changed_range);

        point_t offsetToPointCached(const offset_t &offset, const offset_t &start_offset = 0, const point_t &start_pt = {0, 0});
        offset_t pointToOffsetCached(const point_t &point);

    private:
        std::string string_; ///< Internal string buffer
        std::string tmp_string_; ///< Internal string buffer for edited content
        TSTree *tree_ = nullptr; ///< Pointer to structure (with concrete syntax tree) maintained by TS

        std::vector<offset_t> row_to_offset_;
        cache_t offset_to_point_cache_;
        cache_t point_to_offset_cache_;
};

