/**
 * @file expression.h 
 * @file Contains declaration of Expression class and its specializations for 
 * each operator of YARA language
 * 
 * @author Vojtěch Dvořák
 */

#pragma once

#include "visitor.h"
#include "common.h"
#include "json/json.hpp"
#include "literal.h"
#include "rule_element.h"

#include <unordered_set>

#include "forward.h"
#include "expression_types.h"


/**
 * @brief Represents expression in condition and in definition of internal 
 * variables.
 * 
 * It's base class for concrete expression classes, that represents 
 * corresponding type of expression. Because concrete classes have operands 
 * also of Expression type, Expression objects make up tree-like structure. 
 */
class Expression : public RuleElement, public Printable {
    public:
        /**
         * @brief Data type of expression (for semantics checks) 
         */
        enum class Type {
            Undefined = 0,
            Object,
            Bool,
            Int,
            Float,
            String,
            Regexp,
        };

        Expression();
        ~Expression();

        /**
         * Provides conversion of Expression::Type value to readable string. 
         */
        static std::string_view typeToString(Expression::Type type);

        /**
         * Check whether two expressions are arithmetically compatible (if
         * implicit conversion exists of their data types). 
         */
        static bool isArithmeticCompatible(
            const ExpressionPtr &l,
            const ExpressionPtr &r
        ); 

        /**
         * Returns the data type of expression
         */
        virtual Expression::Type getType() const = 0;

        /**
         * Checks the validity of expression - data types of operands
         * are typically checked and their combinations (checks the semantic
         * of expression). It is called recursively for subexpressions of 
         * expression (e. g. operands) so semantic errors are propagated to the
         * top level expression.
         * @param msg error message, that describes an error
         * @return true if expression (and its subexpressions) is valid
         */
        virtual bool isValid(std::string_view &msg) const = 0;

        /**
         * Checks if operands are valid. This method is used to determine, 
         * whether if semantic error appears in current expression or in its
         * subexpressions (operands)
         */
        virtual bool areOperandsValid() const = 0;

        /**
         * Checks whther expression is complete - if it has all mandatory
         * operands. 
         */
        virtual bool isComplete() const = 0;

        /*
         * Type checking methods
         */

        bool isUndefined() const;
        bool isObject() const;
        bool isBool() const;
        bool isInt() const;
        bool isFloat() const;
        bool isString() const;
        bool isRegexp() const;

        /**
         * Accept method for visitor pattern 
         */
        virtual void accept(Visitor *v) = 0;
};


/**
 * @brief Class, that represents integer range (for example "(1..200)") 
 */
class Range : public Expression {
    public:
        Range(
            ExpressionPtr from,
            ExpressionPtr to
        ) : from_(std::move(from)), to_(std::move(to)) {};

    bool areOperandsValid() const override;

    /**
     * @brief Provides way how to determine if range as all necessary members 
     * defined 
     */
    bool isComplete() const override;

    /**
     * @brief Performs check if range is valid (both integer expressions must be 
     * valid as well as range itself) 
     */
    bool isValid(std::string_view &msg) const override;

    Expression::Type getType() const override;

    std::stringstream getTextFormatted() const override;

    void accept(Visitor *v) override {
        v->visitRange(this);
    }

    const ExpressionPtr &getFrom() const;

    const ExpressionPtr &getTo() const;

    private:
        ExpressionPtr from_ = nullptr;
        ExpressionPtr to_ = nullptr;
};


class Enum : public Expression {
    public:
        Enum(
            std::vector<ExpressionPtr> values
        ) : values_(values) {};


        bool areOperandsValid() const override;
        
        bool isComplete() const override;

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitEnum(this);
        }

        const std::vector<ExpressionPtr> &getValues() const;

    private:
        std::vector<ExpressionPtr> values_;
};


/**
 * @brief Base class for parentheses () in expressions
 */
class ParenthesesExpression : public Expression {
    public:
        ParenthesesExpression(
            ExpressionPtr inner_expr 
        ) : inner_expr_(std::move(inner_expr)) {};

        /**
         * Performs check of validity of inner expression 
         */
        bool isValid(std::string_view &msg) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        /**
         * Only propagates type of inner expression (or returns undefined
         * type if it does nto have any)
         */
        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitParentheses(this);
        }

        const ExpressionPtr &getInnerExpression() const;

    private:
        ExpressionPtr inner_expr_;
};

/**
 * @brief Class representing 'in' operator 
 */
class InExpression : public Expression {
    public:
        InExpression(
            ExpressionPtr el, 
            std::shared_ptr<Range> range
        ) : element_(el), range_(range) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        inline std::string opsign() const {
            return "in";
        }

        void accept(Visitor *v) override {
            v->visitIn(this);
        }

        const ExpressionPtr &getElementExpression() const;
        
        const std::shared_ptr<Range> &getRangeExpression() const;

    private:
        ExpressionPtr element_;
        std::shared_ptr<Range> range_;
};


/**
 * @brief Class, that represents binary operators 
 * 
 * Base class for binary operators such as +,-,\,* etc.
 */
class BinaryExpression : public Expression {
    public:
        BinaryExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : lop_(std::move(lop)), rop_(std::move(rop)) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        /**
         * Returns the string with printable sign or symbol of corresponding,
         * that is represented by specialized class. 
         */
        virtual inline std::string opsign() const = 0;

        /**
         * Returns stringstream with printable representation of expression.
         * If there is invalid operand, it is missing in output stringstream. 
         */
        std::stringstream getTextFormatted() const override;

        const ExpressionPtr &getLop() const;

        const ExpressionPtr &getRop() const;

    protected:
        ExpressionPtr lop_ = nullptr;
        ExpressionPtr rop_ = nullptr;
};


class MatchesExpression : public BinaryExpression {
    public:
        MatchesExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        inline std::string opsign() const override {
            return "matches";
        }

        void accept(Visitor *v) override {
            v->visitMatches(this);
        }
};


class AtExpression : public BinaryExpression {
    public:
        AtExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        inline std::string opsign() const override {
            return "at";
        }

        void accept(Visitor *v) override {
            v->visitAt(this);
        }
};


// BinaryArithmeticExpression

/**
 * @brief Groups all binary arithmetics operators for more generic type 
 * checking 
 */
class BinaryArithmeticExpression : public BinaryExpression {
    public:
        BinaryArithmeticExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        virtual bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class AddExpression : public BinaryArithmeticExpression {
    public:
        AddExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryArithmeticExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "+";
        }

        void accept(Visitor *v) override {
            v->visitAdd(this);
        }
};


class SubExpression : public BinaryArithmeticExpression {
    public:
        SubExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryArithmeticExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "-";
        }

        void accept(Visitor *v) override {
            v->visitSub(this);
        }
};


class MulExpression : public BinaryArithmeticExpression {
    public:
        MulExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryArithmeticExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "*";
        }

        void accept(Visitor *v) override {
            v->visitMul(this);
        }
};


class RemainderExpression : public BinaryArithmeticExpression {
    public:
        RemainderExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryArithmeticExpression(std::move(lop), std::move(rop)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        inline std::string opsign() const override {
            return "%";
        }

        void accept(Visitor *v) override {
            v->visitRemainder(this);
        }
};


class DivExpression : public BinaryArithmeticExpression {
    public:
        DivExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryArithmeticExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "\\";
        }

        void accept(Visitor *v) override {
            v->visitDiv(this);
        }
};

// BinaryBitwiseExpression

/**
 * @brief Groups all binary bitwise operators for more generic type checking 
 */
class BinaryBitwiseExpression : public BinaryExpression {
    public:
        BinaryBitwiseExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        virtual bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};

class LeftShiftExpression : public BinaryBitwiseExpression {
    public:
        LeftShiftExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBitwiseExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "<<";
        }

        void accept(Visitor *v) override {
            v->visitLeftShift(this);
        }
};


class RightShiftExpression : public BinaryBitwiseExpression {
    public:
        RightShiftExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBitwiseExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return ">>";
        }

        void accept(Visitor *v) override {
            v->visitRightShift(this);
        }
};


class BitwiseOrExpression : public BinaryBitwiseExpression {
    public:
        BitwiseOrExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBitwiseExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "|";
        }

        void accept(Visitor *v) override {
            v->visitBitwiseOr(this);
        }
};


class BitwiseAndExpression : public BinaryBitwiseExpression {
    public:
        BitwiseAndExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBitwiseExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "&";
        }

        void accept(Visitor *v) override {
            v->visitBitwiseAnd(this);
        }
};


class BitwiseXorExpression : public BinaryBitwiseExpression {
    public:
        BitwiseXorExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBitwiseExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "^";
        }

        void accept(Visitor *v) override {
            v->visitBitwiseXor(this);
        }
};

// BinaryRelationalExpression

/**
 * @brief Groups all binary bitwise operators for more generic type checking 
 */
class BinaryRelationalExpression : public BinaryExpression {
    public:
        BinaryRelationalExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        virtual bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class LtExpression : public BinaryRelationalExpression {
    public:
        LtExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "<";
        }

        void accept(Visitor *v) override {
            v->visitLt(this);
        }
};


class LteExpression : public BinaryRelationalExpression {
    public:
        LteExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "<=";
        }

        void accept(Visitor *v) override {
            v->visitLte(this);
        }
};


class GtExpression : public BinaryRelationalExpression {
    public:
        GtExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return ">";
        }

        void accept(Visitor *v) override {
            v->visitGt(this);
        }
};


class GteExpression : public BinaryRelationalExpression {
    public:
        GteExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return ">=";
        }

        void accept(Visitor *v) override {
            v->visitGte(this);
        }
};


class EqExpression : public BinaryRelationalExpression {
    public:
        EqExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "==";
        }

        void accept(Visitor *v) override {
            v->visitEq(this);
        }
};


class NeqExpression : public BinaryRelationalExpression {
    public:
        NeqExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "!=";
        }

        void accept(Visitor *v) override {
            v->visitNeq(this);
        }
};


// BinaryStrRelationalExpression

/**
 * @brief Groups all binary bitwise operators for more generic type checking 
 */
class BinaryStrRelationalExpression : public BinaryExpression {
    public:
        BinaryStrRelationalExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        virtual bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class ContainsExpression : public BinaryStrRelationalExpression {
    public:
        ContainsExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "contains";
        }

        void accept(Visitor *v) override {
            v->visitContains(this);
        }
};


class IContainsExpression : public BinaryStrRelationalExpression {
    public:
        IContainsExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "icontains";
        }

        void accept(Visitor *v) override {
            v->visitIContains(this);
        }
};


class StartsWithExpression : public BinaryStrRelationalExpression {
    public:
        StartsWithExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "startswith";
        }

        void accept(Visitor *v) override {
            v->visitStartsWith(this);
        }
};


class IStartsWithExpression : public BinaryStrRelationalExpression {
    public:
        IStartsWithExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "istartswith";
        }

        void accept(Visitor *v) override {
            v->visitIStartsWith(this);
        }
};


class EndsWithExpression : public BinaryStrRelationalExpression {
    public:
        EndsWithExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "endswith";
        }

        void accept(Visitor *v) override {
            v->visitEndsWith(this);
        }
};


class IEndsWithExpression : public BinaryStrRelationalExpression {
    public:
        IEndsWithExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "iendswith";
        }

        void accept(Visitor *v) override {
            v->visitIEndsWith(this);
        }
};


class IEqExpression : public BinaryStrRelationalExpression {
    public:
        IEqExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryStrRelationalExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "iequals";
        }

        void accept(Visitor *v) override {
            v->visitIEq(this);
        }
};


// BinaryBoolExpression

/**
 * @brief Groups all binary bitwise operators for more generic type checking 
 */
class BinaryBoolExpression : public BinaryExpression {
    public:
        BinaryBoolExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryExpression(std::move(lop), std::move(rop)) {};

        virtual bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class AndExpression : public BinaryBoolExpression {
    public:
        AndExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBoolExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "and";
        }

        void accept(Visitor *v) override {
            v->visitAnd(this);
        }
};


class OrExpression : public BinaryBoolExpression {
    public:
        OrExpression(
            ExpressionPtr lop, 
            ExpressionPtr rop
        ) : BinaryBoolExpression(std::move(lop), std::move(rop)) {};


        inline std::string opsign() const override {
            return "or";
        }

        void accept(Visitor *v) override {
            v->visitOr(this);
        }
};


// UnaryExpression


/**
 * @brief Base class for unary operators (such as unary minus, negation...).
 * 
 * Similar to Binary expression ( BinaryExpression ) 
 */
class UnaryExpression : public Expression {
    public:
        UnaryExpression(
            ExpressionPtr op
        ) : op_(std::move(op)) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        virtual inline std::string opsign() const = 0;

        std::stringstream getTextFormatted() const override;

        const ExpressionPtr &getOp() const;

    protected:
        ExpressionPtr op_ = nullptr;
};


class DefinedExpression : public UnaryExpression {
    public:
        DefinedExpression(
            ExpressionPtr op
        ) : UnaryExpression(std::move(op)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        inline std::string opsign() const override {
            return "defined ";
        }

        void accept(Visitor *v) override {
            v->visitDefined(this);
        }
};


// UnaryArithmeticExpression

/**
 * @brief Base class for all arithmetic operators (expressions), which have 
 * one operand 
 */
class UnaryArithmeticExpression : public UnaryExpression {
    public:
        UnaryArithmeticExpression(
            ExpressionPtr op
        ) : UnaryExpression(std::move(op)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class UnaryMinusExpression : public UnaryArithmeticExpression {
    public:
        UnaryMinusExpression(
            ExpressionPtr op
        ) : UnaryArithmeticExpression(std::move(op)) {};


        inline std::string opsign() const override {
            return "-";
        }

        void accept(Visitor *v) override {
            v->visitUnaryMinus(this);
        }
};


// UnaryBitwiseExpression

/**
 * @brief Base class for bitwise unary operators (expressions), that have
 * one operand. For example bitwise not (~) operator.  
 */
class UnaryBitwiseExpression : public UnaryExpression {
    public:
        UnaryBitwiseExpression(
            ExpressionPtr op
        ) : UnaryExpression(std::move(op)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class BitwiseNotExpression : public UnaryBitwiseExpression {
    public:
        BitwiseNotExpression(
            ExpressionPtr op
        ) : UnaryBitwiseExpression(std::move(op)) {};


        inline std::string opsign() const override {
            return "~";
        }

        void accept(Visitor *v) override {
            v->visitBitwiseNot(this);
        }
};



// UnaryBoolExpression

/**
 * @brief Base class for binary operators, that need boolean operands. For 
 * example and, or. 
 */
class UnaryBoolExpression : public UnaryExpression {
    public:
        UnaryBoolExpression(
            ExpressionPtr op
        ) : UnaryExpression(std::move(op)) {};

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;
};


class NotExpression : public UnaryBoolExpression {
    public:
        NotExpression(
            ExpressionPtr op
        ) : UnaryBoolExpression(std::move(op)) {};


        inline std::string opsign() const override {
            return "not";
        }

        void accept(Visitor *v) override {
            v->visitNot(this);
        }
};


// TerminalExpression

/**
 * @brief Base class for expression, that have not any subexpressions (operands)
 */
class TerminalExpression : public Expression {
    public:
        TerminalExpression() {};

        /**
         * Returns always true (terminal node can itself cannot be semantically
         * invalid)
         */
        bool isValid(std::string_view &msg) const override;

        /**
         * Returns always true (this type of expression does not have
         * any operands)
         */
        bool areOperandsValid() const override;
};


/**
 * @brief Represents Literal in expression. Possible terminal node of 
 * expression tree (it does not have any other Expression operands). 
 */
class LiteralExpression : public TerminalExpression {
    public:
        LiteralExpression(
            std::unique_ptr<Literal> &&l
        ) : l_(std::move(l)) {};

        bool isComplete() const override;

        std::stringstream getTextFormatted() const override;

        Expression::Type getType() const override;

        void accept(Visitor *v) override {
            v->visitLiteral(this);
        }

        const std::unique_ptr<Literal> &getLiteral() const;

    private:
        std::unique_ptr<Literal> l_ = nullptr;
};


class SizeExpression : public TerminalExpression {
    public:
        enum class Unit {
            MB,
            KB
        };

        SizeExpression(
            uint32_t value,
            SizeExpression::Unit unit
        ) : unit_(unit), value_(value) {};

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitSize(this);
        }

        const SizeExpression::Unit &getUnit() const;

        uint32_t getValue() const;

        /**
         * Converts enum value to readable string with unit
         * @return String view with readable representation of unit or with
         * empty string
         */
        static std::string_view unitToString(SizeExpression::Unit unit);

    private:
        SizeExpression::Unit unit_ = SizeExpression::Unit::KB;
        uint32_t value_;
};


/**
 * @brief Represents regular expression literal in expressions 
 */
class RegexpExpression : public TerminalExpression {
    public:
        RegexpExpression(
            const std::string &content
        ) : content_(content) {};

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitRegexp(this);
        }

        const std::string &getContent() const;

    private:
        std::string content_ = {};
};


/**
 * @brief Represents string identifier in expression ($<string_id>) 
 */
class StringExpression : public TerminalExpression {
    public:
        StringExpression(
            std::string_view string_id
        ) : string_id_(string_id) {};

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitString(this);
        }

        const std::string &getId() const;

    private:
        std::string string_id_ = {};
};


class StringWildcardExpression : public StringExpression {
    public:
        StringWildcardExpression(
            std::string_view string_id
        ) : StringExpression(string_id) {};

        void accept(Visitor *v) override {
            v->visitStringWildcard(this);
        }
};



class StringCountExpression : public TerminalExpression {
    public:
        StringCountExpression(
            std::string_view string_id
        ) : string_id_(string_id) {};

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitStringCount(this);
        }

        const std::string &getId() const;

    private:
        std::string string_id_ = {};
};


// StringPropertyExpression

class StringPropertyExpression : public Expression {
    public:
        StringPropertyExpression(
            std::string_view string_id,
            ExpressionPtr index = nullptr
        ) : string_id_(string_id), index_(index) {};

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        bool isComplete() const override;

        const std::string &getId() const;

        const ExpressionPtr &getIndex() const;

    protected:
        std::string string_id_ = {};

        ExpressionPtr index_ = nullptr;
};


class StringOffsetExpression : public StringPropertyExpression {
    public:
        StringOffsetExpression(
            std::string_view string_id,
            ExpressionPtr index = nullptr
        ) : StringPropertyExpression(string_id, index) {};

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitStringOffset(this);
        }
};


class StringMatchLengthExpression : public StringPropertyExpression {
    public:
        StringMatchLengthExpression(
            std::string_view string_id,
            ExpressionPtr index = nullptr
        ) : StringPropertyExpression(string_id, index) {};

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override; 

        void accept(Visitor *v) override {
            v->visitStringMatchLength(this);
        }
};


/**
 * @brief Base class for all symbols in expressions (structures, arrays, 
 * variables function calls...) 
 */
class Symbol : public Expression {
    public:

        /**
         * @brief Type of symbol 
         */
        enum class Type {
            Unknown,
            Function,
            Array,
            Dict,
            Struct,
            Value
        };

        Symbol() {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        void setType(Expression::Type type);
        Expression::Type getType() const override;

        /**
         * Sets symbol type ( @see Symbol::Type )
         */
        void setSymType(Symbol::Type type);

        /**
         * Gets symbol type 
         */
        Symbol::Type getSymType();

        /**
         * Sets the JSON context of the symbol 
         */
        void setContext(const json *attr_array);

        /**
         * Gets pointer to JSON context of the symbol 
         */
        const json *getContext() const;
        
        std::stringstream getTextFormatted() const override;

        /**
         * Sets the 'defined' flag  
         */
        void setDefined(bool defined);

        /**
         * Gets the value of defined flag. It is used to determine, whether
         * symbol is defined or not 
         */
        bool isDefined() const;

        bool isFunction() const;
        bool isArray() const;
        bool isDict() const;
        bool isStruct() const;
        bool isValue() const;

    protected:
        /**
         * Holds data type (Expression::Type) of symbol 
         */
        Expression::Type dtype_ = Expression::Type::Undefined;

        /**
         * Holds type of the symbol
         */
        Symbol::Type stype_ = Symbol::Type::Unknown;

        /**
         * Holds JSON context (related JSON object) of the symbol
         */
        const json *ctx_ = nullptr;

        /**
         * Defined flag - it should be set to true if symbol is defined 
         * otherwise it should be false 
         */
        bool defined_ = false;
};


class ArrayExpression : public Symbol {
    public:
        ArrayExpression(
            std::shared_ptr<Symbol> array, 
            ExpressionPtr key
        ) : array_(array), key_(key) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitArray(this);
        }

        const std::shared_ptr<Symbol> &getArray() const;

        const ExpressionPtr &getKey() const;

    private:
        std::shared_ptr<Symbol> array_;
        ExpressionPtr key_;
};



class StructExpression : public Symbol {
    public:
        StructExpression(
            std::shared_ptr<Symbol> structure, 
            std::shared_ptr<Symbol> member
        ) : structure_(structure), member_(member) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitStruct(this);
        }

        const std::shared_ptr<Symbol> &getStructure() const;

        const std::shared_ptr<Symbol> &getMember() const;

    private:
        std::shared_ptr<Symbol> structure_;
        std::shared_ptr<Symbol> member_;
};


class FunctionCallExpression : public Symbol {
    public:
        FunctionCallExpression(
            std::shared_ptr<Symbol> id, 
            std::vector<ExpressionPtr> args
        ) : id_(id), args_(std::move(args)) {};

        bool isComplete() const override;

        bool areOperandsValid() const override;

        bool isValid(std::string_view &msg) const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        /**
         * Performs check of arguments of the function
         * @return true if arguments have correct data types and there is correct 
         * number of arguments, otherwise false 
         */
        bool hasValidArgs() const;

        void accept(Visitor *v) override {
            v->visitFunctionCall(this);
        }

        const std::shared_ptr<Symbol> &getFunction() const;

        /**
         * Returns vector with arguments of function (pointer to expressions)
         * @note Returned vector of arguments can be invalid (use 
         * FunctionCallExpression::hasValidArgs to check them them)
         */
        const std::vector<ExpressionPtr> &getArgs() const;

    private:
        std::shared_ptr<Symbol> id_;
        std::vector<ExpressionPtr> args_;
};


class PlainSymbol : public Symbol {
    public:
        PlainSymbol(
            std::string_view id
        ) : id_(id) {};

        PlainSymbol() {};

        const std::string &get() const;

        std::stringstream getTextFormatted() const override;

        const std::string &getId() const;

        void accept(Visitor *v) override {
            v->visitPlainSymbol(this);
        }

    private:
        std::string id_ = {};
};


class VariableExpression : public PlainSymbol {
    public:
        VariableExpression(
            std::string_view id
        ) : PlainSymbol(id) {};

        void accept(Visitor *v) override {
            v->visitVariable(this);
        }
};


class RuleWildcardExpression : public PlainSymbol {
    public:
        RuleWildcardExpression(
            std::string_view string_id
        ) : PlainSymbol(string_id) {};


        void accept(Visitor *v) override {
            v->visitRuleWildCard(this);
        }
};


// QuantifierExpression

class QuantifierExpression : public Expression {
    public:
        QuantifierExpression() {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        Expression::Type getType() const override;
};


/**
 * @brief Represents 'none' keyword
 */
class NoneExpression : public QuantifierExpression {
    public:
        NoneExpression() {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitNone(this);
        }
};


/**
 * @brief Represents 'any' keyword
 */
class AnyExpression : public QuantifierExpression {
    public:
        AnyExpression() {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitAny(this);
        }
};


/**
 * @brief Represents 'all' keyword
 */
class AllExpression : public QuantifierExpression {
    public:
        AllExpression() {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitAll(this);
        }
};


class PercentQuantifierExpression : public QuantifierExpression {
    public:
        PercentQuantifierExpression(ExpressionPtr value) : value_(value) {};

        bool isValid(std::string_view &msg) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitPercentQuantifier(this);
        }

        const ExpressionPtr &getValue() const;

    private:
        ExpressionPtr value_;
};


// ThemExpression


/**
 * @brief Represents 'them' keyword
 */
class ThemExpression : public Expression {
    public:
        ThemExpression() {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        Expression::Type getType() const override;

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitThem(this);
        }
};


// SetExpression


/**
 * @brief Base class for all set expressions
 */
class SetExpression : public Expression {
    public:
        SetExpression(
            std::vector<ExpressionPtr> elements
        ) : elements_(elements) {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        Expression::Type getType() const override;

        /**
         * Returns vector with all elements (pointers to expressions) of set 
         */
        const std::vector<ExpressionPtr> &getElements() const;

    protected:
        std::vector<ExpressionPtr> elements_;
};


/**
 * @brief Represents set of strings
 */
class StringSetExpression : public SetExpression {
    public:
        StringSetExpression(
            std::vector<ExpressionPtr> elements
        ) : SetExpression(elements) {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitStringSet(this);
        }
};


/**
 * @brief Represents set of rules
 */
class RuleSetExpression : public SetExpression {
    public:
        RuleSetExpression(
            std::vector<ExpressionPtr> elements
        ) : SetExpression(elements) {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitRuleSet(this);
        }

    private:
        std::vector<ExpressionPtr> elements_;
};


/**
 * @brief Represents set of expressions
 * @note Avast specific object 
 */
class ExpressionSetExpression : public SetExpression {
    public:
        ExpressionSetExpression(
            std::vector<ExpressionPtr> elements
        ) : SetExpression(elements) {};

        std::stringstream getTextFormatted() const override;

        void accept(Visitor *v) override {
            v->visitExpressionSet(this);
        }
};


/**
 * @brief Represents all variants of 'of' operator in YARA 
 */
class OfExpression : public Expression {
    public:
        OfExpression(
            ExpressionPtr quantifier,
            ExpressionPtr set,
            std::shared_ptr<Range> range
        ) : quantifier_(quantifier), set_(set), range_(range) {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        std::stringstream getTextFormatted() const override;

        Expression::Type getType() const override;

        void accept(Visitor *v) override {
            v->visitOf(this);
        }

        /**
         * Returns quantifier of 'of' operator
         */
        const ExpressionPtr &getQuantifier() const;

        /**
         * Returns set of 'of' operator
         */
        const ExpressionPtr &getSet() const;

    private:
        /**
         * Quantifier of 'of' operator
         */
        ExpressionPtr quantifier_;
        
        /**
         * Set of 'of' operator
         */
        ExpressionPtr set_;

        std::shared_ptr<Range> range_ = nullptr;
};


/**
 * @brief Represents 'for' operator that have rule set as iterable set 
 * 
 * It is used for  applying the same condition to many strings
 * 
 * @note Syntax structure in YARA : 
 * for <quantifier> of <string_set> : ( <boolean_expression> )
 */
class ForExpression : public Expression {
    public:
        ForExpression(
            ExpressionPtr quantifier,
            ExpressionPtr set,
            ExpressionPtr inner_expr
        ) : quantifier_(quantifier), set_(set), expression_(inner_expr) {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        std::stringstream getTextFormatted() const override;

        Expression::Type getType() const override;

        void accept(Visitor *v) override {
            v->visitFor(this);
        }

        const ExpressionPtr &getQuantifier() const;

        const ExpressionPtr &getSet() const;

        const ExpressionPtr &getExpression() const;

    private:
        ExpressionPtr quantifier_; ///< Quantifier
        ExpressionPtr set_; ///< Set of strings
        ExpressionPtr expression_; ///< Condition that is applied for all elements in set 
};


/**
 * @brief Represents list of variables, that are defined in 'for' expression 
 * for usage inside internal expression of 'for' 
 */
class VarListExpression : public Expression {
    public:
        VarListExpression(
            std::vector<std::string> &&vars
        ) : vars_(vars) {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        std::stringstream getTextFormatted() const override;

        Expression::Type getType() const override;

        void accept(Visitor *v) override {
            v->visitVarList(this);
        }

        const std::vector<std::string> &getVariables() const;

    private:
        std::vector<std::string> vars_;
};


/**
 * @brief Represents 'for' operator with defined variables for inner expression
 * and with any iterable 
 * 
 * There is another type of 'for' expression - ForExpression , that have quite
 * different semantics
 * 
 * @note Syntax structure in YARA :
 * for <quantifier> <var_list> in <iterable> : ( <inner_expression> )  
 */
class ForIntExpression : public Expression {
    public:
        ForIntExpression(
            ExpressionPtr quantifier,
            std::shared_ptr<VarListExpression> list,
            ExpressionPtr iterable,
            ExpressionPtr expr
        ) : quantifier_(quantifier), list_(list), iterable_(iterable), expression_(expr) {};

        bool isValid(std::string_view &) const override;

        bool areOperandsValid() const override;

        bool isComplete() const override;

        std::stringstream getTextFormatted() const override;

        Expression::Type getType() const override;

        void accept(Visitor *v) override {
            v->visitForInt(this);
        }

        const ExpressionPtr &getQuantifier() const;

        const std::shared_ptr<VarListExpression>  &getVarList() const;

        const ExpressionPtr &getIterable() const;

        const ExpressionPtr &getExpression() const;

    private:
        ExpressionPtr quantifier_; ///< Quantifier of 'for'
        std::shared_ptr<VarListExpression> list_; ///< List of variables
        ExpressionPtr iterable_; ///< Iterable
        ExpressionPtr expression_; ///< Inner expression
};


/**
 * @brief Structure, that hold all information about in-expression defined 
 * variable
 * 
 * In case of YARA, it is possible to define variables only in 'for' 
 * expression 
 */
struct var_def_t {
    std::string id; ///< Identifier of variable
    Symbol::Type type = Symbol::Type::Unknown; ///< Type of variable 
    Expression::Type dtype = Expression::Type::Undefined; ///< Data type of variables
    const json *module_ctx = nullptr; ///< JSON context of variable
};
