/**
 * @file error_collector.h 
 * @brief Contains template class, that is responsible for 
 * collecting errors (and throwing error exceptions)
 * 
 * @author Vojtěch Dvořák
 */

#pragma once


#include "types.h"
#include "common.h"
#include "yaramod_config.h"
#include <vector>
#include <type_traits>
#include <map>



/**
 * @brief Class responsible for collecting all types of errors. It is basically
 * a map of vectors. Vectors in map are indexed by the offset where errors 
 * occurred in YaraFile. 
 */
template<class E>
class ErrorCollector {
    public:
        ErrorCollector();

        /**
         * Returns pointer with the all errors, that were collected 
         */
        SymTabOffset<E> &get();

        /**
         * Returns pointer with the all errors, that were collected 
         */
        const SymTabOffset<E> &getConst() const;

        /**
         * Clears the map with errors 
         */
        void clear();

        /**
         * Adds an error to the map 
         */
        void add(std::unique_ptr<E> new_error, YaramodConfig::ErrorMode mode);

        /**
         * Removes errors from the map. Errors, that should be deleted, are 
         * specified by given bool function
         * @return Number of removed errors 
         */
        template <typename Fn>
        uint32_t remove(const Fn &f);

        template <typename Fn>
        uint32_t remove(const Fn &f, offset_t start, offset_t end);

        /**
         * Removes errors backwards from starting at offset  
         */
        template<typename Fn>
        uint32_t revRemoveUntil(const Fn &f, offset_t rstart);

        /**
         * Updates errors. The key (error position/offset), can be also 
         * modified.
         * @return Number of updated errors
         */
        template <typename Fn>
        void update(const Fn &f);

        template <typename Fn>
        void update(const Fn &f, offset_t start, offset_t end);
        
        /**
         * Prints all errors to stderr (mainly for debugging purposes) 
         */
        void dump(const std::string &string) const;

    private:
        SymTabOffset<E> errors_;
};


template<class E>
ErrorCollector<E>::ErrorCollector() {
    static_assert(std::is_base_of<Error, E>::value, "type parameter E must be derived from Error class!");
};


template<class E>
void ErrorCollector<E>::add(std::unique_ptr<E> new_error, YaramodConfig::ErrorMode mode) {
    DEBUG_LOG("Error occurred! %s\n", new_error->getDescription().c_str());

    if(mode == YaramodConfig::ErrorMode::Strict) {
        throw new_error->exception();
    }
    else {
        errors_.insert(std::move(new_error));
    }
}


template<class E>
const SymTabOffset<E> &ErrorCollector<E>::getConst() const {
    return errors_;
}


template<class E>
SymTabOffset<E> &ErrorCollector<E>::get() {
    return errors_;
}


template<class E>
void ErrorCollector<E>::clear() {
    errors_.clear();
}


template<class E>
template<typename Fn>
uint32_t ErrorCollector<E>::remove(const Fn &f) {
    return errors_.remove(f);
}


template<class E>
template<typename Fn>
uint32_t ErrorCollector<E>::remove(const Fn &f, offset_t start, offset_t end) {
    return errors_.remove(f, start, end);
}



template<class V>
template<typename Fn>
uint32_t ErrorCollector<V>::revRemoveUntil(const Fn &f, offset_t rstart) {
    return errors_.revRemoveUntil(f, rstart);
}



template<class E>
template<typename Fn>
void ErrorCollector<E>::update(const Fn &f) {
    errors_.update(f);
}


template<class E>
template<typename Fn>
void ErrorCollector<E>::update(const Fn &f, offset_t start, offset_t end) {
    errors_.update(f, start, end);
}


template<class E>
void ErrorCollector<E>::dump(const std::string &string) const {
    for(auto &e: errors_.data()) {
        point_t point = offsetToPoint(
            string.c_str(), 
            e.second->getOffset());

        std::cerr << "ERROR at ("
        << point.row << ", " 
        << point.col << ") " 
        << e.second->getDescription().c_str() << std::endl;
    }
}
