/**
 * @file error.h 
 * @brief Contains declaration of error and exception hierarchies
 * 
 * @author Vojtěch Dvořák    
 */


#pragma once


#include "yara_file_element.h"
#include <string>

#include "forward.h"


/**
 * @brief Base class for exceptions
 */
class ErrorException : public std::exception {
    public:
        ErrorException(const std::string &msg);

        virtual const char *what() const noexcept override;

    private:
        const std::string msg_;
};


/**
 * @brief Exception for unexpected errors, that are probably not caused by user 
 * (nonrecoverable errors of internal parser etc.)
 */
class InternalErrorException : public ErrorException {
    public:
        InternalErrorException(std::string msg = {}) : ErrorException(msg) {};
};


/**
 * @brief Exception for other errors, that can occur during parsing 
 */
class YaramodErrorException : public ErrorException {
    public:
        YaramodErrorException(std::string msg = {}) : ErrorException(msg) {};
};


/**
 * @brief Base Exception class for errors located in input YARA code 
 */
class YaraErrorException : public ErrorException {
    public:
        YaraErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ErrorException(msg), offset_(offset), len_(len) {};

        size_t getLen() const noexcept;

        offset_t getOffset() const noexcept;

        range_t getRange(std::string_view str) const noexcept;

    private:
        offset_t offset_;
        size_t len_;
};


/**
 * @brief Exception representing syntax error in parsed source code 
 */
class SyntaxErrorException : public YaraErrorException {
    public:
        SyntaxErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : YaraErrorException(offset, len, msg) {};
};


/**
 * @brief Exception representing syntax error in parsed source code 
 */
class UnexpectedTokenErrorException : public SyntaxErrorException {
    public:
        UnexpectedTokenErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SyntaxErrorException(offset, len, msg) {};
};


/**
 * @brief Exception representing missing token in parsed source code
 */
class MissingTokenErrorException : public SyntaxErrorException {
    public:
        MissingTokenErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SyntaxErrorException(offset, len, msg) {};
};


/**
 * @brief Exception representing semantic errors in parsed source code 
 */
class SemanticErrorException : public YaraErrorException {
    public:
        SemanticErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : YaraErrorException(offset, len, msg) {};
};


/**
 * @brief Base class for exception, that represents error, that occurred 
 * in expression   
 */
class ExpressionErrorException : public SemanticErrorException {
    public:
        ExpressionErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad type of index expression in array access expression  
 */
class IndexErrorException : public ExpressionErrorException {
    public:
        IndexErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad rule reference (e. g. rule does not exists) 
 */
class RuleReferenceErrorException : public ExpressionErrorException {
    public:
        RuleReferenceErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad string reference (e. g. string does not exists) 
 */
class StringReferenceErrorException : public ExpressionErrorException {
    public:
        StringReferenceErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors, that occur in range expression (e. g. higher 
 * left bound than right bound)
 */
class RangeErrorException : public ExpressionErrorException {
    public:
        RangeErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors, that occur in function call (e. g. bad types 
 * of arguments) 
 */
class FunctionCallErrorException : public ExpressionErrorException {
    public:
        FunctionCallErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad symbol reference (e. g. symbol does not exists) 
 */
class SymbolReferenceErrorException : public ExpressionErrorException {
    public:
        SymbolReferenceErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad usage of symbol (e. g. symbol is used as structure, 
 * but it is value) 
 */
class SymbolErrorException : public ExpressionErrorException {
    public:
        SymbolErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : ExpressionErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for bad file include (e. g. file is included twice)
 */
class IncludeErrorException : public SemanticErrorException {
    public:
        IncludeErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors, that occurred in string definition (e. g. 
 * collision of string identifier)
 */
class StringErrorException : public SemanticErrorException {
    public:
        StringErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception error of string modifier (e. g, invalid combination of 
 * string type and modifier)
 */
class StringModifierErrorException : public StringErrorException {
    public:
        StringModifierErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : StringErrorException(offset, len, msg) {};
};


/**
 * @brief Exception of semantic error of regular expression (e. g. bad repeat 
 * interval) 
 */
class RegexpErrorException : public StringErrorException {
    public:
        RegexpErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : StringErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for semantic errors of hexadecimal string (e. g. bad 
 * jump interval)
 */
class HexStrErrorException : public StringErrorException {
    public:
        HexStrErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : StringErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for overflows
 */
class OverflowErrorException : public SemanticErrorException {
    public:
        OverflowErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors, that occurred in module import  
 */
class ModuleErrorException : public SemanticErrorException {
    public:
        ModuleErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors occurred in rule definition (e. g. collision 
 * of rule identifier)
 */
class RuleErrorException : public SemanticErrorException {
    public:
        RuleErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors of rule modifier (e. g. duplicated rule modifier)
 */
class RuleModifierErrorException : public RuleErrorException {
    public:
        RuleModifierErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : RuleErrorException(offset, len, msg) {};
};


/**
 * @brief Exception for errors in variable definition 
 */
class VariableErrorException : public SemanticErrorException {
    public:
        VariableErrorException(
            offset_t offset, 
            size_t len, 
            std::string msg = {}
        ) : SemanticErrorException(offset, len, msg) {};
};



/* --------------------------------------------------------------------------*
 *                                   Errors                                  *
 * --------------------------------------------------------------------------*/




/**
 * @brief Class, that represents errors in analyzed source code. The main 
 * difference between ErrorExceptions and Errors is in usage - Errors are 
 * stored in some vector/list or in something and ErrorException are thrown
 */
class Error : public TopLevelYaraFileElement {
    public:
        Error(const std::string &desc, const offset_t &pos, const size_t &len);

        /**
         * Returns textual description of error - meaning of textual 
         * description depends on concrete type of error 
         */
        const std::string &getDescription() const;

        /**
         * Provides conversion to throwable exception 
         */
        ErrorException exception() const;

    protected:
        std::string desc_; ///< Textual information about error

        YaraFileElement *parent_element_ = nullptr; ///< Parent element of an error
};



/**
 * @brief Base class for syntax errors 
 */
class SyntaxError : public Error {
    public:
        SyntaxError(const std::string &description, const offset_t &offset, const size_t &len) : Error(description, offset, len) {};
        ~SyntaxError();

        SyntaxErrorException exception() {
            return SyntaxErrorException(len_, offset_, desc_);
        };    
};



/**
 * @brief Class for syntax error caused by unexpected token 
 */
class UnexpectedTokenError : public SyntaxError {
    public:
        UnexpectedTokenError(const std::string &token, const offset_t &offset, const size_t &len) : SyntaxError(token, offset, len) {};

        MissingTokenErrorException exception() {
            return MissingTokenErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Class for syntax error caused missing token
 * @note These errors are identified by the tree-sitter parser when it is clear
 * what is missing 
 */
class MissingTokenError : public SyntaxError {
    public:
        MissingTokenError(const std::string &missing_str, const offset_t &offset, const size_t &len) : SyntaxError(missing_str, offset, len) {};

        MissingTokenErrorException exception() {
            return MissingTokenErrorException(len_, offset_, desc_);
        };
};



/**
 * @brief Base class for semantic errors 
 */
class SemanticError : public Error {
    public:
        SemanticError(const std::string &desc, const offset_t &offset, const size_t &len) : Error(desc, offset, len) {};
        ~SemanticError();

        SemanticErrorException exception() {
            return SemanticErrorException(len_, offset_, desc_);
        };

        /**
         * Returns global flag - this flag determines whether SemanticError was 
         * caused some token of YaraFile and so it cannot be fixed just by 
         * removing/updating this token (e.g. duplicated rule in included file ->
         * we don't know which rule will user delete). Token with this token set 
         * to true are completely removed while reparsing and builded again (if 
         * there were not fixed). 
         */

        bool isGlobal() const;

        /**
         * Sets the state of global flag, @see isGlobal 
         */
        SemanticError *setGlobal(bool new_state);

    private:
        bool is_global_ = false; ///< Flag, that signalizing, that error must be deleted before reparsing
};



/**
 * @brief Base class for errors of expression (e. g. type mismatch)
 */
class ExpressionError : public SemanticError {
    public:
        ExpressionError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        ExpressionErrorException exception() {
            return ExpressionErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error of index in array access expression
 */
class IndexError : public ExpressionError {
    public:
        IndexError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        IndexErrorException exception() {
            return IndexErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad rule reference (referenced rule does not exists) 
 */
class RuleReferenceError : public ExpressionError {
    public:
        RuleReferenceError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        RuleReferenceErrorException exception() {
            return RuleReferenceErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad string reference (referenced string does not exists) 
 */
class StringReferenceError : public ExpressionError {
    public:
        StringReferenceError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        StringReferenceErrorException exception() {
            return StringReferenceErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad range (e. g. left bound is greater than right bound) 
 */
class RangeError : public ExpressionError {
    public:
        RangeError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        RangeErrorException exception() {
            return RangeErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad function call (e. g. bad types or bad count of arguments)
 */
class FunctionCallError : public ExpressionError {
    public:
        FunctionCallError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        FunctionCallErrorException exception() {
            return FunctionCallErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad symbol reference (e. g. reference to undefined symbol) 
 */
class SymbolReferenceError : public ExpressionError {
    public:
        SymbolReferenceError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        SymbolReferenceErrorException exception() {
            return SymbolReferenceErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad usage of symbol 
 */
class SymbolError : public ExpressionError {
    public:
        SymbolError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : ExpressionError(desc, offset, len) {};

        SymbolErrorException exception() {
            return SymbolErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Bad file include (e. g. included file does not exists) 
 */
class IncludeError : public SemanticError {
    public:
        IncludeError(
            const std::string &desc,
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        IncludeErrorException exception() {
            return IncludeErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error in string definition (e. g. collision of string identifiers)
 */
class StringError : public SemanticError {
    public:
        StringError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        StringErrorException exception() {
            return StringErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error of string modifier (e. g. invalid type of string modifier 
 * for specific string) 
 */
class StringModifierError : public StringError {
    public:
        StringModifierError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : StringError(desc, offset, len) {};

        StringModifierErrorException exception() {
            return StringModifierErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Regular expression error (e. g. bad repeat interval) 
 */
class RegexpError : public StringError {
    public:
        RegexpError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : StringError(desc, offset, len) {};

        RegexpErrorException exception() {
            return RegexpErrorException(len_, offset_, desc_);
        };
};



/**
 * @brief Error in hexadecimal string (e. g. bad jump boundaries) 
 */
class HexStrError : public StringError {
    public:
        HexStrError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : StringError(desc, offset, len) {};

        HexStrErrorException exception() {
            return HexStrErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Overflow error (e. g. integer overflow) 
 */
class OverflowError : public SemanticError {
    public:
        OverflowError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        OverflowErrorException exception() {
            return OverflowErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error that occurred in module import
 */
class ModuleError : public SemanticError {
    public:
        ModuleError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len) : SemanticError(desc, offset, len) {};

        ModuleErrorException exception() {
            return ModuleErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error, that occurred in error definition (e. g. collision of 
 * rule identifiers)
 */
class RuleError : public SemanticError {
    public:
        RuleError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        RuleErrorException exception() {
            return RuleErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error of rule modifier (e. g. duplicated rule modifier) 
 */
class RuleModifierError : public RuleError {
    public:
        RuleModifierError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : RuleError(desc, offset, len) {};

        RuleModifierErrorException exception() {
            return RuleModifierErrorException(len_, offset_, desc_);
        };
};


/**
 * @brief Error of variable definition 
 */
class VariableError : public SemanticError {
    public:
        VariableError(
            const std::string &desc, 
            const offset_t &offset, 
            const size_t &len
        ) : SemanticError(desc, offset, len) {};

        VariableErrorException exception() {
            return VariableErrorException(len_, offset_, desc_);
        };
};

