/**
 * @file file_context.cpp
 * @brief Contains implementation of FileContext methods
 * 
 * @author Vojtěch Dvořák 
 */


#include "headers/file_context.h"


FileContext::FileContext() {
}


FileContext::~FileContext() {
    ts_tree_delete(tree_);
}


const std::string &FileContext::getString() const {
    return string_;
}


const std::string &FileContext::getTempString() const {
    return tmp_string_;
}



void FileContext::setString(const std::string &string) {
    string_ = string;
    tmp_string_ = string;

    offset_to_point_cache_.is_valid = false; ///< Invalidate cache because string is changed
}


void FileContext::stringEdit(const offset_t &offset, const size_t &size, const std::string &new_text) {
    tmp_string_.replace(offset, size, new_text); ///< Update the part of tmp internal buffer with string
    DEBUG_LOG("Editing string (offset: %d, size: %ld, new_text: %s)\n", offset, size, new_text.c_str());
    //DEBUG_LOG("New tmp_string_:\n%s\n", tmp_string_.c_str());
}


void FileContext::stringUpdate() {
    string_ = tmp_string_;

    offset_to_point_cache_.is_valid = false; ///< Invalidate cache because string is changed
}


void FileContext::undoEdits() {
    tmp_string_ = string_;
}


TSTree *FileContext::getTree() const {
    return tree_;
}


void FileContext::setTree(TSTree *tree) {
    tree_ = tree;
} 


void FileContext::clearTree() {
    if(tree_) {
        ts_tree_delete(tree_);
    }

    tree_ = nullptr;
}


void FileContext::diffTree(TSTree *new_tree, std::vector<offset_edit_range_t> &diff_ranges) {
    uint32_t len;
    TSRange *ts_ranges = ts_tree_get_changed_ranges( ///< Get modified ranges report by the tree sitter
        getTree(), 
        new_tree, 
        &len
    );

    for(uint32_t i = 0; i < len; i++) { ///< Convert TSRange to offset_edit_range_t struct
        diff_ranges.push_back(offset_edit_range_t(
            ts_ranges[i].start_byte, 
            ts_ranges[i].end_byte,
            offset_edit_range_t::EditType::MOD
        ));
    }

    free(ts_ranges); ///< Free the original TS structure
} 


void FileContext::treeEdit(const TSInputEdit &ts_edit) {
    ts_tree_edit(getTree(), &ts_edit);
}


TSInputEdit FileContext::toTSEdit(const offset_t &offset, const size_t &ins_n, const size_t &del_n) {
    offset_t old_end_offset = offset + del_n;
    offset_t new_end_offset = offset + ins_n;
    
    point_t new_end_pt = offsetToPointCached(new_end_offset);
    point_t old_end_pt = offsetToPointCached(old_end_offset);
    point_t start_pt = offsetToPointCached(offset);

    TSInputEdit ts_edit = { ///< Convert edit params to TSInputEdit for tree-sitter
        offset,
        old_end_offset,
        new_end_offset,
        { start_pt.row, start_pt.col },
        { old_end_pt.row, old_end_pt.col },
        { new_end_pt.row, new_end_pt.col },
    };

    return ts_edit;
}


TSInputEdit FileContext::toTSEdit(const offset_t &offset, const size_t &ins_n, const size_t &del_n, const range_t &changed_range) {
    offset_t old_end_offset = offset + del_n;
    offset_t new_end_offset = offset + ins_n;
    
    point_t old_end_pt = changed_range.end;
    point_t start_pt = changed_range.start;
    point_t new_end_pt = offsetToPoint(
        tmp_string_.c_str(), 
        new_end_offset,
        start_pt,
        offset
    );

    TSInputEdit ts_edit = { ///< Convert edit params to TSInputEdit for tree-sitter
        offset,
        old_end_offset,
        new_end_offset,
        { start_pt.row, start_pt.col },
        { old_end_pt.row, old_end_pt.col },
        { new_end_pt.row, new_end_pt.col },
    };
    
    return ts_edit;
}


point_t FileContext::offsetToPointCached(const offset_t &offset, const offset_t &start_offset, const point_t &start_pt) {

    point_t result;

    if(offset_to_point_cache_.is_valid && offset_to_point_cache_.entry.offset < offset) {
        //DEBUG_LOG("Cache HIT\n");
        result = offsetToPoint(
            string_.c_str(), 
            offset, 
            offset_to_point_cache_.entry.point, 
            offset_to_point_cache_.entry.offset
        );
    }
    else if(offset_to_point_cache_.is_valid && offset_to_point_cache_.entry.offset == offset) {
        //DEBUG_LOG("Cache HIT\n");
        return offset_to_point_cache_.entry.point;
    }
    else {
        //DEBUG_LOG("Cache MISS\n");
        result = offsetToPoint(string_.c_str(), offset, start_pt, start_offset);
    }

    offset_to_point_cache_.entry.offset = offset;
    offset_to_point_cache_.entry.point = result;
    offset_to_point_cache_.is_valid = true;

    return result;
}


offset_t FileContext::pointToOffsetCached(const point_t &point) {
    offset_t result;

    if(point_to_offset_cache_.is_valid && point_to_offset_cache_.entry.point < point) {
        //DEBUG_LOG("Cache HIT\n");
        result = pointToOffset(
            string_.c_str(), 
            point, 
            point_to_offset_cache_.entry.point, 
            point_to_offset_cache_.entry.offset
        );
    }
    else if(point_to_offset_cache_.is_valid && point_to_offset_cache_.entry.point == point) {
        //DEBUG_LOG("Cache HIT\n");
        return point_to_offset_cache_.entry.offset;
    }
    else {
        //DEBUG_LOG("Cache MISS\n");
        result = pointToOffset(string_.c_str(), point);
    }

    point_to_offset_cache_.entry.offset = result;
    point_to_offset_cache_.entry.point = point;
    point_to_offset_cache_.is_valid = true;

    return result;
}

