/**
 * @file reparsing.cpp 
 * IT IS NOT PART OF LIBRARY, it is just simple demo for
 * convenient continuous testing and benchmarking of the new features. 
 * 
 * The program parses ruleset_path, prints result of its analysis, incrementaly 
 * reparses it (due to edit parameters, that are provided as the second 
 * argument) and prints analysis result again. 
 * 
 * 
 * 
 * USAGE:
 *  ./reparsing <ruleset_path>  <modified_file_path>  <edit_offset>:<ins_n>:<del_n>[:<edited_file>]
 * 
 *                                 OR
 * 
 *  ./reparsing <ruleset_path>  <modified_file_path>  <edited_range>[:<edited_file>]  <new_text>
 * 
 *  Where:
 *  <edited_range> = <start_row>,<start_col>:<end_row>,<end_col>
 * 
 * 
 * @note If the edit is specified by the second way (by edited range), 
 * the edited range is inclusive from the left and exclusive from the start (it 
 * should be similar to LSP's conventions). If text is only deleted (without
 * any substitution), use empty string ("") as the third argument. Modified 
 * file is necessary only, if edit was not performed in the entry file of the
 * ruleset.
 * 
 * @author Vojtěch Dvořák    
 */


#define MIN_ARG_NUM 4 ///< The minimal number of program arguments


#define EDIT_PARAMS_NUM 3 ///< Number of parameters characterizing edit  


#include <charconv>
#include <iostream>
#include "yaramod.h"

#include <sys/time.h>
#include <sys/resource.h>


// Computes elapsed time by subtraction of two timeval structures
#define ELAPSED_TIME(end_timeval, start_timeval)\
(end_timeval.tv_sec + end_timeval.tv_usec*1e-6) - (start_timeval.tv_sec + start_timeval.tv_usec*1e-6)


/**
 * Print lists with all symbols of given yara_src
 */
void dumpYaraSource(const YaraSourcePtr &yara_src) {
    std::cout << "- Rules:" << std::endl;
    for(auto r: yara_src->getAllRules()) {
        std::cout << r->getId() << " (at " << r->getOffset() << ") " << std::endl;

        std::cout << "-- Metas:" << std::endl;
        for(auto m: r->getMetas()) {
            std::cout << m->getId() << " (value: " << m->getValue()->getTextFormatted().str() << ") " << std::endl;
        }

        std::cout << "-- Strings:" << std::endl;
        for(auto s: r->getStrings()) {
            std::cout << "$" << s->getId(true) << " (at: " << s->getOffset() << ", value: " << s->getContent() << " ) " << std::endl;
        }

        std::cout << std::endl;
    }
    std::cout << "- Syntax errors:" << std::endl;
    for(auto s: yara_src->getAllSyntaxErrors()) {
        std::cout << s->getDescription() << " (at " << s->getOffset() << ") " << std::endl;
    }
    std::cout << "- Semantic errors:" << std::endl;
    for(auto s: yara_src->getAllSemanticErrors()) {
        std::cout << s->getDescription() << " (at " << s->getOffset() << ") " << std::endl;
    }

    std::cout << std::endl;
}


/**
 * Provides conversion of program arg to vector with uints 
 */
void getEditParams(std::string_view param_str, std::vector<uint32_t> &result, std::string &edited_file) {
    result.clear();

    uint32_t tmp;
    for(int i = 0; i < EDIT_PARAMS_NUM; i++) {
        std::string_view num_str = param_str.substr(0, param_str.find(':'));
        auto [ptr, ec] = std::from_chars(num_str.data(), num_str.data() + num_str.size(), tmp);
        if(ec != std::errc()) {
            throw "Bad edit parameters!";
        }

        result.push_back(tmp);
        if(param_str.find(':') == std::string::npos) {
            param_str = {};
        }
        else {
            param_str = param_str.substr(param_str.find(':') + 1);
        }
    }
    if(!param_str.empty()) {
        edited_file = std::string(param_str);
    }
}


/**
 * Helper function, that converts string view in format <row>,<col> to point_t struct 
 */
point_t getPointFromStr(std::string_view sv) {
    point_t result;

    std::string_view row_str = sv.substr(0, sv.find(','));
    auto [st_ptr, std_ec] = std::from_chars(
        row_str.data(), 
        row_str.data() + row_str.size(), 
        result.row
    );
    if(std_ec != std::errc()) {
        throw "Bad edit point!";
    }


    sv = sv.substr(sv.find(',') + 1); // Shift sv to col number

    std::string_view col_str = sv.substr(0, sv.find(','));
    auto [end_ptr, end_ec] = std::from_chars(
        col_str.data(), 
        col_str.data() + col_str.size(), 
        result.col
    );
    if(end_ec != std::errc()) {
        throw "Bad edit point!";
    }

    return result;
}


/**
 * Converts string with edited range (in format <start_row>,<start_col>:<end_row>,<end_col>[:<file>]) 
 */
range_t getEditedRange(std::string_view param_str, std::string &edited_file) {
    range_t result;

    std::string_view start_pt_str = param_str.substr(0, param_str.find(':'));
    result.start = getPointFromStr(start_pt_str);

    param_str = param_str.substr(param_str.find(':') + 1); // Shift param str to the next pair of integers
    
    std::string_view end_pt_str = param_str.substr(0, param_str.find(':'));
    result.end = getPointFromStr(end_pt_str);

    if(param_str.find(':') == std::string::npos) {
        param_str = {};
    }
    else {
        param_str = param_str.substr(param_str.find(':') + 1);
    }
    
    if(!param_str.empty()) {
        edited_file = std::string(param_str);
    }

    return result;
}


/**
 * Prints statistics to stdout 
 */
void printStatistics(const struct rusage &start, const struct rusage &end) {
    std::cout << "utime " << ELAPSED_TIME(end.ru_utime, start.ru_utime) << std::endl;
    std::cout << "stime " << ELAPSED_TIME(end.ru_stime, start.ru_stime) << std::endl;
    std::cout << "maxrss "  << end.ru_maxrss << std::endl << std::endl;
}


int main(int argc, char **argv) {
    if(argc < MIN_ARG_NUM) {
        std::cerr << "Missing argument!" << std::endl;
        return 0;
    }

    std::unique_ptr y = std::make_unique<Yaramod>();

    struct rusage start, end;
    std::vector<uint32_t> edit_params;
    range_t edited_range;
    std::string edited_file = {};

    // Conversion of arguments to integer values

    if(argc == MIN_ARG_NUM) {
        getEditParams(argv[3], edit_params, edited_file);
    }
    else {
        edited_range = getEditedRange(argv[3], edited_file);
    }


    // Parsing

    // Get the information about used resources before the parsing begins
    if(getrusage(RUSAGE_SELF, &start) != 0) {
        return 2;
    }

    YaraSourcePtr yara_src = y->parseFile(argv[1]);

    if(getrusage(RUSAGE_SELF, &end) != 0) {
        return 2;
    }

    // Print results
    std::cout << "*** File " << argv[1] << " parsing ***" << std::endl;
    dumpYaraSource(yara_src);

    std::cout << "- Parsing statistics: " << std::endl;
    printStatistics(start, end);

    // Reparsing 

    if(getrusage(RUSAGE_SELF, &start) != 0) {
        return 2;
    }

    if(argc > MIN_ARG_NUM) {
        if(edited_file.empty()) {
            yara_src->edit(edited_range, argv[4]);
        }
        else {
            yara_src->edit(edited_range, argv[4], edited_file);
        }
    }
    else {
        if(edited_file.empty()) {
            yara_src->edit(edit_params.at(0), edit_params.at(1), edit_params.at(2));
        }
        else {
            yara_src->edit(edit_params.at(0), edit_params.at(1), edit_params.at(2), edited_file);
        }
    }
    
    y->parseFile(argv[2], yara_src);

    if(getrusage(RUSAGE_SELF, &end) != 0) {
        return 2;
    }

     // Print results
    std::cout << "*** File " << argv[1] << " reparsing ***" << std::endl;
    dumpYaraSource(yara_src);

    std::cout << "- Reparsing statistics: " << std::endl;
    printStatistics(start, end);

    return 0;
}
