/**
 * @file parsing.cpp 
 * IT IS NOT PART OF LIBRARY, it is just simple demo for
 * convenient continuous testing and benchmarking of the new features. Parses
 * file located at path given as the first argument of this program and prints
 * static analysis results.
 * 
 * @author Vojtěch Dvořák    
 */

#include <iostream>
#include "yaramod.h"

#include <sys/time.h>
#include <sys/resource.h>

// Computes elapsed time by substraction of two timeval structures
#define ELAPSED_TIME(end_timeval, start_timeval)\
(end_timeval.tv_sec + end_timeval.tv_usec*1e-6) - (start_timeval.tv_sec + start_timeval.tv_usec*1e-6)


/**
 * Print lists with all symbols of given yara_file 
 */
void dumpYaraFile(const YaraFilePtr yara_file) {
    std::cout << "Rule cnt: " << yara_file->getLocalRules().size() << std::endl;
    std::cout << "- Rules:" << std::endl;
    for(auto r: yara_file->getLocalRules()) {
        range_t rule_range = r->getRange();
        std::cout << r->getId() << " (at " << r->getOffset();
        std::cout << ", [" << rule_range.start.row << ", " << rule_range.start.col << "]";
        std::cout << ", (" << rule_range.end.row << ", " << rule_range.end.col << "))" << std::endl;

        std::cout << "-- Metas:" << std::endl;
        for(auto m: r->getMetas()) {
            std::cout << m->getId() << " (value: " << m->getValue()->getTextFormatted().str() << ") " << std::endl;
        }

        std::cout << "-- Strings:" << std::endl;
        for(auto s: r->getStrings()) {
            std::cout << "$" << s->getId(true) << " (at: " << s->getOffset() << ", value: " << s->getContent() << " ) " << std::endl;
        }

        std::cout << "-- Variables:" << std::endl; // Avast feature
        for(auto v: r->getVars()) {
            std::cout << v->getId() << " (value:  " << v->getValue()->getTextFormatted().str() << ") " << std::endl;
        }

        auto condition = r->getCondition();
        std::cout << "-- Condition (at: " << condition->getOffset() << ", type: " << Expression::typeToString(condition->getType()) << "): " << std::endl; 
        std::cout << condition->getTextFormatted().str() << std::endl << std::endl;
    }
    std::cout << "- Imports:" << std::endl;
    for(auto i: yara_file->getLocalImports()) {
        std::cout << i->getModuleName() << " (at " << i->getOffset() << ") " << std::endl;
    }
    std::cout << "- Includes:" << std::endl;
    for(auto i: yara_file->getLocalFileIncludes()) {
        std::cout << i->getPath() << " (at " << i->getOffset() << ") " << std::endl;
    }
    std::cout << "- Comments:" << std::endl;
    for(auto c: yara_file->getComments()) {
        std::cout << c->getText() << " (at " << c->getOffset() << ") " << std::endl;
    }
    std::cout << "- Syntax errors:" << std::endl;
    for(auto s: yara_file->getSyntaxErrors()) {
        std::cout << s->getDescription() << " (at " << s->getOffset() << ") " << std::endl;
    }
    std::cout << "- Semantic errors:" << std::endl;
    for(auto s: yara_file->getSemanticErrors()) {
        std::cout << s->getDescription() << " (at " << s->getOffset() << ") " << std::endl;
    }

    std::cout << std::endl;
}


int main(int argc, char **argv) {

    if(argc < 2) {
        std::cerr << "No file was specified!" << std::endl;
        return 2;
    }

    // Parsing

    std::unique_ptr y = std::make_unique<Yaramod>();
    
    struct rusage start, end;

    // Get the information about used resource before the parsing begins
    if(getrusage(RUSAGE_SELF, &start) != 0) {
        return 2;
    }

    YaraSourcePtr y_src;
    try {
        y_src = y->parseFile(argv[1]);
    }
    catch(ErrorException &e) {
        std::cerr << "Exception occured: " << std::endl;
        std::cerr << e.what() << std::endl;
        return 1;
    }

    // Get the information about used resources before the parsing begins
    if(getrusage(RUSAGE_SELF, &end) != 0) {
        return 2;
    }

    std::cout << "*** File " << argv[1] << " ***" << std::endl;
    dumpYaraFile(y_src->getEntryFile());

    std::cout << "- Parsing statistics: " << std::endl;
    std::cout << "utime " << ELAPSED_TIME(end.ru_utime, start.ru_utime) << std::endl;
    std::cout << "stime " << ELAPSED_TIME(end.ru_stime, start.ru_stime) << std::endl;
    std::cout << "maxrss "  << end.ru_maxrss << std::endl << std::endl;

    std::cout << "- Serialization of the yara source: " << std::endl;
    std::cout << y_src->getTextFormatted().rdbuf();

    return 0;
}
