/**
 * @file observing_visitor.cpp 
 * Simple demo focused on visitor interface of 
 * expressions. Program prints graphical representation of conditions of all
 * rules in given file. Conditions must have only boolean operators and operands 
 * (and, or, not, true, false), but it can be easily extended by adding more
 * overrides to GraphVisitor class.
 * 
 * USAGE:
 * ./o_visitor <file_path>
 * 
 * EXAMPLE:
 * file.yar
 * ```
 * rule boolean_rule {
 *      condition:
 *          true and true or true and not true or false and true
 * }
 * ```
 * 
 * Output:
 * ```
 * $ ./o_visitor file.yar
 * boolean_rule
 * OR
 * ├───OR
 * │   ├───AND
 * │   │   ├───true
 * │   │   └───true
 * │   └───AND
 * │       ├───true
 * │       └───NOT
 * │           └───true
 * └───AND
 *     ├───false
 *     └───true
 *
 *
 * ```
 * 
 * @author Vojtěch Dvořák
 */

#include <iostream>
#include <string>
#include "yaramod.h"
#include "visitor/observing_visitor.h"

/**
 * Concrete visitor based on ObservingVisitor class 
 */
class GraphVisitor : public ObservingVisitor {
    public:
        GraphVisitor() {
        };

        ~GraphVisitor() {
        };

        void drawBinaryOperator(BinaryExpression *e) {
         
            std::cout << indent_string_;
            std::cout << "├───";

            std::string orig_indent = indent_string_;
            indent_string_ += "│   ";

            if(e->getLop()) {
                e->getLop()->accept(this);
            }

            std::cout << orig_indent;
            std::cout << "└───";
            indent_string_ = indent_string_.replace(orig_indent.length(), strlen("│"), " ");

            if(e->getRop()) {
                e->getRop()->accept(this);
            }

            indent_string_ = orig_indent;
        }


        void drawUnaryOperator(UnaryExpression *e) {

            std::cout << indent_string_;
            std::cout << "└───";

            std::string orig_indent = indent_string_;
            indent_string_ += "    ";

            if(e->getOp()) {
                e->getOp()->accept(this);
            }

            indent_string_ = orig_indent;
        }


        void visitAnd(AndExpression *e) override {
            std::cout << "AND" << std::endl;
            drawBinaryOperator(e);
        }


        void visitOr(OrExpression *e) override {
            std::cout << "OR" << std::endl;
            drawBinaryOperator(e);
        }


        void visitNot(NotExpression *e) override {
            std::cout << "NOT" << std::endl;
            drawUnaryOperator(e);
        }


        void visitLiteral(LiteralExpression *e) override {
            std::cout << e->getLiteral()->getTextFormatted().rdbuf() << std::endl;
        }

    private:
        std::string indent_string_ = "";
};


int main(int argc, char **argv) {

    if(argc < 2) {
        std::cerr << "No file was specified!" << std::endl;
        return 2;
    }

    // Parsing

    std::unique_ptr y = std::make_unique<Yaramod>();

    YaraSourcePtr y_src;
    try {
        y_src = y->parseFile(argv[1]);
    }
    catch(ErrorException &e) {
        std::cerr << "Exception occured: " << std::endl;
        std::cerr << e.what() << std::endl;
        return 1;
    }

    GraphVisitor visitor;
    for(auto &r: y_src->getAllRules()) {
        std::cout << r->getId() << std::endl;
        if(r->getCondition()) {
            r->getCondition()->accept(&visitor);

            std::cout << std::endl;
        }
    }

    std::cout << std::endl;

    return 0;
}
