/**
 * @file yaramod_python_visitor.h
 * @brief Header file of python bindings for visitor class
 * 
 * @author Vojtěch Dvořák 
 */


#pragma once

#include <pybind11/pybind11.h>
#include <pybind11/stl.h>

#include "yaramod.h"
#include "visitor/observing_visitor.h"


/**
 * Creates binding for visitor and its subclasses 
 */

void bindVisitor(pybind11::module &m);


/**
 * Python helper function for Visitor class 
 */
class PyVisitor : public Visitor {
    public:
        using Visitor::Visitor;

        void visitRange(Range *r) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRange, r);
        }

        void visitEnum(Enum *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitEnum, e);
        }

        void visitParentheses(ParenthesesExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitParentheses, e);
        }

        
        void visitIn(InExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitIn, e);
        }

        void visitMatches(MatchesExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitMatches, e);
        }

        void visitAt(AtExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitAt, e);
        }


        // Arithmetics

        void visitAdd(AddExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitAdd, e);
        }

        void visitSub(SubExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitSub, e);
        }

        void visitMul(MulExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitMul, e);
        }

        void visitRemainder(RemainderExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRemainder, e);
        }

        void visitDiv(DivExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitDiv, e);
        }


        // Bitwise operators

        void visitLeftShift(LeftShiftExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitLeftShift, e);
        }

        void visitRightShift(RightShiftExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRightShift, e);
        }

        void visitBitwiseOr(BitwiseOrExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitBitwiseOr, e);
        }

        void visitBitwiseAnd(BitwiseAndExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitBitwiseAnd, e);
        }

        void visitBitwiseXor(BitwiseXorExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitBitwiseXor, e);
        }


        // Relational operators 

        void visitLt(LtExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitLt, e);
        }

        void visitLte(LteExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitLte, e);
        }

        void visitGt(GtExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitGt, e);
        }

        void visitGte(GteExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitGte, e);
        }

        void visitEq(EqExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitEq, e);
        }

        void visitNeq(NeqExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitNeq, e);
        }


        // String relational operators

        void visitContains(ContainsExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitContains, e);
        }

        void visitIContains(IContainsExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitIContains, e);
        }

        void visitStartsWith(StartsWithExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStartsWith, e);
        }

        void visitIStartsWith(IStartsWithExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitIStartsWith, e);
        }

        void visitEndsWith(EndsWithExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitEndsWith, e);
        }

        void visitIEndsWith(IEndsWithExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitIEndsWith, e);
        }

        void visitIEq(IEqExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitIEq, e);
        }


        // Bool operators

        void visitAnd(AndExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitAnd, e);
        }

        void visitOr(OrExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitOr, e);
        }


        // Unary operators

        void visitDefined(DefinedExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitDefined, e);
        }

        void visitUnaryMinus(UnaryMinusExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitUnaryMinus, e);
        }

        void visitBitwiseNot(BitwiseNotExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitBitwiseNot, e);
        }

        void visitNot(NotExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitNot, e);
        }


        // Terminals

        void visitLiteral(LiteralExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitLiteral, e);
        }

        void visitSize(SizeExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitSize, e);
        }

        void visitRegexp(RegexpExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRegexp, e);
        }

        void visitString(StringExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitString, e);
        }

        void visitStringWildcard(StringWildcardExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStringWildcard, e);
        }

        void visitStringCount(StringCountExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStringCount, e);
        }

        void visitStringOffset(StringOffsetExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStringOffset, e);
        }

        void visitStringMatchLength(StringMatchLengthExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStringMatchLength, e);
        }


        // Symbols

        void visitPlainSymbol(PlainSymbol *symbol) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitPlainSymbol, symbol);
        }

        void visitArray(ArrayExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitArray, e);
        }

        void visitStruct(StructExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStruct, e);
        }

        void visitFunctionCall(FunctionCallExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitFunctionCall, e);
        }

        void visitVariable(VariableExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitVariable, e);
        }

        void visitRuleWildCard(RuleWildcardExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRuleWildCard, e);
        }


        // Quantifiers

        void visitNone(NoneExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitNone, e);
        }

        void visitAny(AnyExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitAny, e);
        }

        void visitAll(AllExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitAll, e);
        }

        void visitPercentQuantifier(PercentQuantifierExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitPercentQuantifier, e);
        }

        void visitThem(ThemExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitThem, e);
        }


        // Sets

        void visitStringSet(StringSetExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitStringSet, e);
        }

        void visitRuleSet(RuleSetExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitRuleSet, e);
        }

        void visitExpressionSet(ExpressionSetExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitExpressionSet, e);
        }


        // Of operator

        void visitOf(OfExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitOf, e);
        }


        // For operators

        void visitFor(ForExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitFor, e);
        }

        void visitForInt(ForIntExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitForInt, e);
        }

        void visitVarList(VarListExpression *e) {
            PYBIND11_OVERRIDE_PURE(void, Visitor, visitVarList, e);
        }

};



/**
 * Python helper function for ObservingVisitor class 
 */
class PyObservingVisitor : public ObservingVisitor {
    public:
        using ObservingVisitor::ObservingVisitor;

        void visitRange(Range *r) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Range", visitRange, r);
        }

        void visitEnum(Enum *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Enum", visitEnum, e);
        }

        void visitParentheses(ParenthesesExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Parentheses", visitParentheses, e);
        }

        
        void visitIn(InExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_In", visitIn, e);
        }

        void visitMatches(MatchesExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Matches", visitMatches, e);
        }

        void visitAt(AtExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_At", visitAt, e);
        }


        // Arithmetics

        void visitAdd(AddExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Add", visitAdd, e);
        }

        void visitSub(SubExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Sub", visitSub, e);
        }

        void visitMul(MulExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Mul", visitMul, e);
        }

        void visitRemainder(RemainderExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Remainder", visitRemainder, e);
        }

        void visitDiv(DivExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Div", visitDiv, e);
        }


        // Bitwise operators

        void visitLeftShift(LeftShiftExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_LeftShift", visitLeftShift, e);
        }

        void visitRightShift(RightShiftExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_RightShift", visitRightShift, e);
        }

        void visitBitwiseOr(BitwiseOrExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_BitwiseOr", visitBitwiseOr, e);
        }

        void visitBitwiseAnd(BitwiseAndExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_BitwiseAnd", visitBitwiseAnd, e);
        }

        void visitBitwiseXor(BitwiseXorExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_BitwiseXor", visitBitwiseXor, e);
        }


        // Relational operators 

        void visitLt(LtExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Lt", visitLt, e);
        }

        void visitLte(LteExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Lte", visitLte, e);
        }

        void visitGt(GtExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Gt", visitGt, e);
        }

        void visitGte(GteExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Gte", visitGte, e);
        }

        void visitEq(EqExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Eq", visitEq, e);
        }

        void visitNeq(NeqExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Neq", visitNeq, e);
        }


        // String relational operators

        void visitContains(ContainsExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Contains", visitContains, e);
        }

        void visitIContains(IContainsExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_IContains", visitIContains, e);
        }

        void visitStartsWith(StartsWithExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StartsWith", visitStartsWith, e);
        }

        void visitIStartsWith(IStartsWithExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_IStartsWith", visitIStartsWith, e);
        }

        void visitEndsWith(EndsWithExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_EndsWith", visitEndsWith, e);
        }

        void visitIEndsWith(IEndsWithExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_IEndsWith", visitIEndsWith, e);
        }

        void visitIEq(IEqExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_IEq", visitIEq, e);
        }


        // Bool operators

        void visitAnd(AndExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_And", visitAnd, e);
        }

        void visitOr(OrExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor,"visit_Or", visitOr, e);
        }


        // Unary operators

        void visitDefined(DefinedExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Defined", visitDefined, e);
        }

        void visitUnaryMinus(UnaryMinusExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_UnaryMinus", visitUnaryMinus, e);
        }

        void visitBitwiseNot(BitwiseNotExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_BitwiseNot", visitBitwiseNot, e);
        }

        void visitNot(NotExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Not", visitNot, e);
        }


        // Terminals

        void visitLiteral(LiteralExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Literal", visitLiteral, e);
        }

        void visitSize(SizeExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Size", visitSize, e);
        }

        void visitRegexp(RegexpExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Regexp", visitRegexp, e);
        }

        void visitString(StringExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_String", visitString, e);
        }

        void visitStringWildcard(StringWildcardExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StringWildcard", visitStringWildcard, e);
        }

        void visitStringCount(StringCountExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StringCount", visitStringCount, e);
        }

        void visitStringOffset(StringOffsetExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StringOffset", visitStringOffset, e);
        }

        void visitStringMatchLength(StringMatchLengthExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StringMatchLength", visitStringMatchLength, e);
        }


        // Symbols

        void visitPlainSymbol(PlainSymbol *symbol) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_PlainSymbol", visitPlainSymbol, symbol);
        }

        void visitArray(ArrayExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Array", visitArray, e);
        }

        void visitStruct(StructExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Struct", visitStruct, e);
        }

        void visitFunctionCall(FunctionCallExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_FunctionCall", visitFunctionCall, e);
        }

        void visitVariable(VariableExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Variable", visitVariable, e);
        }

        void visitRuleWildCard(RuleWildcardExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_RuleWildCard", visitRuleWildCard, e);
        }


        // Quantifiers

        void visitNone(NoneExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_None", visitNone, e);
        }

        void visitAny(AnyExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Any", visitAny, e);
        }

        void visitAll(AllExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_All", visitAll, e);
        }

        void visitPercentQuantifier(PercentQuantifierExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_PercentQuantifier", visitPercentQuantifier, e);
        }

        void visitThem(ThemExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Them", visitThem, e);
        }


        // Sets

        void visitStringSet(StringSetExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_StringSet", visitStringSet, e);
        }

        void visitRuleSet(RuleSetExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_RuleSet", visitRuleSet, e);
        }

        void visitExpressionSet(ExpressionSetExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_ExpressionSet", visitExpressionSet, e);
        }


        // Of operator

        void visitOf(OfExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_Of", visitOf, e);
        }


        // For operators

        void visitFor(ForExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_For", visitFor, e);
        }

        void visitForInt(ForIntExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_ForInt", visitForInt, e);
        }

        void visitVarList(VarListExpression *e) {
            PYBIND11_OVERLOAD_NAME(void, ObservingVisitor, "visit_VarList", visitVarList, e);
        }

};

