/**
 * @file yaramod_python_expressions.cpp
 * @brief Implementation of python bindings for Expression class and its 
 * subclasses
 * 
 * @author Vojtěch Dvořák 
 */


#include "yaramod_python_expressions.h"

#include "yaramod.h"

namespace py = pybind11;



template <class T>
py::class_<T> bindBinaryExpressionClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, BinaryExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindUnaryExpressionClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, UnaryExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindTerminalExpressionClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, TerminalExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindSymbolClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, Symbol, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindStringPropertyClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, StringPropertyExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindPlainSymbolClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, PlainSymbol, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindQuantifierExpressionClass(py::module &m, const std::string &py_class_name) {
    return py::class_<T, QuantifierExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}


template <class T>
py::class_<T> bindSetExpression(py::module &m, const std::string &py_class_name) {
    return py::class_<T, SetExpression, std::shared_ptr<T>>(m , py_class_name.c_str());
}



void bindExpressions(py::module &m) {
    // Expression base

    py::class_<Expression, PyExpression, std::shared_ptr<Expression>>(m, "Expression")
        .def_property_readonly("len", &Expression::getLen)
        .def_property_readonly("offset", &Expression::getOffset)
        .def_property_readonly("parent_file", &Expression::getParentFile, py::return_value_policy::reference)
        .def_property_readonly("offset", &Expression::getOffset)
        .def_property_readonly("local_offset", &Expression::getLocalOffset)
        .def_property_readonly("rule", &Expression::getParentRule, py::return_value_policy::reference)
        
        .def("get_type", &Expression::getType)
        .def("is_valid", [](Expression &self) {
                std::string_view msg;
                bool result = self.isValid(msg);
                return py::make_tuple(result, msg);
            })
        .def("are_operands_valid", &Expression::areOperandsValid)
        .def("is_complete", &Expression::isComplete)
        .def("is_undefined", &Expression::isUndefined)
        .def("is_object", &Expression::isObject)
        .def("is_bool", &Expression::isBool)
        .def("is_int", &Expression::isInt)
        .def("is_float", &Expression::isFloat)
        .def("is_string", &Expression::isString)
        .def("is_regexp", &Expression::isRegexp)

        .def("get_text_formatted", [](Expression &self) {
            return self.getTextFormatted().str();
        })

        .def("accept", &Expression::accept)
        .def_static("type_to_str", &Expression::typeToString);

        
    py::class_<Range, Expression, std::shared_ptr<Range>>(m, "RangeExpression")
        .def_property_readonly("from", &Range::getFrom)
        .def_property_readonly("to", &Range::getTo);

    py::class_<Enum, Expression, std::shared_ptr<Enum>>(m, "EnumExpression")
        .def_property_readonly("values", &Enum::getValues);

    py::class_<ParenthesesExpression, Expression, std::shared_ptr<ParenthesesExpression>>(m, "ParenthesesExpression")
        .def_property_readonly("inner_expression", &ParenthesesExpression::getInnerExpression);

    py::class_<InExpression, Expression, std::shared_ptr<InExpression>>(m, "InExpression")
        .def_property_readonly("element", &InExpression::getElementExpression)
        .def_property_readonly("range", &InExpression::getRangeExpression);

    // BinaryExpression

    py::class_<BinaryExpression, PyBinaryExpression, Expression, std::shared_ptr<BinaryExpression>>(m, "BinaryExpression")
        .def_property_readonly("opsing", &BinaryExpression::opsign)
        .def_property_readonly("left", &BinaryExpression::getLop)
        .def_property_readonly("right", &BinaryExpression::getRop);

    bindBinaryExpressionClass<MatchesExpression>(m, "MatchesExpression");
    bindBinaryExpressionClass<AtExpression>(m, "AtExpression");

    bindBinaryExpressionClass<AddExpression>(m, "AddExpression");
    bindBinaryExpressionClass<SubExpression>(m, "SubExpression");
    bindBinaryExpressionClass<MulExpression>(m, "MulExpression");
    bindBinaryExpressionClass<RemainderExpression>(m, "RemainderExpression");
    bindBinaryExpressionClass<DivExpression>(m, "DivExpression");

    bindBinaryExpressionClass<LeftShiftExpression>(m, "LeftShiftExpression");
    bindBinaryExpressionClass<RightShiftExpression>(m, "RightShiftExpression");
    bindBinaryExpressionClass<BitwiseOrExpression>(m, "BitwiseOrExpression");
    bindBinaryExpressionClass<BitwiseAndExpression>(m, "BitwiseAndExpression");
    bindBinaryExpressionClass<BitwiseXorExpression>(m, "BitwiseXorExpression");

    bindBinaryExpressionClass<LtExpression>(m, "LtExpression");
    bindBinaryExpressionClass<LteExpression>(m, "LteExpression");
    bindBinaryExpressionClass<GtExpression>(m, "GtExpression");
    bindBinaryExpressionClass<GteExpression>(m, "GteExpression");
    bindBinaryExpressionClass<EqExpression>(m, "EqExpression");
    bindBinaryExpressionClass<NeqExpression>(m, "NeqExpression");

    bindBinaryExpressionClass<ContainsExpression>(m, "ContainsExpression");
    bindBinaryExpressionClass<IContainsExpression>(m, "IContainsExpression");
    bindBinaryExpressionClass<StartsWithExpression>(m, "StartsWithExpression");
    bindBinaryExpressionClass<IStartsWithExpression>(m, "IStartsWithExpression");
    bindBinaryExpressionClass<EndsWithExpression>(m, "EndsWithExpression");
    bindBinaryExpressionClass<IEndsWithExpression>(m, "IEndsWithExpression");
    bindBinaryExpressionClass<IEqExpression>(m, "IEqExpression");

    bindBinaryExpressionClass<AndExpression>(m, "AndExpression");
    bindBinaryExpressionClass<OrExpression>(m, "OrExpression");


    // UnaryExpression

    py::class_<UnaryExpression, PyUnaryExpression, Expression, std::shared_ptr<UnaryExpression>>(m, "UnaryExpression")
        .def_property_readonly("opsing", &UnaryExpression::opsign)
        .def_property_readonly("operand", &UnaryExpression::getOp);

    bindUnaryExpressionClass<DefinedExpression>(m, "DefinedExpression");
    bindUnaryExpressionClass<UnaryMinusExpression>(m, "UnaryMinusExpression");
    bindUnaryExpressionClass<BitwiseNotExpression>(m, "BitwiseNotExpression");
    bindUnaryExpressionClass<NotExpression>(m, "NotExpression");


    // TerminalExpression

    py::class_<TerminalExpression, Expression, std::shared_ptr<TerminalExpression>>(m, "TerminalExpression");

    bindTerminalExpressionClass<LiteralExpression>(m, "LiteralExpression")
        .def_property_readonly("literal", [](LiteralExpression &self) {
            const auto &literal = self.getLiteral();
            if(literal) {
                return literal.get();
            }
            else {
                return static_cast<Literal *>(nullptr);
            }
        }, py::return_value_policy::reference);

    bindTerminalExpressionClass<SizeExpression>(m, "SizeExpression")
        .def_property_readonly("value", &SizeExpression::getValue)
        .def_property_readonly("unit", &SizeExpression::getUnit)
        .def_static("unit_to_str", &SizeExpression::unitToString);

    bindTerminalExpressionClass<RegexpExpression>(m, "RegexpExpression")
        .def_property_readonly("content", &RegexpExpression::getContent);

    bindTerminalExpressionClass<StringExpression>(m, "StringExpression")
        .def_property_readonly("id", &StringExpression::getId);

    py::class_<StringWildcardExpression, StringExpression, std::shared_ptr<StringWildcardExpression>>(m, "StringWildcardExpression");
    
    bindTerminalExpressionClass<StringCountExpression>(m, "StringCountExpression")
        .def_property_readonly("id", &StringCountExpression::getId);


    // StringPropertyExpression

    py::class_<StringPropertyExpression, Expression, std::shared_ptr<StringPropertyExpression>>(m, "StringPropertyExpression")
        .def_property_readonly("id", &StringPropertyExpression::getId)
        .def_property_readonly("index", &StringPropertyExpression::getIndex);

    bindStringPropertyClass<StringOffsetExpression>(m, "StringOffsetExpression");
    bindStringPropertyClass<StringMatchLengthExpression>(m, "StringMatchLengthExpression");
    
    
    // Symbol

    py::class_<Symbol, Expression, std::shared_ptr<Symbol>>(m, "Symbol")
        .def_property_readonly("sym_type", &Symbol::getSymType)
        .def_property_readonly("ctx", &Symbol::getContext)
        .def("is_function", &Symbol::isFunction)
        .def("is_array", &Symbol::isArray)
        .def("is_dict", &Symbol::isDict)
        .def("is_struct", &Symbol::isStruct)
        .def("is_value", &Symbol::isValue)
        .def("is_defined", &Symbol::isDefined);

    bindSymbolClass<ArrayExpression>(m, "ArrayExpression")
        .def_property_readonly("array", &ArrayExpression::getArray)
        .def_property_readonly("key", &ArrayExpression::getKey);

    bindSymbolClass<StructExpression>(m, "StructExpression")
        .def_property_readonly("structure", &StructExpression::getStructure)
        .def_property_readonly("member", &StructExpression::getMember);

    bindSymbolClass<FunctionCallExpression>(m, "FunctionCallExpression")
        .def_property_readonly("function", &FunctionCallExpression::getFunction)
        .def_property_readonly("args", &FunctionCallExpression::getArgs)
        .def("has_valid_args", &FunctionCallExpression::hasValidArgs);


    bindSymbolClass<PlainSymbol>(m, "PlainSymbol")
        .def_property_readonly("id", &PlainSymbol::getId);

    bindPlainSymbolClass<VariableExpression>(m, "VariableExpression");
    bindPlainSymbolClass<RuleWildcardExpression>(m, "RuleWildcardExpression");
   

    // QuantifierExpression

    py::class_<QuantifierExpression, Expression, std::shared_ptr<QuantifierExpression>>(m, "QuantifierExpression");

    bindQuantifierExpressionClass<NoneExpression>(m, "NoneExpression");
    bindQuantifierExpressionClass<AnyExpression>(m, "AnyExpression");
    bindQuantifierExpressionClass<AllExpression>(m, "AllExpression");
    bindQuantifierExpressionClass<PercentQuantifierExpression>(m, "PercentQuantifierExpression")
        .def_property_readonly("value", &PercentQuantifierExpression::getValue);
    


    py::class_<ThemExpression, Expression, std::shared_ptr<ThemExpression>>(m, "ThemExpression");


    py::class_<SetExpression, Expression, std::shared_ptr<SetExpression>>(m, "SetExpression")
        .def_property_readonly("elements", &SetExpression::getElements);

    bindSetExpression<StringSetExpression>(m, "StringSetExpression");
    bindSetExpression<RuleSetExpression>(m, "RuleSetExpression");
    bindSetExpression<ExpressionSetExpression>(m, "ExpressionSetExpression");


    // Other expression subclasses

    py::class_<OfExpression, Expression, std::shared_ptr<OfExpression>>(m, "OfExpression")
        .def_property_readonly("quantifier", &OfExpression::getQuantifier)
        .def_property_readonly("set", &OfExpression::getSet);


    py::class_<ForExpression, Expression, std::shared_ptr<ForExpression>>(m, "ForExpression")
        .def_property_readonly("quantifier", &ForExpression::getQuantifier)
        .def_property_readonly("set", &ForExpression::getSet)
        .def_property_readonly("inner_expression", &ForExpression::getExpression);

    
    py::class_<VarListExpression, Expression, std::shared_ptr<VarListExpression>>(m, "VarListExpression")
        .def_property_readonly("vars", &VarListExpression::getVariables);


    py::class_<ForIntExpression, Expression, std::shared_ptr<ForIntExpression>>(m, "ForIntExpression")
        .def_property_readonly("quantifier", &ForIntExpression::getQuantifier)
        .def_property_readonly("variables", &ForIntExpression::getVarList)
        .def_property_readonly("iterable", &ForIntExpression::getIterable)
        .def_property_readonly("expression", &ForIntExpression::getExpression);
}

