/**
 * @file yaramod_python.cpp
 * @brief Contains implementation of bindings for python, that are 
 * created through pybind11 library
 * 
 * @author Vojtěch Dvořák 
 */


#include "yaramod_python.h"
#include "yaramod_python_expressions.h"
#include "yaramod_python_visitor.h"

#include "yaramod.h"


#define MODULE_NAME "yaramodv4"

namespace py = pybind11;


/**
 * Python class for virtual class Printable
 */
class PyPrintable : public Printable {
    std::stringstream getTextFormatted() const override {
        PYBIND11_OVERRIDE_PURE(std::stringstream, Printable, getTextFormatted);
    }
};



/**
 * Bind enum values of yaramodv4
 */
void bindEnumClasses(py::module &m) {
    py::enum_<YaramodConfig::ParsingMode>(m, "ParsingMode")
        .value("Manual", YaramodConfig::ParsingMode::Manual)
        .value("Auto", YaramodConfig::ParsingMode::Auto);


    py::enum_<YaramodConfig::ErrorMode>(m, "ErrorMode")
        .value("Strict", YaramodConfig::ErrorMode::Strict)
        .value("Tolerant", YaramodConfig::ErrorMode::Tolerant);


    py::enum_<RuleModifier::Type>(m, "RuleModifierType")
        .value("Global", RuleModifier::Type::Global)
        .value("Private", RuleModifier::Type::Private);


    py::enum_<StringModifier::Type>(m, "StringModifierType")
        .value("Ascii", StringModifier::Type::Ascii)
        .value("Base64", StringModifier::Type::Base64)
        .value("Base64Wide", StringModifier::Type::Base64Wide)
        .value("Fullword", StringModifier::Type::Fullword)
        .value("NoCase", StringModifier::Type::NoCase)
        .value("Private", StringModifier::Type::Private)
        .value("Wide", StringModifier::Type::Wide)
        .value("Xor", StringModifier::Type::Xor);


    py::enum_<String::Type>(m, "StringType")
        .value("Plain", String::Type::Plain)
        .value("Hex", String::Type::Hex)
        .value("Regexp", String::Type::Regexp);


    py::enum_<Expression::Type>(m, "ExpressionType")
        .value("Undefined", Expression::Type::Undefined)
        .value("String", Expression::Type::String)
        .value("Regexp", Expression::Type::Regexp)
        .value("Object", Expression::Type::Object)
        .value("Int", Expression::Type::Int)
        .value("Float", Expression::Type::Float)
        .value("Bool", Expression::Type::Bool);


    py::enum_<Symbol::Type>(m, "SymbolType")
        .value("Unknown", Symbol::Type::Unknown)
        .value("Value", Symbol::Type::Value)
        .value("Struct", Symbol::Type::Struct)
        .value("Function", Symbol::Type::Function)
        .value("Dict", Symbol::Type::Dict)
        .value("Array", Symbol::Type::Array);


    py::enum_<SizeExpression::Unit>(m, "SizeExpressionUnit")
        .value("KB", SizeExpression::Unit::KB)
        .value("MB", SizeExpression::Unit::MB);
}



/**
 * Creates bindings SymTab class that stores objects of type T
 */
template<class T>
void bindSymTab(py::module &m, const std::string &prefix) {
    py::class_<SymTab<T>>(m, std::string(prefix + "SymTab").c_str())
        .def("__iter__", [](SymTab<T> &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &SymTab<T>::size)

        .def("__getitem__", [](SymTab<T> &self, size_t index) {
            return self.operator[](index);
        }, py::return_value_policy::reference)
        .def("__getitem__", [](SymTab<T> &self, const std::string &key) {
            return self.operator[](key);
        }, py::return_value_policy::reference)
        
        .def_property_readonly("size", &SymTab<T>::size)
        .def_property_readonly("data", &SymTab<T>::dataByOffset)

        .def("exists", [](SymTab<T> &self, const std::string &id) {
            return self.exists(id);
        })

        .def("search", [](SymTab<T> &self, const std::string &id) {
            return self.search(id).get();
        }, py::return_value_policy::reference)

        .def("is_empty", &SymTab<T>::empty);
}


/**
 * Creates bindings SymTabOffset, that stores objects of type T
 */
template<class T>
void bindSymtabOffset(py::module &m, const std::string &prefix) {
    py::class_<SymTabOffset<T>>(m, std::string(prefix + "SymtabOffset").c_str())
        .def("__iter__", [](SymTabOffset<T> &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &SymTabOffset<T>::size)
        .def("__getitem__", &SymTabOffset<T>::operator[], py::return_value_policy::reference)
        
        .def_property_readonly("size", &SymTabOffset<T>::size)
        .def_property_readonly("data", [](SymTabOffset<T> &self) {
            std::vector<T *> result;
            for(auto &entry: self.data()) {
                result.push_back(entry.second.get());
            }

            return result;
        })

        .def("exists", &SymTabOffset<T>::exists)
        .def("get", [](SymTabOffset<T> &self, const offset_t &offset) {
            return self.get(offset).get();
        }, py::return_value_policy::reference)

        .def("is_empty", &SymTabOffset<T>::empty);
}


/**
 * Creates bindings SymTabDuplId, that stores objects of type T
 */
template<class T>
void bindSymtabDuplId(py::module &m, const std::string &prefix) {
    using Pool = SymTabDuplId<T>::Pool;
    using Tab = SymTabDuplId<T>;

    // Bindings for Pool, that stores all objects with the same id
    py::class_<Pool>(m, std::string(prefix + "SymTabDuplIdPool").c_str())
        .def("__iter__", [](Pool &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &Pool::size)
        .def("__getitem__", &Pool::operator[], py::return_value_policy::reference)
        .def_property_readonly("first", &Pool::first, py::return_value_policy::reference)
        .def("at", &Pool::at, py::return_value_policy::reference);

    // Bindings for symbol tab itself
    py::class_<Tab>(m, std::string(prefix + "SymTabDuplId").c_str())
        .def("__iter__", [](Tab &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &Tab::size)

        .def("__getitem__", [](Tab &self, size_t index) {
            return self.operator[](index);
        }, py::return_value_policy::reference)
        .def("__getitem__", [](Tab &self, const std::string &key) {
            return &self.operator[](key);
        }, py::return_value_policy::reference)
        
        .def_property_readonly("size", &Tab::size)
        .def_property_readonly("data", &Tab::dataByOffset)

        .def("exists", [](Tab &self, const std::string &id) {
            return self.exists(id);
        })

        .def("get", [](Tab &self, const std::string &id) {
            return self.get(id);
        }, py::return_value_policy::reference)

        .def("get_all", [](Tab &self, const std::string &id) {
            return &(self.getAll(id));
        }, py::return_value_policy::reference)

        .def("is_empty", &Tab::empty);
}


/**
 * Bindings for error collector, that stores errors of type E 
 */
template<class E>
void bindErrorCollector(py::module &m, const std::string &prefix) {
    bindSymtabOffset<E>(m, prefix);

    py::class_<ErrorCollector<E>>(m, std::string(prefix + "ErrorCollector").c_str())
        .def("get", &ErrorCollector<E>::get, py::return_value_policy::reference_internal)
        .def("dump", &ErrorCollector<E>::dump);
}


/**
 * Helper function for translating Exceptions 
 */
void setYaraExceptionObject(const YaraErrorException &e, const char *name) {
    py::object exc_obj = py::module_::import(MODULE_NAME)
        .attr(name)(e.getOffset(), e.getLen(), e.what());

    PyErr_SetObject(PyExc_RuntimeError, exc_obj.ptr());
}


/**
 * Creates bindings for exceptions hierarchy of Yaramodv4 
 */
void bindExceptions(py::module &m) {
    py::register_exception<InternalErrorException>(m, "InternalErrorException", PyExc_RuntimeError);
    py::register_exception<YaramodErrorException>(m, "YaramodErrorException", PyExc_RuntimeError);

    py::register_exception<MissingTokenErrorException>(m, "YaraMissingTokenError", PyExc_RuntimeError);
    py::register_exception<UnexpectedTokenErrorException>(m, "YaraUnexpectedTokenError", PyExc_RuntimeError);
    py::register_exception<SyntaxErrorException>(m, "YaraSyntaxError", PyExc_RuntimeError);

    py::register_exception<OverflowErrorException>(m, "YaraOverflowError", PyExc_RuntimeError);
    
    py::register_exception<IndexErrorException>(m, "YaraIndexError", PyExc_RuntimeError);
    py::register_exception<RuleReferenceErrorException>(m, "YaraRuleReferenceError", PyExc_RuntimeError);
    py::register_exception<StringReferenceErrorException>(m, "YaraStringReferenceError", PyExc_RuntimeError);
    py::register_exception<RangeErrorException>(m, "YaraRangeError", PyExc_RuntimeError);
    py::register_exception<FunctionCallErrorException>(m, "YaraFunctionCallError", PyExc_RuntimeError);
    py::register_exception<SymbolReferenceErrorException>(m, "YaraSymbolReferenceError", PyExc_RuntimeError);
    py::register_exception<SymbolErrorException>(m, "YaraSymbolError", PyExc_RuntimeError);
    py::register_exception<IncludeErrorException>(m, "YaraIncludeError", PyExc_RuntimeError);
    py::register_exception<ExpressionErrorException>(m, "YaraExpressionError", PyExc_RuntimeError);
    
    py::register_exception<StringModifierErrorException>(m, "YaraStringModifierError", PyExc_RuntimeError);
    py::register_exception<RegexpErrorException>(m, "YaraRegexpError", PyExc_RuntimeError);
    py::register_exception<HexStrErrorException>(m, "YaraHexStrError", PyExc_RuntimeError);
    py::register_exception<StringErrorException>(m, "YaraStringError", PyExc_RuntimeError);

    py::register_exception<ModuleErrorException>(m, "YaraModuleError", PyExc_RuntimeError);
    
    py::register_exception<RuleModifierErrorException>(m, "YaraRuleModifierError", PyExc_RuntimeError);
    py::register_exception<RuleErrorException>(m, "YaraRuleError", PyExc_RuntimeError);

    py::register_exception<VariableErrorException>(m, "YaraVariableError", PyExc_RuntimeError);
    py::register_exception<SemanticErrorException>(m, "YaraSemanticError", PyExc_RuntimeError);
    
    // Exception translator (idiomatic way how to transform Yaramod-v4 exception hierarchy to python)
    py::register_exception_translator([](std::exception_ptr e_ptr) {
        try {
            if(e_ptr) {
                std::rethrow_exception(e_ptr);
            }
        }
        catch(const InternalErrorException &e) {
            PyErr_SetString(PyExc_RuntimeError, e.what());
        }
        catch(const YaramodErrorException &e) {
            PyErr_SetString(PyExc_RuntimeError, e.what());
        }
        catch(const UnexpectedTokenErrorException &e) {
            setYaraExceptionObject(e , "YaraUnexpectedTokenError");
        }
        catch(const MissingTokenErrorException &e) {
            setYaraExceptionObject(e , "YaraMissingTokenError");
        }
        catch(const SyntaxErrorException &e) {
            setYaraExceptionObject(e , "YaraSyntaxError");
        }
        catch(const IndexErrorException &e) {
            setYaraExceptionObject(e , "YaraIndexError");
        }
        catch(const RuleReferenceErrorException &e) {
            setYaraExceptionObject(e , "YaraRuleReferenceError");
        }
        catch(const StringReferenceErrorException &e) {
            setYaraExceptionObject(e , "YaraStringReferenceError");
        }
        catch(const RangeErrorException &e) {
            setYaraExceptionObject(e , "YaraRangeError");
        }
        catch(const FunctionCallErrorException &e) {
            setYaraExceptionObject(e , "YaraFunctionCallError");
        }
        catch(const SymbolReferenceErrorException &e) {
            setYaraExceptionObject(e , "YaraSymbolReferenceError");
        }
        catch(const SymbolErrorException &e) {
            setYaraExceptionObject(e , "YaraSymbolError");
        }
        catch(const ExpressionErrorException &e) {
            setYaraExceptionObject(e , "YaraExpressionError");
        }
        catch(const IncludeErrorException &e) {
            setYaraExceptionObject(e , "YaraIncludeError");
        }
        catch(const StringModifierErrorException &e) {
            setYaraExceptionObject(e , "YaraStringModifierError");
        }
        catch(const RegexpErrorException &e) {
            setYaraExceptionObject(e , "YaraRegexpError");
        }
        catch(const HexStrErrorException &e) {
            setYaraExceptionObject(e , "YaraHexStrError");
        }
        catch(const StringErrorException &e) {
            setYaraExceptionObject(e , "YaraStringError");
        }
        catch(const OverflowErrorException &e) {
            setYaraExceptionObject(e , "YaraOverflowError");
        }
        catch(const ModuleErrorException &e) {
            setYaraExceptionObject(e , "YaraModuleError");
        }
        catch(const RuleModifierErrorException &e) {
            setYaraExceptionObject(e , "YaraRuleModifierError");
        }
        catch(const RuleErrorException &e) {
            setYaraExceptionObject(e , "YaraRuleError");
        }
        catch(const VariableErrorException &e) {
            setYaraExceptionObject(e , "YaraVariableError");
        }
        catch(const SemanticErrorException &e) {
            setYaraExceptionObject(e , "YaraSemanticError");
        }
    });
}


/**
 * Creates bindings for strings and its components
 */
void bindStrings(py::module &m) {
    py::class_<StringModifier, RuleElement, Printable>(m, "StringModifier")
        .def_property_readonly("type", &StringModifier::getType)
        .def("has_any_arg", &StringModifier::hasAnyArg)
        .def("has_key_arg", [](StringModifier &self) {
            return self.hasArg<StringModifier::Key>();
        })
        .def("has_range_arg", [](StringModifier &self) {
            return self.hasArg<StringModifier::Range>();
        })
        .def("has_alpha_arg", [](StringModifier &self) {
            return self.hasArg<StringModifier::Alphabet>();
        })
        .def("get_arg", &StringModifier::getArg)
        .def_static("str_to_type", &StringModifier::stringToType)
        .def_static("type_to_str", &StringModifier::typeToString);

    py::class_<StringModifierContainer>(m, "StringModifierContainer")
        .def("__iter__", [](StringModifierContainer &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &StringModifierContainer::size)
        .def("empty", &StringModifierContainer::empty)
        .def("has", &StringModifierContainer::has)
        .def("get", &StringModifierContainer::get, py::return_value_policy::reference);


    py::class_<String, RuleElement, Printable>(m, "String")
        .def_property_readonly("id", [](String &self) {
            return self.getId();
        })
        .def_property_readonly("type", &String::getType)
        .def_property_readonly("content", &String::getContent)
        .def_property_readonly("modifiers", &String::getModifiers)
        .def("get_real_id", &String::getId, py::arg("ignore_internal_id") = true)
        .def("is_anonymous", &String::isAnonymous)
        .def_static("type_to_str", &String::typeToString);
}


/**
 * Creates bindings for rule and its components
 */
void bindRules(py::module &m) {
    bindExpressions(m);

    bindSymtabDuplId<Meta>(m, "Meta");
    bindSymTab<String>(m, "String");
    bindSymTab<IntVariable>(m, "IntVariable");


    py::class_<RuleModifier, YaraFileElement>(m, "RuleModifier")
        .def_property_readonly("type", &RuleModifier::getType)
        .def_static("str_to_type", &RuleModifier::stringToType)
        .def_static("type_to_str", &RuleModifier::typeToString);

    py::class_<RuleModifierContainer>(m, "RuleModifierContainer")
        .def("__iter__", [](RuleModifierContainer &self) {
            return py::make_iterator(self.begin(), self.end());
        }, py::keep_alive<0, 1>())
        .def("__len__", &RuleModifierContainer::size)
        .def("empty", &RuleModifierContainer::empty)
        .def("has", &RuleModifierContainer::has)
        .def("has_global", &RuleModifierContainer::hasGlobal)
        .def("has_private", &RuleModifierContainer::hasPrivate)
        .def("get_global", &RuleModifierContainer::getGlobal, py::return_value_policy::reference)
        .def("get_private", &RuleModifierContainer::getPrivate, py::return_value_policy::reference);
    
    py::class_<Rule, TopLevelYaraFileElement, Printable>(m, "Rule")
        .def_property_readonly("id", &Rule::getId)
        .def_property_readonly("modifiers", &Rule::getModifiers)
        .def_property_readonly("tags", &Rule::getTags)
        .def_property_readonly("metas", &Rule::getMetas)
        .def_property_readonly("strings", &Rule::getStrings)
        .def_property_readonly("vars", &Rule::getVars)
        .def_property_readonly("condition", &Rule::getCondition)
        .def("is_global", &Rule::isGlobal)
        .def("is_private", &Rule::isPrivate);

    
    py::class_<RuleElement, YaraFileElement>(m, "RuleElement")
        .def_property_readonly("offset", &RuleElement::getOffset)
        .def_property_readonly("local_offset", &RuleElement::getLocalOffset)
        .def_property_readonly("rule", &RuleElement::getParentRule, py::return_value_policy::reference)
        .def("get_range", &RuleElement::getRange)
        .def("get_position", &RuleElement::getPosition);


    bindStrings(m);
    
    py::class_<Meta, RuleElement, Printable>(m, "Meta")
        .def_property_readonly("id", &Meta::getId)
        .def_property_readonly("value", [](Meta &self) {
            const auto &result = self.getValue();
            if(result) {
                return result.get();
            }
            else {
                return static_cast<Literal *>(nullptr);
            }
        });

    py::class_<IntVariable, RuleElement, Printable>(m, "IntVariable")
        .def_property_readonly("id", &IntVariable::getId)
        .def_property_readonly("value", &IntVariable::getValue);
} 


/**
 * Creates binding for Import class 
 */
void bindImports(py::module &m) {
    py::class_<Module>(m, "Module")
        .def_property_readonly("name", &Module::getName)
        .def_property_readonly("data", [](Module &self) {
            return self.getData().get();
        }, py::return_value_policy::reference);


    py::class_<Import, TopLevelYaraFileElement, Printable>(m, "Import")
        .def_property_readonly("module_name", &Import::getModuleName)
        .def_property_readonly("module", &Import::getModule, py::return_value_policy::reference);
}


/**
 * Creates binding for FileInclude class 
 */
void bindErrors(py::module &m) {
    py::class_<Error, TopLevelYaraFileElement>(m, "Error")
        .def_property_readonly("description", &Error::getDescription)
        .def("exception", &Error::exception);

    py::class_<SyntaxError, Error>(m, "SyntaxError");
    py::class_<UnexpectedTokenError, SyntaxError>(m, "UnexpectedTokenError");
    py::class_<MissingTokenError, SyntaxError>(m, "MissingTokenError");

    py::class_<SemanticError, Error>(m, "SemanticError")
        .def_property_readonly("is_global", &SemanticError::isGlobal);

    py::class_<ExpressionError, SemanticError>(m, "ExpressionError");
    py::class_<IndexError, ExpressionError>(m, "IndexError");
    py::class_<RuleReferenceError, ExpressionError>(m, "RuleReferenceError");
    py::class_<StringReferenceError, ExpressionError>(m, "StringReferenceError");
    py::class_<RangeError, ExpressionError>(m, "RangeError");
    py::class_<FunctionCallError, ExpressionError>(m, "FunctionCallError");
    py::class_<SymbolReferenceError, ExpressionError>(m, "SymbolReferenceError");
    py::class_<SymbolError, ExpressionError>(m, "SymbolError");

    py::class_<IncludeError, SemanticError>(m, "IncludeError");

    py::class_<StringError, SemanticError>(m, "StringError");
    py::class_<StringModifierError, StringError>(m, "StringModifierError");
    py::class_<RegexpError, StringError>(m, "RegexpError");
    py::class_<HexStrError, StringError>(m, "HexStrError");

    py::class_<OverflowError, SemanticError>(m, "OverflowError");
    py::class_<ModuleError, SemanticError>(m, "ModuleError");

    py::class_<RuleError, SemanticError>(m, "RuleError");
    py::class_<RuleModifierError, RuleError>(m, "RuleModifierError");

    py::class_<VariableError, SemanticError>(m, "VariableError");
}


/**
 * Creates error bindings for classes, that represent errors 
 */
void bindFileIncludes(py::module &m) {
     py::class_<FileInclude, TopLevelYaraFileElement, Printable>(m, "FileInclude")
        .def_property_readonly("path", &FileInclude::getPath)
        .def_property_readonly("file", &FileInclude::getFile, py::return_value_policy::reference);
}


/**
 * Creates bindings for comments 
 */
void bindComments(py::module &m) {
    py::class_<Comment, TopLevelYaraFileElement, Printable>(m, "Comment")
        .def_property_readonly("text", &Comment::getText);
}


/**
 * Creates bindings for YaraFile and its components
 */
void bindYaraFile(py::module &m) {
    py::class_<YaraFileElement>(m, "YaraFileElement")
        .def_property_readonly("len", &YaraFileElement::getLen)
        .def_property_readonly("offset", &YaraFileElement::getOffset)
        .def_property_readonly("parent_file", &YaraFileElement::getParentFile, py::return_value_policy::reference)
        .def("get_position", &YaraFileElement::getPosition)
        .def("get_range", &YaraFileElement::getRange);

    py::class_<TopLevelYaraFileElement, YaraFileElement>(m, "TopLevelYaraFileElement")
        .def("get_position", &TopLevelYaraFileElement::getPosition)
        .def("get_range", &TopLevelYaraFileElement::getRange);


    bindExceptions(m);

    bindErrors(m);
    bindErrorCollector<SyntaxError>(m, "SyntaxError");
    bindErrorCollector<SemanticError>(m, "SemanticError");

    bindRules(m);
    bindSymTab<Rule>(m, "Rule");

    bindFileIncludes(m);
    bindSymTab<FileInclude>(m, "FileInclude");
    
    bindImports(m);
    bindSymtabDuplId<Import>(m, "Import");
    
    bindComments(m);
    bindSymtabOffset<Comment>(m, "Comment");


    py::class_<YaraFile, std::shared_ptr<YaraFile>>(m, "YaraFile")
        .def_property_readonly("name", &YaraFile::getName)
        .def_property_readonly("orig_str", [](YaraFile &self) {
            return self.ctx().getString();
        })
        .def_property_readonly("syntax_errors", &YaraFile::getSyntaxErrors)
        .def_property_readonly("semantic_errors", &YaraFile::getSemanticErrors)
        .def_property_readonly("local_rules", &YaraFile::getLocalRules)
        .def_property_readonly("rules", [](YaraFile &file) {
            return file.getRules();
        })
        .def_property_readonly("local_imports", &YaraFile::getLocalImports)
        .def_property_readonly("imports", &YaraFile::getImports)
        .def_property_readonly("comments", &YaraFile::getComments)
        .def_property_readonly("local_file_includes", &YaraFile::getLocalFileIncludes)
        .def_property_readonly("file_includes", &YaraFile::getFileIncludes)
        .def("get_text_formatted", [](YaraFile &self) {
            return self.getTextFormatted().str();
        })
        .def("get_rule", [](YaraFile &self, const std::string &name) {
            return self.getRule(name);
        }, py::return_value_policy::reference)
        .def("get_local_rule_by_row", [](YaraFile &self, uint32_t &row) {
            return self.getLocalRuleByRow(row);
        }, py::return_value_policy::reference)
        .def("has_rule", [](YaraFile &self, const std::string &name) {
            return self.hasRule(name);
        })
        .def("get_import", [](YaraFile &self, const std::string &name) {
            return self.getImport(name);
        }, py::return_value_policy::reference)
        .def("was_edited", &YaraFile::wasEdited)
        .def("must_be_edited", &YaraFile::mustBeReparsed)
        .def("edit", py::overload_cast<const offset_t &, const size_t &, const size_t &>(&YaraFile::edit))
        .def("edit", py::overload_cast<const range_t &, const std::string &>(&YaraFile::edit))
        .def("undo", &YaraFile::undo);
}



PYBIND11_MODULE(yaramodv4, m) {


    // Enums

    bindEnumClasses(m);


    // Other types and structs

    py::class_<point_t>(m, "Point")
        .def(py::init<>())
        .def(py::init<uint32_t, uint32_t>())
        .def_readwrite("row", &point_t::row)
        .def_readwrite("col", &point_t::col);


    py::class_<range_t>(m, "Range")
        .def(py::init<>())
        .def(py::init<uint32_t, uint32_t, uint32_t, uint32_t>())
        .def(py::init<point_t, point_t>())
        .def_readwrite("start", &range_t::start)
        .def_readwrite("end", &range_t::end);


    // Classes

    py::class_<Printable, PyPrintable>(m, "Printable")
        .def("get_text_formatted", [](Printable &self) {
            return self.getTextFormatted().str();
        });


    bindYaraFile(m);


    py::class_<YaraSource, Printable>(m, "YaraSource")
        .def_property_readonly("entry_file", &YaraSource::getEntryFile)
        .def_property_readonly("files", &YaraSource::getFiles)
        .def_property_readonly("all_rules", &YaraSource::getAllRules)
        .def_property_readonly("all_syntax_errors", &YaraSource::getAllSyntaxErrors)
        .def_property_readonly("all_semantic_errors", &YaraSource::getAllSemanticErrors)
        .def_property("base_dir", &YaraSource::getBaseDir, &YaraSource::setBaseDir)
        .def("get_file", &YaraSource::getFile)
        .def("rename_file", &YaraSource::renameFile)
        .def("has_file", &YaraSource::hasFile)
        .def("remove_file", &YaraSource::removeFile)
        .def("set_entry_file", &YaraSource::setEntryFile)
        .def("edit", py::overload_cast<const offset_t &, const size_t &, const size_t &>(&YaraSource::edit))
        .def("edit", py::overload_cast<const offset_t &, const size_t &, const size_t &, const std::string &>(&YaraSource::edit))
        .def("edit", py::overload_cast<const range_t &, const std::string &>(&YaraSource::edit))
        .def("edit", py::overload_cast<const range_t &, const std::string &, const std::string &>(&YaraSource::edit))
        .def("undo", &YaraSource::undo);
    


    py::class_<Literal, Printable>(m, "Literal")
        .def("is_undefined", &Literal::isUndefined)
        .def("is_str", &Literal::isString)
        .def("is_int", &Literal::isInt)
        .def("is_float", &Literal::isFloat)
        .def("is_bool", &Literal::isBool)
        .def("get_value", &Literal::getRaw)
        .def_property_readonly("str", &Literal::get<std::string>)
        .def_property_readonly("int", &Literal::get<int64_t>)
        .def_property_readonly("float", &Literal::get<float>)
        .def_property_readonly("bool", &Literal::get<bool>);



    py::class_<ExtVariable>(m, "ExtVariable")
        .def(py::init<>())
        .def(py::init<const std::string &, int64_t>())
        .def(py::init<const std::string &, bool>())
        .def(py::init<const std::string &, const std::string &>())
        .def_property_readonly("id", &ExtVariable::getId);



    py::class_<YaramodConfig>(m, "YaramodConfig")
        .def(py::init<>())

        .def_property("module_dir", &YaramodConfig::getModuleDir, &YaramodConfig::setModuleDir)
        .def_property("base_dir", &YaramodConfig::getBaseDir, &YaramodConfig::setBaseDir)
        .def_property("error_mode", &YaramodConfig::getErrorMode, &YaramodConfig::setErrorMode)

        .def("add_var", py::overload_cast<const std::string &, const std::string &>(&YaramodConfig::addVar))
        .def("add_var", py::overload_cast<const std::string &, int64_t>(&YaramodConfig::addVar))
        .def("get_var", &YaramodConfig::getVar)
        .def("remove_var", &YaramodConfig::removeVar);



    py::class_<Yaramod>(m, "Yaramod")
        .def(py::init<>())
        .def(py::init<
                const std::string &, 
                const std::string &, 
                YaramodConfig::ParsingMode, 
                YaramodConfig::ErrorMode
            >(),
            py::arg("base_dir"),
            py::arg("module_dir") = "modules", 
            py::arg("p_mode") = YaramodConfig::ParsingMode::Auto, 
            py::arg("e_mode") = YaramodConfig::ErrorMode::Tolerant
        )
        .def_property_readonly("config", &Yaramod::config)
        .def("load_modules", &Yaramod::loadModules)
        .def("parse_string", [](Yaramod &self, const std::string &str) {
            return self.parseString(str);
        })
        .def("parse_string", [](Yaramod &self, const std::string &str, YaraSource *old) {
            self.parseString(str, old);
            return old;
        })
        .def("parse_string", [](Yaramod &self, const std::string &str, const std::string &entry_filename) {
            return self.parseString(str, entry_filename);
        })
        .def("parse_string", [](Yaramod &self, const std::string &str, YaraSource *old, const std::string &entry_filename) {
            self.parseString(str, old, entry_filename);
            return old;
        })

        .def("parse_file", [](Yaramod &self, const std::string &path) {
            return self.parseFile(path);
        })
        .def("parse_file", [](Yaramod &self, const std::string &path, YaraSource *old) {
            self.parseFile(path, old);
            return old;
        });


        // Binding of Visitor class

        bindVisitor(m);
}
