# incremental_parsing.py
#
# Author: Vojtěch Dvořák
#
# Script for pytest framework, that tests yaramod python bindings, especially
# interface for incremental parsing in python.
#
# Requirements: pytest, yaramod_python


import os
import yaramodv4 as y

TMP_FILENAMES = [os.path.abspath("__test_file1__.tmp"), os.path.abspath("__test_file2__.tmp")]


def test_reparsing_without_edit():
    """Tests of reparsing file, that was not edited"""

    yara_string = """
rule a { 
    condition: true
}

rule b { 
    condition: true
}

rule c {
    condition: false
}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    yaramod = y.Yaramod()

    yara_src = yaramod.parse_file(TMP_FILENAMES[0])
    assert(len(yara_src.entry_file.rules) == 3)
    assert(yara_src.entry_file.rules[2].id == "c")

    yara_src = yaramod.parse_file(TMP_FILENAMES[0], yara_src)
    assert(len(yara_src.entry_file.rules) == 3)
    assert(yara_src.entry_file.rules[2].id == "c")


def test_not_fixed_error():
    """Tests of reparsing file that contains error, that is not fixed by edit"""

    yara_string = """
rule a { 
    condition: true
}

rule b a {
    condition: true
}

rule c {
    condition: false
}
    """


    yara_string_edited = """
rule a { 
    condition: f
}

rule b a {
    condition: true
}

rule c {
    condition: false
}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    yaramod = y.Yaramod()

    yara_src = yaramod.parse_file(TMP_FILENAMES[0])
    assert(len(yara_src.entry_file.syntax_errors) == 1)
    assert(yara_src.entry_file.syntax_errors[0].offset == 41)

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string_edited)
    file.close()

    yara_src.edit(y.Range(2, 15, 2, 19), "f")

    yara_src = yaramod.parse_file(TMP_FILENAMES[0], yara_src)

    assert(len(yara_src.entry_file.syntax_errors) == 1)
    assert(yara_src.entry_file.syntax_errors[0].offset == 38)

    assert(len(yara_src.entry_file.semantic_errors) == 1)
    assert(yara_src.entry_file.semantic_errors[0].offset == 26)



def test_error_fix():
    """Tests of reparsing file that contains error and it is fixed by edit"""

    yara_string = """
rule a { 
    condition: true
}

rule b  
    condition: true
}

rule c {
    condition: false
}
    """


    yara_string_edited = """
rule a { 
    condition: true
}

rule b {
    condition: true
}

rule c {
    condition: false
}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    yaramod = y.Yaramod()

    yara_src = yaramod.parse_file(TMP_FILENAMES[0])
    assert(len(yara_src.entry_file.syntax_errors) == 4)

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string_edited)
    file.close()

    yara_src.edit(y.Range(5, 7, 5, 8), "{")

    yara_src = yaramod.parse_file(TMP_FILENAMES[0], yara_src)
    assert(len(yara_src.entry_file.syntax_errors) == 0)



def test_rule_insert_to_start():
    """Tests of reparsing file in which was added the new rule"""

    yara_string = """


rule b {
    condition: true
}

rule c {
    condition: false
}
    """


    yara_string_edited = """
rule new_malware { condition: true }

rule b {
    condition: true
}

rule c {
    condition: false
}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    yaramod = y.Yaramod()

    yara_src = yaramod.parse_file(TMP_FILENAMES[0])
    assert(len(yara_src.entry_file.syntax_errors) == 0)
    assert(len(yara_src.entry_file.semantic_errors) == 0)

    assert(yara_src.entry_file.local_rules[0].id == "b")
    assert(len(yara_src.entry_file.local_rules) == 2)

    # for r in yara_src.entry_file.local_rules:
    #     print(f"RULE {r.id}  {r.offset}   {r.len}")

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string_edited)
    file.close()

    yara_src.edit(y.Range(1, 0, 1, 0), "rule new_malware { condition: true }")

    yara_src = yaramod.parse_file(TMP_FILENAMES[0], yara_src)
    assert(len(yara_src.entry_file.syntax_errors) == 0)
    assert(len(yara_src.entry_file.semantic_errors) == 0)

    assert(yara_src.entry_file.local_rules[0].id == "new_malware")
    assert(len(yara_src.entry_file.local_rules) == 3)



def test_file_include_insert():
    """Tests of reparsing file in which was inserted the new file include"""

    yara_string = """


rule b {
    condition: true
}

rule c {
    condition: false
}
    """

    yara_string_edited = f"""
include "{TMP_FILENAMES[1]}"

rule b {{
    condition: true
}}

rule c {{
    condition: false
}}
    """

    included_string = """
rule d {
    condition: false
}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    yaramod = y.Yaramod()

    yara_src = yaramod.parse_file(TMP_FILENAMES[0])
    assert(len(yara_src.files) == 1)

    file = open(TMP_FILENAMES[1], "w")
    file.write(included_string)
    file.close()

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string_edited)
    file.close()

    yara_src.edit(y.Range(y.Point(1, 0), y.Point(1, 0)), f"include \"{TMP_FILENAMES[1]}\"")

    yara_src = yaramod.parse_file(TMP_FILENAMES[0], yara_src)
    assert(len(yara_src.files) == 2)
    assert(yara_src.get_file(TMP_FILENAMES[1]).local_rules[0].id == "d")
    
    assert(yara_src.entry_file.rules[0].id == "d") # rules property has all files that can be accessed from file 




def test_undo():
    """Tests of Yaramod-v4 undo feature"""

    yara_string = """


rule b {
    condition: true
}

rule c {
    condition: false
}
    """

    yara_string_orig_edited = """


rule b {
    condition: true
}


    """

    yara_string_edited = """
rule new { 
    condition: false 
}

rule b {
    condition: true
}

rule c {
    condition: false
}
    """


    yaramod = y.Yaramod()

    yara_src = yaramod.parse_string(yara_string)

    assert(len(yara_src.entry_file.local_rules) == 2)

    yara_src.edit(y.Range(7, 0, 9, 2) , "") # Removing rule c
    yaramod.parse_string(yara_string_orig_edited, yara_src)

    assert(len(yara_src.entry_file.local_rules) == 1)

    yara_src = yaramod.parse_string(yara_string)
    
    yara_src.edit(y.Range(7, 0, 9, 2) , "") # Removing rule

    yara_src.undo() # I changed my mind

    yara_src.edit(y.Range(1, 0, 1, 0) , """rule new { 
    condition: false 
}""") # I want to add the rule "new"
                  
    yaramod.parse_string(yara_string_edited, yara_src)
                  
    assert(len(yara_src.entry_file.local_rules) == 3)
    assert(yara_src.entry_file.local_rules[0].id == "new")




def test_cyclic_file_include():
    """Tests of reparsing file in which cyclic file inclusion occurs"""

    yara_string = f"""
include "{TMP_FILENAMES[1]}"

rule z {{
    condition: true
}}

rule y {{
    condition: false
}}
    """

    yara_string_edited = f"""
include "{TMP_FILENAMES[1]}"

rule z {{
    condition: true
}}


    """


    yara_file_included = f"""
include "{TMP_FILENAMES[0]}"

rule b {{
    condition: true
}}

rule c {{
    condition: false
}}


    """

    yara_file_included_edited = f"""
include "{TMP_FILENAMES[0]}"

rule b {{
    condition: true
}}

rule c {{
    condition: false
}}

rule y {{
    condition: false
}}
    """

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string)
    file.close()

    file = open(TMP_FILENAMES[1], "w")
    file.write(yara_file_included)
    file.close()

    yaramod = y.Yaramod()

    file = open(TMP_FILENAMES[0], "r")
    file_content = file.read()
    yara_src = yaramod.parse_string(file_content, TMP_FILENAMES[0])
    assert(len(yara_src.files) == 2)

    assert(yara_src.get_file(TMP_FILENAMES[0]).local_rules[0].id == "z")
    assert(yara_src.get_file(TMP_FILENAMES[1]).local_rules[0].id == "b")

    yara_src.edit(y.Range(7, 0, 9, 2), "")
    yara_src.edit(y.Range(11, 0, 11, 0), f"""rule y {{
    condition: false
}}"""  , TMP_FILENAMES[1])

    file = open(TMP_FILENAMES[0], "w")
    file.write(yara_string_edited)
    file.close()

    file = open(TMP_FILENAMES[1], "w")
    file.write(yara_file_included_edited)
    file.close()

    file = open(TMP_FILENAMES[0], "r")
    file_content = file.read()
    yara_src = yaramod.parse_string(file_content, yara_src, TMP_FILENAMES[0])

    assert(yara_src.get_file(TMP_FILENAMES[0]).local_rules[0].id == "z")
    assert(len(yara_src.get_file(TMP_FILENAMES[0]).local_rules) == 1)

    assert(yara_src.get_file(TMP_FILENAMES[1]).local_rules[0].id == "b")
    assert(yara_src.get_file(TMP_FILENAMES[1]).local_rules[2].id == "y")
    assert(len(yara_src.get_file(TMP_FILENAMES[1]).local_rules) == 3)

