using Ethanol.MalwareSonar.Fuzzy;
/// <summary>
/// Represents a match between a host and a piece of malware, including detailed information about the match across various indicators.
/// </summary>
/// <param name="Host">The host involved in the malware match, such as a server or network device.</param>
/// <param name="Start">The start date and time of the period when the match was detected.</param>
/// <param name="End">The end date and time of the period when the match was detected.</param>
/// <param name="Malware">The name or identifier of the malware involved in the match.</param>
/// <param name="Urls">A FuzzySet of URLs associated with the malware match, indicating the potential relevance of each URL.</param>
/// <param name="Domains">A FuzzySet of domain names associated with the malware match, indicating the potential relevance of each domain.</param>
/// <param name="Ips">A FuzzySet of IP addresses associated with the malware match, indicating the potential relevance of each IP address.</param>
/// <param name="UrlsInitialThreshold">A tuple representing the mean and stdev for URL scores in the match.</param>
/// <param name="DomainsInitialThreshold">A tuple representing the the mean and stdev for domain scores in the match.</param>
/// <param name="IpsInitialThreshold">A tuple representing the the mean and stdev for IP scores in the match.</param>
/// <remarks>
/// This record is used to encapsulate all relevant data for a malware match, including time frame, involved host, and detailed indicators with their respective thresholds.
/// </remarks>
public record MalwareMatch(string Host, DateTimeOffset Start, DateTimeOffset End, string Malware, 
FuzzySet<string> Urls, FuzzySet<string> Domains, FuzzySet<string> Ips, 
MeanAndStdev UrlsInitialThreshold , MeanAndStdev DomainsInitialThreshold, MeanAndStdev IpsInitialThreshold)
{
    /// <summary>
    /// Gets the score for the match based on the IP addresses.
    /// </summary>
    public MalwareMatchScore ScoreIps =>
        new MalwareMatchScore(this.Host, this.Malware, this.Ips.GetMembershipValues().Sum(x=>x.Value), this.IpsInitialThreshold);

    /// <summary>
    /// Gets the score for the match based on the domains.
    /// </summary>
    public MalwareMatchScore ScoreDomains =>
        new MalwareMatchScore(this.Host, this.Malware, this.Domains.GetMembershipValues().Sum(x=>x.Value), this.DomainsInitialThreshold);

    /// <summary>
    /// Gets the score for the match based on the URLs.
    /// </summary>
    public MalwareMatchScore ScoreUrls =>
        new MalwareMatchScore(this.Host, this.Malware, this.Urls.GetMembershipValues().Sum(x=>x.Value),this.UrlsInitialThreshold);
}

/// <summary>
/// Represents a statistical record holding the mean (average) and standard deviation of a set of data.
/// </summary>
/// <param name="Mean">The mean (average) value of the data set. It represents the central or typical value in the set.</param>
/// <param name="Stdev">The standard deviation of the data set. This value indicates the amount of variation or dispersion in the set of values.</param>
/// <remarks>
/// This record is typically used in statistical analyses to summarize the central tendency and variability of a data set.
/// The mean provides a measure of the center of the data, while the standard deviation measures how spread out the values are around the mean.
/// </remarks>
public record MeanAndStdev(double Mean, double Stdev);

