/**
 * \file plane.h
 * \brief Reading and writing image files to/from framebuffer
 */
#ifndef __DENOISING_PLANE_H_
#define __DENOISING_PLANE_H_

#include <stddef.h>
#include "common.h"

/**
 * \brief Framebuffer holding either image or wavelet coefficients
 *
 * The \c data is a buffer having dimensions to be multiples of 2^levels.
 */
struct plane {
	size_t height; /**< \brief number of rows */
	size_t width;  /**< \brief number of columns */
	size_t stride_x, stride_y; /* = (1, width) by default */

	float *data;     /**< \brief framebuffer */
};

/**
 * \brief Debugging dump
 *
 * Writes the content of \p plane into PPM format.
 */
int plane_dump(const struct plane *plane, const char *path, float factor);

/**
 * \brief Release resources
 */
void plane_destroy(struct plane *plane);

/**
 * \brief Pixel-wise addition assignment
 */
int plane_add(struct plane *plane, const struct plane *rhs);

/**
 * \brief Multiply by constant and then clamp the pixels
 */
int plane_normalize(struct plane *plane, float a);

int plane_alloc_data(struct plane *plane);

/**
 * \brief Set all pixels to zero
 */
int plane_zero_data(struct plane *plane);

int plane_create(struct plane *dest, const struct plane *src);

int plane_madd(struct plane *dest, float a, const struct plane *src);

#endif /* __DENOISING_PLANE_H_ */
