#include "plane.h"
#include "common.h"
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>

int plane_alloc_data(struct plane *plane)
{
	size_t stride_y, height;
	float *data;

	assert(plane != NULL);

	stride_y = plane->stride_y;
	height = plane->height;

	data = malloc(height * stride_y * sizeof *data);

	if (NULL == data) {
		return RET_FAILURE_MEMORY_ALLOCATION;
	}

	plane->data = data;

	return RET_SUCCESS;
}

void plane_destroy(struct plane *plane)
{
	assert(plane != NULL);

	free(plane->data);

	plane->data = NULL;
}

int plane_dump(const struct plane *plane, const char *path, float factor)
{
	FILE *stream;
	size_t width, height, depth;
	size_t stride_x, stride_y;
	size_t bpp;
	size_t stride;
	size_t y, x;
	const float *data;
	int maxval;
	void *line;

	stream = fopen(path, "w");

	if (NULL == stream) {
		return RET_FAILURE_FILE_OPEN;
	}

	assert(plane != NULL);

	bpp = 8;

	maxval = (int) convert_bpp_to_maxval(bpp);

	width = plane->width;
	height = plane->height;
	stride_x = plane->stride_x;
	stride_y = plane->stride_y;

	if (width > ULONG_MAX || height > ULONG_MAX) {
		return RET_FAILURE_OVERFLOW_ERROR;
	}

	if (fprintf(stream, "P5\n%lu %lu\n%lu\n", (unsigned long) width, (unsigned long) height, (unsigned long) maxval) < 0) {
		return RET_FAILURE_FILE_IO;
	}

	data = plane->data;

	assert(data != NULL);
	assert(factor != 0);

	depth = convert_bpp_to_depth(bpp);
	stride = width * depth;
	line = malloc(stride);

	if (NULL == line) {
		return RET_FAILURE_MEMORY_ALLOCATION;
	}

	for (y = 0; y < height; ++y) {
		for (x = 0; x < width; ++x) {
			float sample = data[y * stride_y + x * stride_x];
			float magnitude;

			magnitude = fabsf(sample) / factor;

			switch (depth) {
				case sizeof(char): {
					unsigned char *line_ = line;

					line_ [x] = (unsigned char) clamp(int_roundf(magnitude * (float)(maxval + 1)), 0, maxval);

					break;
				}
				case sizeof(short): {
					unsigned short *line_ = line;

					line_ [x] = native_to_be_s((unsigned short) clamp(int_roundf(magnitude * (float)(maxval + 1)), 0, maxval));

					break;
				}
				default:
					dprint (("[ERROR] unhandled bit depth\n"));
					return RET_FAILURE_LOGIC_ERROR;
			}
		}

		if (1 != fwrite(line, stride, 1, stream)) {
			return RET_FAILURE_FILE_IO;
		}
	}

	free(line);

	if (EOF == fclose(stream)) {
		return RET_FAILURE_FILE_IO;
	}

	return RET_SUCCESS;
}
