#ifndef __DENOISING_COMMON_H
#define __DENOISING_COMMON_H

#include <stddef.h>
#include <limits.h>

/**
 * \brief Formatted output, write output to \c stderr
 */
int eprintf(const char *format, ...);

/**
 * \brief Debugging \c printf
 *
 * If the macro \c NDEBUG is defined, this macro generates no code, and hence
 * does nothing at all.
 */
#ifdef NDEBUG
#	define dprint(arg)
#else
#	define dprint(arg) eprintf arg
#endif

/**
 * \brief Error codes
 *
 * The standard (C89, 6.1.3.3 Enumeration constants) states that
 * an identifier declared as an enumeration constant has type \c int.
 * Therefore, it is fine if the function returning these constants
 * has return type \c int.
 */
enum {
	/* 0x0000 successful completion */
	RET_SUCCESS                   = 0x0000, /**< success */
	/* 0x1xxx input/output errors */
	RET_FAILURE_FILE_IO           = 0x1000, /**< I/O error */
	RET_FAILURE_FILE_UNSUPPORTED  = 0x1001, /**< unsupported feature or file type */
	RET_FAILURE_FILE_OPEN         = 0x1002, /**< file open failure */
	/* 0x2xxx memory errors */
	RET_FAILURE_MEMORY_ALLOCATION = 0x2000, /**< unable to allocate dynamic memory */
	/* 0x3xxx general exceptions */
	RET_FAILURE_LOGIC_ERROR       = 0x3000, /**< faulty logic within the program */
	RET_FAILURE_OVERFLOW_ERROR    = 0x3001, /**< result is too large for the destination type */
	RET_LAST
};

/**
 * \brief Find the smallest bit depth needed to hold the \p maxval
 */
size_t convert_maxval_to_bpp(unsigned long maxval);

/**
 * \brief Convert \p bpp into the number of bytes required by the smallest type that can hold the \p bpp bit samples
 *
 * If no suitable type can be found, returns zero.
 */
size_t convert_bpp_to_depth(size_t bpp);

/**
 * \brief Get the maximum value that can be represented on \p bpp bit word
 */
unsigned long convert_bpp_to_maxval(size_t bpp);

void set_block_size(size_t n);

size_t get_block_size();

/**
 * \brief Round \p n up to the nearest multiple of 8
 */
size_t ceil_multiple(size_t n);

float convert_mse_to_psnr(float mse, size_t bpp);

/**
 * \brief Round to nearest integer
 *
 * The result of this macro is similar to casting the result of roundf() to \c int.
 * The roundf() function is not present in C89.
 */
int int_roundf(float x);

/**
 * \brief Returns the value of \p v constrained to the range \p lo to \p hi
 */
int clamp(int v, int lo, int hi);

/**
 * \brief Smallest power of 2 greater than or equal to \p n
 */
size_t next_pow2(size_t n);

#ifndef SIZE_MAX
#	define SIZE_MAX ((size_t)-1)
#endif

#define PATH_LEN 4096

/**
 * \brief Indicates the layout of multi-byte integers
 *
 * The macro should be defined on little endian architectures.
 */
#define SWAP_BYTE_ORDER

/**
 * \brief Convert big-endian to native byte order
 *
 * This function is similar to htons(). However the htons() is not present in C89.
 */
__attribute__((unused))
static unsigned short be_to_native_s(unsigned short a)
{
#ifdef SWAP_BYTE_ORDER
	return (unsigned short) (
		((a & 0xff00U) >> 8U) |
		((a & 0x00ffU) << 8U)
	);
#else
	return a;
#endif
}

/**
 * \brief Convert native byte order to big endian
 *
 * This function is similar to ntohs(). However the ntohs() is not present in C89.
 */
__attribute__((unused))
static unsigned short native_to_be_s(unsigned short a)
{
#ifdef SWAP_BYTE_ORDER
	return (unsigned short) (
		((a & 0xff00U) >> 8U) |
		((a & 0x00ffU) << 8U)
	);
#else
	return a;
#endif
}

int is_even(ptrdiff_t n);

#endif /* __DENOISING_COMMON_H */
