#include "common.h"
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <libgen.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdlib.h>
#include <math.h>

int eprintf(const char *format, ...)
{
	va_list ap;
	int n;

	va_start(ap, format);
	n = vfprintf(stderr, format, ap);
	va_end(ap);

	return n;
}

/**
 * \brief Base-2 logarithm
 *
 * The result is undefined for zero \p n.
 */
static unsigned long floor_log2_l(unsigned long n)
{
	unsigned long r = 0;

	while (n >>= 1) {
		r++;
	}

	return r;
}

size_t convert_maxval_to_bpp(unsigned long maxval)
{
	if (maxval != 0) {
		return floor_log2_l(maxval) + 1;
	}

	return 0;
}

size_t convert_bpp_to_depth(size_t bpp)
{
	return bpp <= CHAR_BIT ? 1
		: ( bpp <= CHAR_BIT * sizeof(short) ? sizeof(short)
			: 0);
}

size_t g_block_size = 1;

void set_block_size(size_t n)
{
	assert(n > 0);

	g_block_size = n;
}

size_t get_block_size()
{
	return g_block_size;
}

size_t ceil_multiple(size_t n)
{
	return (n + (g_block_size - 1)) / g_block_size * g_block_size;
}

int is_even(ptrdiff_t n)
{
	/* size_t is unsigned integer type */
	return !((size_t) n & 1);
}

unsigned long convert_bpp_to_maxval(size_t bpp)
{
	if (bpp != 0) {
		return (1UL << bpp) - 1;
	}

	return 0;
}

float convert_mse_to_psnr(float mse, size_t bpp)
{
	unsigned long maxval = convert_bpp_to_maxval(bpp);

	return 10.f * log10f( (float) maxval * (float) maxval / mse );
}

int int_roundf(float x)
{
	int i;

	/* convert round to floor */
	x = x + .5f;

	/*
	 * per C89 standard, 6.2.1.3 Floating and integral:
	 *
	 * When a value of floating type is convened to integral type,
	 * the fractional part is discarded. If the value of the integral part
	 * cannot be represented by the integral type, the behavior is
	 * undetined.
	 *
	 * "... discarded", i.e., the value is truncated toward zero
	 */

	/* truncate */
	i = (int) x;

	/* convert trunc to floor */
	return i - (int) ( (float) i > x );
}

int clamp(int v, int lo, int hi)
{
	return v < lo ? lo : ( hi < v ? hi : v );
}

size_t next_pow2(size_t n)
{
	size_t r = 1;

	while (r < n) {
		r <<= 1;
	}

	return r;
}
