# time_wrapper.py: Converts various time representations to Unix timestamp.
# Copyright (C) 2017 Libor Polčák <ipolcak@fit.vutbr.cz>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import calendar
import copy
import time
from dateutil import parser

months = {
        "Jan": 1,
        "Feb": 2,
        "Mar": 3,
        "Apr": 4,
        "May": 5,
        "Jun": 6,
        "Jul": 7,
        "Aug": 8,
        "Sep": 9,
        "Oct": 10,
        "Nov": 11,
        "Dec": 12,
    }

DAY_HAS_SECONDS = 24*60*60

class TimeWrapperBase():
    """ Base abstract class.
    
    Expects that subclasses create v that stores time as Unix timestamp.
    """
    def get(self):
        return self.v

    def format(self, fmt):
        return time.strftime(fmt, time.gmtime(self.v))

    @staticmethod
    def __get_days_in_month(year, month):
        return calendar.monthrange(year, month)[1]

    def next_day(self):
        r = copy.copy(self)
        r.v += DAY_HAS_SECONDS
        return r

    def prev_day(self):
        r = copy.copy(self)
        r.v -= DAY_HAS_SECONDS
        return r

    def next_month(self):
        r = copy.copy(self)
        t = time.gmtime(self.v)
        days_orig_month = self.__get_days_in_month(t.tm_year, t.tm_mon)
        days_left_orig_month = days_orig_month - t.tm_mday
        days_after_this_month = days_orig_month - days_left_orig_month
        if t.tm_mon == 12:
            next_year = t.tm_year + 1
            next_mon = 1
        else:
            next_year = t.tm_year
            next_mon = t.tm_mon + 1
        days_next_month = self.__get_days_in_month(next_year, next_mon)
        if days_next_month < days_after_this_month:
            add_days = days_left_orig_month + days_next_month
        else:
            add_days = days_orig_month
        r.v += add_days * DAY_HAS_SECONDS
        return r

    def prev_month(self):
        r = copy.copy(self)
        t = time.gmtime(self.v)
        if t.tm_mon == 1:
            year = t.tm_year - 1
            mon = 12
        else:
            year = t.tm_year
            mon = t.tm_mon - 1
        days_prev_month = self.__get_days_in_month(year, mon)
        sub_day = max(days_prev_month, t.tm_mday)
        r.v -= sub_day * DAY_HAS_SECONDS
        return r

    def __str__(self):
        return time.strftime("%Y-%m-%d %H:%M:%S", time.gmtime(self.v))

class TimeWrapper(TimeWrapperBase):
    """ Converts various string representation of time to Unix timestamp. """

    def __init__(self, s):
        try:
            self.v = float(s)
        except ValueError:
            self.v = None
        except TypeError:
            s = " ".join(s) # The argument might be an iterable of strings
            self.v = None
        if not self.v:
            t = None
            try:
                # See https://github.com/dateutil/dateutil/issues/94 for the explanation of the
                # default value
                t = parser.parse(s, default=parser.parse("00:00Z"))
            except:
                try:
                    t = parser.parse(s.replace(":", "T", 1), default=parser.parse("00:00Z"))
                except:
                    pass
            if t == None:
                raise ValueError("%s is not a supported time format" % s)
            self.v = t.timestamp()

class TorTimeWrapper(TimeWrapperBase):
    """ Converts Tor timestamp to Unix timestamp. """

    def __init__(self, datestr, timestr):
        """ Overloaded constructor
        
        The datestr string is expected in format %Y-%m-%d.
        The timestr string is expected in format %H:%M:%S.

        Note that the super constructor is not called at all.
        """
        self.v = calendar.timegm(time.strptime("%s %s" % (datestr, timestr), "%Y-%m-%d %H:%M:%S"))

class FormatTimeWrapper(TimeWrapperBase):
    """ Converts timestamp with given format to Unix timestamp. """

    def __init__(self, timestr, formatstr):
        """ Overloaded constructor
        
        timestr string with the time data.
        format specifies the format of the timestr (see Python time modul doc for details).

        Note that the super constructor is not called at all.
        """
        self.v = calendar.timegm(time.strptime(timestr, formatstr))
