#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for functions concerning file oprations """
# Copyright (C) 2016 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import parametrizable_tc as ptc

from consensus_parser import create_ip_filename
import ipfiles

class test_files(unittest.TestCase, metaclass=ptc.parametrizable_tc_meta):

    def scenario_create_ip_filename(self, path_no_slash, d1, d2, ipaddr):
        self.assertEqual(create_ip_filename(path_no_slash, ipaddr),
                "%s/%s/%s/%s" % (path_no_slash, d1, d2, ipaddr),
                "Testing addr %s" % ipaddr)
        self.assertEqual(create_ip_filename(path_no_slash + "/", ipaddr),
                "%s//%s/%s/%s" % (path_no_slash, d1, d2, ipaddr),
                "Testing addr %s" % ipaddr)

    def scenario_abspath(self, d1, d2, ipaddr):
        self.scenario_create_ip_filename("/", d1, d2, ipaddr)

    def scenario_abspath_long(self, d1, d2, ipaddr):
        self.scenario_create_ip_filename("/abc/fds/fds/", d1, d2, ipaddr)

    def scenario_relpath(self, d1, d2, ipaddr):
        self.scenario_create_ip_filename("..", d1, d2, ipaddr)

    def scenario_relpath_long(self, d1, d2, ipaddr):
        self.scenario_create_ip_filename("../fdas/fasdfa/fds", d1, d2, ipaddr)

    def scenario_addr(self, d1, d2, ipaddr):
        self.scenario_abspath(d1, d2, ipaddr)
        self.scenario_abspath_long(d1, d2, ipaddr)
        self.scenario_relpath(d1, d2, ipaddr)
        self.scenario_relpath_long(d1, d2, ipaddr)

    @ptc.parametrizable_test([
        ("global", ["1.2.3.4"]),
        ("private10", ["10.10.10.10"]),
        ("localhost", ["127.0.0.1"]),
        ("global_vut", ["147.229.1.1"]),
        ("private192", ["192.168.0.1"]),
        ])
    def test_create_ip_filename_ipv4(self, ipaddr):
        b = ipaddr.split(".")
        self.scenario_addr(b[0], b[1], ipaddr)

    @ptc.parametrizable_test([
        ("example128", ("2001", "db8", "2001:db8:1234:5678:aaaa:bbbb:cccc:dddd")),
        ("example32", ("2001", "db8", "2001:db8::")),
        ("2001_8", ("2001", "0000", "2001::")),
        ("2001_32", ("2001", "0000", "2001::a")),
        ("localhost", ("0000", "0000", "::1")),
        ("0_1_content", ("0000", "1", ":1::a")),
        ])
    def test_create_ip_filename_ipv6(self, *args):
        self.scenario_addr(*args)

    @ptc.parametrizable_test([
        ("top", ["0.0.0.0", 0, ['67.0.0.0/8', "147.0.0.0/8"]]),
        ("128", ["128.0.0.0", 1, ["147.0.0.0/8"]]),
        ("128none", ["128.0.0.0", 4, []]),
        ("147-8", ["147.0.0.0", 8, ["147.229.0.0/16"]]),
        ("147-9", ["147.0.0.0", 9, []]),
        ("147.128-9", ["147.128.0.0", 9, ["147.229.0.0/16"]]),
        ("147.229-16", ["147.229.0.0", 16, ["147.229.13.223"]]),
        ("147.229.13-23", ["147.229.12.0", 23, ["147.229.13.223"]]),
        ("dirnotexist", ["127.0.0.0", 16, []]),
        ])
    def test_get_ranges_addresses_ipv4(self, network_addr, plen, expected):
        self.assertEqual(ipfiles.get_ranges_addresses_ipv4("testfiles/preprocessed",
            network_addr, plen), expected)
        self.assertEqual(ipfiles.get_ranges_addresses("testfiles/preprocessed",
            network_addr, plen), expected)

    @ptc.parametrizable_test([
        ("top", ["::", 0, ['67::/16', "147::/16", "2001::/16"]]),
        ("2000_3", ["2000::", 3, ["2001::/16"]]),
        ("147_16", ["147::", 16, ["147:229::/32"]]),
        ("147_229_32", ["147:229::", 32, []]),
        ("2001_16", ["2001::", 16, ["2001:67c::/32"]]),
        ("2001_4000_20", ["2001:4000::", 20, []]),
        ("2001_600_24", ["2001:600::", 24, ["2001:67c::/32"]]),
        ("2001_67c_32", ["2001:67c::", 32, ["2001:67c:15e4:5be:216:3eff:fe15:74cb"]]),
        ])
    def test_get_ranges_addresses_ipv6(self, network_addr, plen, expected):
        self.assertEqual(ipfiles.get_ranges_addresses_ipv6("testfiles/preprocessed",
            network_addr, plen), expected)
        self.assertEqual(ipfiles.get_ranges_addresses("testfiles/preprocessed",
            network_addr, plen), expected)

def suite():
        test = unittest.makeSuite(test_files, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
