# Copyright (C) 2018 Libor Polčák <ipolcak@fit.vutbr.cz>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import collections
import flask
import functools
import json

import consensus_parser as cp
import ipfiles
import geolite2 as gl
from serializable import serializable_links, serializable_ors

from time_parser import FormatTimeWrapper, TimeWrapper

app = flask.Flask(__name__)

PREPORCESSED_DIR = "/mnt/data/preprocessedip/"
GEOLITE_DIR = "/mnt/data/preprocessed_geolite/"

def request_wants_json():
    """ Check if we should return json

    Copied from http://flask.pocoo.org/snippets/45/
    """
    best = flask.request.accept_mimetypes \
        .best_match(['application/json', 'text/html'])
    return best == 'application/json' and \
        flask.request.accept_mimetypes[best] > \
        flask.request.accept_mimetypes['text/html']

def htmlize(result, related, func):
    res = ["<!DOCTYPE html>",
            "<html lang='en'>",
            "    <head>",
            "        <title>Toreator REST API - %s</title>" % func.__name__.replace("_", " ").capitalize(),
            "    </head>",
            "    <body>",
        ]
    if result:
        res.append("<main>")
        res.append("<h1>Result</h1>")
        res.extend(result.htmlize_lines())
        res.append("</main>")
    if related:
        res.append("<nav>")
        res.append("<h1>Related resources</h1>")
        res.extend(related.htmlize_lines())
        res.append("</nav>")
    res.append("</body>")
    res.append("</html>")
    return "\n".join(res)

def evaluate_accept_header_convert_json(func):
    @functools.wraps(func)
    def decorated_function(*args, **kwargs):
        result, related = func(*args, **kwargs)
        resdict = collections.OrderedDict()
        if request_wants_json():
            if result:
                resdict["result"] = result.jsonify()
            if related:
                resdict["related"] = related.jsonify()
            return (json.dumps(resdict, indent=0), \
                    {'Content-Type': 'application/json; charset=utf-8'})
        else:
            return htmlize(result, related, func)
    return decorated_function

@app.route('/')
@evaluate_accept_header_convert_json
def entry_point():
    related = serializable_links((
        ("Search by IP addresses", "/addresses/"),
        ))
    return None, related

@app.route('/addresses/')
@evaluate_accept_header_convert_json
def show_known_address_ranges():
    result = enum_addresses_generic("0.0.0.0", 0)
    result.extend(enum_addresses_generic("2000::", 3))
    related = serializable_links([("Search for IPv4 addresses", "/addresses/0.0.0.0-0/"),
            ("Search for IPv6 addresses", "/addresses/2000::-3/")])
    return result, related

@app.route('/addresses/<address_detail>/')
def process_route_addresses(address_detail):
    if "-" in address_detail:
        tokens = address_detail.split("-")
        return show_known_address_subranges(tokens[0], int(tokens[1]))
    else:
        return show_all_ip_address_information(address_detail)

@evaluate_accept_header_convert_json
def show_all_ip_address_information(address):
    result = serializable_ors(
                cp.merge_subsequent_ors(
                    cp.find_preprocessed_ip_address(address, \
                        PREPORCESSED_DIR, gl.geolite2_accessor(GEOLITE_DIR))
                ), "/addresses", address, "date", "time"
            )
    related = serializable_links((
        ("Search by IP addresses", "/addresses/"),
        ("Show dates when the IP address %s was active" % address, "/addresses/%s/date/" % address),
        ("Show months when the IP address %s was active" % address, "/addresses/%s/month/" % address),
        ("Show years when the IP address %s was active" % address, "/addresses/%s/year/" % address),
        ))
    return result, related

@app.route('/addresses/<addr>/date/')
@evaluate_accept_header_convert_json
def show_ip_address_activity_dates(addr):
    reslist = cp.get_ip_address_activity(addr, PREPORCESSED_DIR)
    result = serializable_links([
            ("%s activity at %s" % (addr, date), "/addresses/%s/date/%s/" % (addr, date))
            for date in reslist])
    related = serializable_links((
        ("Search by IP addresses", "/addresses/"),
        ("All information about IP address", "/addresses/%s/" % addr),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ))
    return result, related

@app.route('/addresses/<addr>/date/<date>/')
@evaluate_accept_header_convert_json
def show_ip_address_activity_on_specific_date(addr, date):
    today = FormatTimeWrapper(date, "%Y-%m-%d")
    next = today.next_day().format("%Y-%m-%d")
    prev = today.prev_day().format("%Y-%m-%d")
    result = serializable_ors(
                cp.merge_subsequent_ors(
                    cp.find_preprocessed_ip_address_time_filter(addr, \
                        PREPORCESSED_DIR, None, date, \
                        gl.geolite2_accessor(GEOLITE_DIR))
                ), "/addresses", addr, "date", "time"
            )
    related = serializable_links([
        ("Show the IP address %s activity on %s" % (addr, prev),
            "/addresses/%s/date/%s/" % (addr, prev)),
        ("Show the IP address %s activity on %s" % (addr, next),
            "/addresses/%s/date/%s/" % (addr, next)),
        ("Show the IP address %s activity in %s" % (addr, date[:-3]),
            "/addresses/%s/month/%s/" % (addr, date[:-3])),
        ("Show the IP address %s activity in %s" % (addr, date[:-6]),
            "/addresses/%s/year/%s/" % (addr, date[:-6])),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ("Show all information about the IP address %s" % addr, "/addresses/%s/" % addr),
        ("Search by IP addresses", "/addresses/"),
        ])
    return result, related

@app.route('/addresses/<addr>/time/')
def show_ip_address_activity_times(addr):
    return flask.redirect(flask.url_for('show_ip_address_activity_dates', addr=addr), code = 301)

@app.route('/addresses/<addr>/time/<time>/')
@evaluate_accept_header_convert_json
def show_ip_address_activity_at_specific_time(addr, time):
    t = TimeWrapper(time)
    today = t.format("%Y-%m-%d")
    month = t.format("%Y-%m")
    year = t.format("%Y")
    result = serializable_ors(
                cp.merge_subsequent_ors(
                    cp.find_preprocessed_ip_address_time_filter(addr, \
                        PREPORCESSED_DIR, t, None, \
                        gl.geolite2_accessor(GEOLITE_DIR))
                ), "/addresses", addr, "date", "time"
            )
    related = serializable_links([
        ("Show the IP address %s activity on %s" % (addr, today),
            "/addresses/%s/date/%s/" % (addr, today)),
        ("Show the IP address %s activity in %s" % (addr, month),
            "/addresses/%s/month/%s/" % (addr, month)),
        ("Show the IP address %s activity in %s" % (addr, year),
            "/addresses/%s/year/%s/" % (addr, year)),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ("Show all information about the IP address %s" % addr, "/addresses/%s/" % addr),
        ("Search by IP addresses", "/addresses/"),
        ])
    return result, related

@app.route('/addresses/<addr>/month/')
@evaluate_accept_header_convert_json
def show_ip_address_active_months(addr):
    reslist = cp.get_ip_address_activity(addr, PREPORCESSED_DIR, "%Y-%m")
    result = serializable_links([
            ("%s activity during %s" % (addr, month), "/addresses/%s/month/%s/" % (addr, month))
            for month in reslist])
    related = serializable_links([
        ("Search by IP addresses", "/addresses/"),
        ("All information about IP address", "/addresses/%s/" % addr),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ])
    return result, related

@app.route('/addresses/<addr>/month/<month>/')
@evaluate_accept_header_convert_json
def show_ip_address_activity_during_specific_month(addr, month):
    today = FormatTimeWrapper(month, "%Y-%m")
    next = today.next_month().format("%Y-%m")
    prev = today.prev_month().format("%Y-%m")
    result = serializable_ors(
                cp.merge_subsequent_ors(
                    cp.find_preprocessed_ip_address_time_filter(addr, \
                        PREPORCESSED_DIR, None, month, \
                        gl.geolite2_accessor(GEOLITE_DIR))
                ), "/addresses", addr, "date", "time"
            )
    related = serializable_links([
        ("Show the IP address %s activity in %s" % (addr, prev),
            "/addresses/%s/month/%s/" % (addr, prev)),
        ("Show the IP address %s activity in %s" % (addr, next),
            "/addresses/%s/month/%s/" % (addr, next)),
        ("Show the IP address %s activity in %s" % (addr, month[:-3]),
            "/addresses/%s/year/%s/" % (addr, month[:-3])),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ("Show all information about the IP address %s" % addr, "/addresses/%s/" % addr),
        ("Search by IP addresses", "/addresses/"),
        ])
    return result, related

@app.route('/addresses/<addr>/year/')
@evaluate_accept_header_convert_json
def show_ip_address_active_years(addr):
    reslist = cp.get_ip_address_activity(addr, PREPORCESSED_DIR, "%Y")
    result = serializable_links([
            ("%s activity in %s" % (addr, year), "/addresses/%s/year/%s/" % (addr, year))
            for year in reslist])
    related = serializable_links([
        ("Search by IP addresses", "/addresses/"),
        ("All information about IP address", "/addresses/%s/" % addr),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ])
    return result, related

@app.route('/addresses/<addr>/year/<int:year>/')
@evaluate_accept_header_convert_json
def show_ip_address_activity_during_specific_year(addr, year):
    next = year + 1
    prev = year - 1
    result = serializable_ors(
                cp.merge_subsequent_ors(
                    cp.find_preprocessed_ip_address_time_filter(addr, \
                        PREPORCESSED_DIR, None, str(year), \
                        gl.geolite2_accessor(GEOLITE_DIR))
                ), "/addresses", addr, "date", "time"
            )
    related = serializable_links([
        ("Show the IP address %s activity in %d" % (addr, prev),
            "/addresses/%s/year/%d/" % (addr, prev)),
        ("Show the IP address %s activity in %d" % (addr, next),
            "/addresses/%s/year/%d/" % (addr, next)),
        ("Show years when the IP address %s was active" % addr, "/addresses/%s/year/" % addr),
        ("Show months when the IP address %s was active" % addr, "/addresses/%s/month/" % addr),
        ("Show dates when the IP address %s was active" % addr, "/addresses/%s/date/" % addr),
        ("Show all information about the IP address %s" % addr, "/addresses/%s/" % addr),
        ("Search by IP addresses", "/addresses/"),
        ])
    return result, related

def enum_addresses_generic(netaddr, plen):
    result = serializable_links([])
    ranges = ipfiles.get_ranges_addresses(PREPORCESSED_DIR, netaddr, plen)
    for r in ranges:
        if "/" in r:
            result.append((r, "/addresses/%s-%s/" % tuple(r.split("/"))))
        else:
            result.append((r, "/addresses/%s/" % r))
    return result

@evaluate_accept_header_convert_json
def show_known_address_subranges(netaddr, plen):
    result = enum_addresses_generic(netaddr, plen)
    related = serializable_links([("Search by IP addresses", "/addresses/")])
    return result, related
