#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for ORouters """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
from io import StringIO

import consensus_parser as cp

class test_or(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.__or = cp.parse_ors_from_file("testfiles/preprocessed/2001/67c/2001:67c:15e4:5be:216:3eff:fe15:74cb")[0]

    def test_get_nickname(self):
        self.assertEqual(self.__or.get_nickname(), "GWMRelay")

    def test_get_identity(self):
        self.assertEqual(self.__or.get_identity(), "jnOKu1gT/J6xioM7uE74ID0STFo")

    def test_get_digest(self):
        self.assertEqual(self.__or.get_digest(), "nff8qIHEDx0ZxyCz2xrUbPhS9Tk")

    def test_get_publication(self):
        self.assertEqual(self.__or.get_publication(), ("2018-04-21", "15:11:45"))

    def test_get_ip(self):
        self.assertEqual(self.__or.get_ip(), "31.131.23.186")

    def test_get_orport(self):
        self.assertEqual(self.__or.get_orport(), "443")

    def test_get_dirport(self):
        self.assertEqual(self.__or.get_dirport(), "0")

    def test_get_ipv6(self):
        self.assertEqual(self.__or.get_ipv6(), [("2001:67c:15e4:5be:216:3eff:fe15:74cb", "9001")])

    def test_get_flags(self):
        self.assertEqual(self.__or.get_flags(), ("Running", "V2Dir", "Valid"))

    def test_get_version(self):
        self.assertEqual(self.__or.get_version(), "Tor 0.3.2.10")

    def test_get_supported_proto(self):
        self.assertEqual(self.__or.get_supported_proto(),
                ("Cons=1-2", "Desc=1-2", "DirCache=1-2", "HSDir=1-2", "HSIntro=3-4",
                "HSRend=1-2", "Link=1-5", "LinkAuth=1,3", "Microdesc=1-2", "Relay=1-2"))

    def test_get_bandwidth(self):
        self.assertEqual(self.__or.get_bandwidth(), {'Unmeasured': '1', 'Bandwidth': '0'})

    def test_get_allow_ports(self):
        self.assertEqual(self.__or.get_allow_ports(), "reject 1-65535")

    def test_get_inconsensus_val_after(self):
        self.assertEqual(self.__or.get_inconsensus_val_after(), ("2018-04-21", "16:00:00"))

    def test_get_inconsensus_fresh_until(self):
        self.assertEqual(self.__or.get_inconsensus_fresh_until(), ("2018-04-21", "17:00:00"))

    def test_get_inconsensus_val_until(self):
        self.assertEqual(self.__or.get_inconsensus_val_until(), ("2018-04-21", "19:00:00"))

    def test_get_json_dict(self):
        self.assertEqual(self.__or.get_json_dict(),
            {'allow_ports': 'reject 1-65535',
              'bandwidth': {'Bandwidth': '0', 'Unmeasured': '1'},
              'digest': 'nff8qIHEDx0ZxyCz2xrUbPhS9Tk',
              'dirport': '0',
              'flags': ('Running', 'V2Dir', 'Valid'),
              'identity': 'jnOKu1gT/J6xioM7uE74ID0STFo',
              'inconsensus_fresh_until': ('2018-04-21', '17:00:00'),
              'inconsensus_val_after': ('2018-04-21', '16:00:00'),
              'inconsensus_val_until': ('2018-04-21', '19:00:00'),
              'ip': '31.131.23.186',
              'ipv6': [('2001:67c:15e4:5be:216:3eff:fe15:74cb', '9001')],
              'nickname': 'GWMRelay',
              'orport': '443',
              'publication': ('2018-04-21', '15:11:45'),
              'supported_proto': ('Cons=1-2',
                                  'Desc=1-2',
                                  'DirCache=1-2',
                                  'HSDir=1-2',
                                  'HSIntro=3-4',
                                  'HSRend=1-2',
                                  'Link=1-5',
                                  'LinkAuth=1,3',
                                  'Microdesc=1-2',
                                  'Relay=1-2'),
              'version': 'Tor 0.3.2.10'})

    def test_output_json(self):
        fake_file = StringIO()
        self.__or.output_json(fake_file)
        fake_file.seek(0)
        self.assertEqual(fake_file.read(), '''{
    "allow_ports": "reject 1-65535",
    "bandwidth": {
        "Bandwidth": "0",
        "Unmeasured": "1"
    },
    "digest": "nff8qIHEDx0ZxyCz2xrUbPhS9Tk",
    "dirport": "0",
    "flags": [
        "Running",
        "V2Dir",
        "Valid"
    ],
    "identity": "jnOKu1gT/J6xioM7uE74ID0STFo",
    "inconsensus_fresh_until": [
        "2018-04-21",
        "17:00:00"
    ],
    "inconsensus_val_after": [
        "2018-04-21",
        "16:00:00"
    ],
    "inconsensus_val_until": [
        "2018-04-21",
        "19:00:00"
    ],
    "ip": "31.131.23.186",
    "ipv6": [
        [
            "2001:67c:15e4:5be:216:3eff:fe15:74cb",
            "9001"
        ]
    ],
    "nickname": "GWMRelay",
    "orport": "443",
    "publication": [
        "2018-04-21",
        "15:11:45"
    ],
    "supported_proto": [
        "Cons=1-2",
        "Desc=1-2",
        "DirCache=1-2",
        "HSDir=1-2",
        "HSIntro=3-4",
        "HSRend=1-2",
        "Link=1-5",
        "LinkAuth=1,3",
        "Microdesc=1-2",
        "Relay=1-2"
    ],
    "version": "Tor 0.3.2.10"
}
''')

def suite():
        test = unittest.makeSuite(test_or, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
