#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for geolite2 related functionality """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import parametrizable_tc as ptc
from time_parser import FormatTimeWrapper

from geolite2 import geolite2_accessor

t20180605 = FormatTimeWrapper("20180605", "%Y%m%d").get()
t20180619 = FormatTimeWrapper("20180619", "%Y%m%d").get()
t20180626 = FormatTimeWrapper("20180626", "%Y%m%d").get()
t20180703 = FormatTimeWrapper("20180703", "%Y%m%d").get()
t20180709 = FormatTimeWrapper("20180709", "%Y%m%d").get()

class test_geolite2(unittest.TestCase, metaclass=ptc.parametrizable_tc_meta):

    def setUp(self):
        self.maxDiff = None
        self.db = geolite2_accessor("testfiles/geolite2")
        self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["City"].cache), 0)
        self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["ASN"].cache), 0)

    def test_empty(self):
        self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["ASN"].dirs), 0)
        self.assertEqual(self.db._geolite2_accessor__geolite2_dbs["ASN"].last, 0)
        self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["City"].dirs), 0)
        self.assertEqual(self.db._geolite2_accessor__geolite2_dbs["City"].last, 0)

    def test_get_data_asn_firstrow_firstdb(self):
        ipaddr = "1.0.0.1"
        ts = t20180619
        for i in range(2): # Uncached and cached
            self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["ASN"].cache), i)
            ret = self.db.get_data("1.0.0.1", "ASN", t20180619, t20180619)
            self.assertEqual(len(ret), 1)
            t, d = ret[0]
            self.assertEqual(t, t20180619)
            self.assertEqual(d["AS network"], "1.0.0.0/24")
            self.assertEqual(d["AS number"], "13335")
            self.assertEqual(d["AS organization"], "Cloudflare Inc")

    def test_get_data_city_firstrow_firstdb(self):
        ipaddr = "1.0.0.1"
        ts = t20180619
        for i in range(2): # Uncached and cached
            self.assertEqual(len(self.db._geolite2_accessor__geolite2_dbs["City"].cache), i)
            ret = self.db.get_data("1.0.0.1", "City", t20180605, t20180605)
            self.assertEqual(len(ret), 1)
            t, d = ret[0]
            self.assertEqual(t, t20180605)
            self.assertEqual(d["network"], "1.0.0.0/24")
            self.assertEqual(d["continent"], "Oceania")
            self.assertEqual(d["country code"], "AU")
            self.assertEqual(d["country"], "Australia")
            self.assertEqual(d["country part"], "Victoria")
            self.assertEqual(d["city"], "Fountain Gate")
            self.assertEqual(d["time zone"], "Australia/Melbourne")

    def test_get_data_city_noneu_database(self):
        self.db = geolite2_accessor("testfiles/geolite2-noeu")
        ret = self.db.get_data("46.35.84.0", "City", 0, t20180605)
        self.assertEqual(len(ret), 1)
        t, d = ret[0]
        self.assertEqual(t, 1483401600)
        self.assertEqual(d["network"], "46.35.84.0/23")
        self.assertEqual(d["continent"], "Asia")
        self.assertEqual(d["country code"], "YE")
        self.assertEqual(d["country"], "Yemen")
        self.assertEqual(d["country part"], "Dhamār")
        self.assertEqual(d["city"], "Ta`izz")
        self.assertEqual(d["time zone"], "Asia/Aden")

    @ptc.parametrizable_test([
                ("asn_exact_time_single", ["ASN", t20180626, t20180626,
                    [(t20180626, "GeoLite2-ASN-CSV_20180626")]]),
                ("asn_just_after_two", ["ASN", t20180619+1, t20180619+10,
                    [(t20180619, "GeoLite2-ASN-CSV_20180619"),
                    (t20180626, "GeoLite2-ASN-CSV_20180626"),
                    ]]),
                ("asn_before_and_stop_single", ["ASN", t20180619-10, t20180619,
                    [(t20180619, "GeoLite2-ASN-CSV_20180619")]]),
                ("asn_before_and_after_two", ["ASN", t20180619-10, t20180619+10,
                    [(t20180619, "GeoLite2-ASN-CSV_20180619"),
                    (t20180626, "GeoLite2-ASN-CSV_20180626"),
                    ]]),
                ("asn_before_and_all", ["ASN", t20180619-10, t20180709,
                    [(t20180619, "GeoLite2-ASN-CSV_20180619"),
                    (t20180626, "GeoLite2-ASN-CSV_20180626"),
                    (t20180703, "GeoLite2-ASN-CSV_20180703"),
                    (t20180709, "GeoLite2-ASN-CSV_20180709"),
                    ]]),
                ("asn_not_first", ["ASN", t20180619+10, t20180709,
                    [(t20180619, "GeoLite2-ASN-CSV_20180619"),
                    (t20180626, "GeoLite2-ASN-CSV_20180626"),
                    (t20180703, "GeoLite2-ASN-CSV_20180703"),
                    (t20180709, "GeoLite2-ASN-CSV_20180709"),
                    ]]),
                ("city_before_and_after_two", ["City", t20180605-10, t20180605+10,
                    [(t20180605, "GeoLite2-City-CSV_20180605"),
                    (t20180703, "GeoLite2-City-CSV_20180703"),
                    ]]),
                ("city_no_record", ["City", 0, 10, []]),
            ])
    def test_get_directories(self, geolite2_db, start, end, expected):
        # Unbuffered
        self.assertEqual(self.db._geolite2_accessor__get_directories(
            geolite2_db, start, end), expected)
        # Check buffers
        self.assertTrue(len(self.db._geolite2_accessor__geolite2_dbs[geolite2_db].dirs) > 0)
        self.assertNotEqual(self.db._geolite2_accessor__geolite2_dbs[geolite2_db].last, 0)
        # Already buffered
        self.assertEqual(self.db._geolite2_accessor__get_directories(
            geolite2_db, start, end), expected)

    def test_geolite2_asn_ipv6(self):
        res = self.db.get_data("2a01:e35:8be7:65f0:43:7ff:fe82:ac61", "ASN", t20180626, t20180626)
        self.assertEqual(len(res), 1)
        t, d = res[0]
        self.assertEqual(t, t20180626)
        self.assertEqual(d["AS network"], "2a01:e00::/26")
        self.assertEqual(d["AS number"], "12322")
        self.assertEqual(d["AS organization"], "Free SAS")

    def test_geolite2_missing_data(self):
        self.db = geolite2_accessor("testfiles/geolite2-noeu")
        ret = self.db.get_data("14.192.209.152", "City", 0, t20180605)
        self.assertEqual(len(ret), 1)
        t, d = ret[0]
        self.assertEqual(d["network"], "14.192.209.152/29")

def suite():
        test = unittest.makeSuite(test_geolite2, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=True)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
