/******************************************************************************
* TASTE (Testability Analysis SuiTE) version 1.00
* Copyright (c) 2008 Josef Strnadel, all rights reserved
* Brno University of Technology, Faculty of Information Technology
* -----------------------------------------------------------------------------
* This is a free software: you can redistribute it and/or modify it 
* under the terms of the latest version of the GNU Lesser General Public License 
* as published by the Free Software Foundation.
* 
* This software is distributed in the hope that it will be useful, 
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
* See the GNU General Public License and the GNU Lesser General Public License 
* for more details.
* 
* You should have received a copy of the GNU General Public License
* and GNU Lesser General Public License along with this software. 
* If not, see <http://www.gnu.org/licenses/>.
* -----------------------------------------------------------------------------
* filename:     model.h
* version:      1.00
* started:      25. 4. 2007
* last revised: 21. 7. 2008
* language:     ANSI C++
* author:       Josef Strnadel
*               web:      http://www.fit.vutbr.cz/~strnadel/index.php.en
*               e-mail:   strnadel@fit.vutbr.cz
*               address:  Bozetechova 2, 61266 Brno, Czech Republic
*               phone:    +420 54114-1211
*               fax:      +420 54114-1270
* -----------------------------------------------------------------------------
* description:	circuit structure model
* -----------------------------------------------------------------------------
* notes:        -
* -----------------------------------------------------------------------------
* history of modifications:
*               -
* -----------------------------------------------------------------------------
* known bugs:   -
******************************************************************************/
//! \file model.h header of module with circuit structure model and functions

#ifndef _MODEL_H_
#define _MODEL_H_

#include <set>
#include <map>
#include <string>
#include <iostream>
#include <sstream>

#include "expressionevaluator/ExpressionEvaluator.h"
#include "fileio.h"
#include "misc.h"
#include "stringtokenizer/StringTokenizer.h"

using namespace std;

extern bool debug_mode;
extern unsigned long nerrors;

#include "defs.h"
#include "error.h"

/******************************************************************************
*	enum tmodeSel
******************************************************************************/
/*! \brief testability analysis mode selection. Possibilities are: controllability, observability and testability mode */
typedef enum tmodeSel {CON_MODE, OBS_MODE, TST_MODE};

/******************************************************************************
*	struct mTypeData - clsModuleType inner data
******************************************************************************/
/*! \brief transparency data block utilized by testability engine 
*/
/*! \a from_name represents source of transparency data-transport, \a to_name destination, 
 *  \a cond transport condition, \a composed is an inner-used composed name stored in 
 *  the form \a from_name|to_name|cond. \a orig flag is set to true for all original 
 *  data and to false for data added to original design.
 */
typedef struct mTypeData
{
  string from_name;  // from-bits
  string to_name;    // to-bits
  string cond;  // cond-bits
  string composed;
  bool orig;
} 
tMTypeData;

/*! \brief structure containing definition of '<' operator used to compare two tMTypeData structures. Operator is needed for placement of struct instances into STL containers like sets, maps etc. */
struct lt_mtdata
{
  bool operator()(const tMTypeData d1, const tMTypeData d2) const
  {
    return (d1.composed < d2.composed);
  }
};

/******************************************************************************
*	clsModuleType class - interface header
******************************************************************************/
/*! \class clsModuleType
 *  \brief object of the class is utilized to store information about templatized module types
 *  
 *  templatized module types are read from a library whose name can be specified
 *  by means of tool's command line parameters (see tool's help to \a -L command line parameter).
 *  Map of the class objects is common and stored globally (in clsDesign::mapModType) for all circuits within clsDesign scope.
 */
class clsModuleType
{
  // --- properties
  string name;      //!< module-type template name
  string area;      //!< module-type formula to be used for computing area of particular template instance
  string power;     //!< module-type formula to be used for computing power consumption of particular template instance
  set<tMTypeData, lt_mtdata> setOfTRANIS; //!< set of transparency data of the module-type
  set<string> setOfPorts; //!< set of port names of the module-type
  
	public:
	// --- constructor+destructor 
  clsModuleType(string nm){ name=nm; } //!< constructor
  ~clsModuleType(){;}                 //!< destructor
	// --- cointainer-related methods
	bool operator<(clsModuleType const& c) const {return(name < c.name);}  //!< definition of the '<' operator over class objects
	bool operator!=(clsModuleType c) const {return(name != c.name);}  //!< definition of the '!=' operator over class objects
	bool operator*(clsModuleType const& c) const {return(this);}  //!< definition of the '*' operator over class objects
	// --- data-access methods
	string getName() const {return(name);} //!< returns caller's name 
  bool addTranis(string s);              //!< adds transparency data to an object
  bool addPort(string s);                //!< adds port to an object
  void setArea(string s){area = s;}      //!< sets formula for area computation to an object
  string getArea(){return(area);}        //!< gets object's formula for area computation
  void setPower(string s){power = s;}    //!< sets formula for power-consumption computation to an object
  string getPower(){return(power);}      //!< gets object's formula for power-consumption 
  set<tMTypeData, lt_mtdata>::iterator getSTranisBegin(){return(setOfTRANIS.begin());}  //!< returns begining of \a setOfTRANIS set 
  set<tMTypeData, lt_mtdata>::iterator getSTranisEnd(){return(setOfTRANIS.end());}      //!< returns end of \a setOfTRANIS set
  set<string>::iterator getSPortsBegin(){return(setOfPorts.begin());}                   //!< returns begining of \a setOfPorts set
  set<string>::iterator getSPortsEnd(){return(setOfPorts.end());}                       //!< returns end of \a setOfPorts set
};

typedef map<string, clsModuleType>::value_type vt_moduletype;

/******************************************************************************
*	clsWire class - interface header
******************************************************************************/
/*! \class clsWire
 *  \brief object of the class is utilized to store information about single wire (physical connection) between two interface components
 *  
 *  wires are read from a file whose name must be specified by means of tool's command line parameters (see tool's help to \a -i command line parameter).
 *  Each circuit within the design is assigned an independent set of the class objects (stored in clsCircuit::setOfWires set).
 *  Wires are stored at bit-level, so information about interconnection is stored for each two bits which are connected. 
 */
class clsWire
{
  public:
  string from_module; //!< module the wire starts in (from-module)
  string from_port;   //!< port's module the wire starts in (from-port)
  int from_bit;       //!< port's bit the wire starts in (from-bit)
  string to_module;   //!< module the wire ends in (to-module)
  string to_port;     //!< port's module the wire ends in (to-port)
  int to_bit;         //!< port's bit the wire ends in (to-bit)

  string from_composed; //!< composed name in the form \a from_module.from_port.from_bit
  string to_composed;   //!< composed name in the form \a to_module.to_port.to_bit
  string composed;      //!< composed name in the form \a from_composed.to_composed

  bool orig;            //!< originality flag of the wire. Set to true for wires originally present in the design

	// --- constructor+destructor 
	/*!
	 *  @param[in] fm from-module
	 *  @param[in] fp from-port
	 *  @param[in] fb from-bit
	 *  @param[in] tm to-module
	 *  @param[in] tp to-port
	 *  @param[in] tb to-bit
	 *  @param[in] orig originality flag (implicitly set to true)
   */
  clsWire(string fm, string fp, int fb, string tm, string tp, int tb, bool orig=true); //!< constructor
	~clsWire(){};  //!< destructor
	
	// --- cointainer-related methods
	bool operator<(clsWire const& c) const {return(composed < c.composed);}  //!< definition of the '<' operator over class objects
	bool operator!=(clsWire const& c) const {return(composed != c.composed);}//!< definition of the '!=' operator over class objects
	bool operator*(clsWire const& c) const {return(this);}                   //!< definition of the '*' operator over class objects

  void setOrig(bool v) { orig = v; }    //!< sets originality flag of the object to \a v value
  bool getOrig() const { return(orig); }//!< gets value of object's originality flag 
};

/******************************************************************************
*	testability analysis related data-structures
******************************************************************************/
/*! \brief structure for storing data utilized for transporting accessibility marks during testability analysis process
*/
typedef struct ta_data 
{
  unsigned long nbitsC;  //!< nbits needed for control a node
  unsigned long nclksC; //!< nclks needed for control a node
  unsigned long nbitsO;  //!< nbits needed for observe a node
  unsigned long nclksO; //!< nclks needed for observe a node

  double multCprev; //!< multC of previous node (bit/port) in data-path
  double multCctrl; //!< multC of control-nodes
  double multOprev; //!< multO of previous node (bit/port) in data-path
  double multOctrl; //!< multO of control-nodes

  double con; //!< controllability value of a node
  double obs; //!< observability value of a node
  double tst; //!< testability value of a node
}
t_TAdata;

/******************************************************************************
*	clsBit class - interface header
******************************************************************************/
/*! \class clsBit
 *  \brief object of the class represents particular bit within module/circuit/design interface port
 *  
 *  bits are formed automatically after clsPort object is created. In clsPort object,
 *  bits are stored in map clsPort::mapBit. 
 */
class clsBit
{
  // --- properties
  int bit_number;     //!< bit number within port
  int node_type;      //!< node type (see \a enum \a tNode)
  bool orig;          //!< originality flag. Set to true for a bit originally present in the design

  t_TAdata ta_data;   // testability analysis related data

	public:
	// --- constructor+destructor 
	/*!
	 *  @param[in] bn bit number within the port
	 *  @param[in] nt bit type (see \a enum \a tNode)
	 *  @param[in] orig originality flag (implicitly set to true)
   */
  clsBit(int bn, int nt, bool orig=true); //! constructor
	~clsBit(); //! destructor
	
	// --- cointainer-related methods
	bool operator<(clsBit const& c) const {return(bit_number < c.bit_number);} //!< definition of the '<' operator over class objects
	bool operator!=(clsBit c) const {return(bit_number != c.bit_number);}      //!< definition of the '!=' operator over class objects
	bool operator*(clsBit const& c) const {return(this);}                      //!< definition of the '*' operator over class objects
	// --- data-access methods
	int getNumber() const {return(bit_number);}            //!< gets bit's number within the port
	int getType() const {return(node_type);}               //!< gets bit's type within the port
	t_TAdata getTAdata() const {return(ta_data);}          //!< gets bit's number within the port
	void setTAdata(t_TAdata _ta_data) {ta_data = _ta_data;}  //!< sets testability mark to a bit
	void clrTAdata(){ta_data.nbitsC=ta_data.nclksC=ta_data.nbitsO=ta_data.nclksO=0; 
  ta_data.multCprev=ta_data.multCctrl=ta_data.multOprev=ta_data.multOctrl=ta_data.con=ta_data.obs=ta_data.tst=0.0;} //!< clears testability mark of a bit
  void evalTst() {ta_data.tst = ta_data.con * ta_data.obs; }  //!< evaluates testability of a bit by means of controllability and observability
  void setOrig(bool v) { orig = v; }    //!< sets originality flag of the object to \a v value
  bool getOrig() const { return(orig); }//!< gets value of object's originality flag 
};

typedef map<int, clsBit>::value_type vt_bit;

/******************************************************************************
*	clsPort class - interface header
******************************************************************************/
/*! \class clsPort
 *  \brief object of the class represents port within module/circuit/design interface 
 *  
 *  information about ports related to particular module/circuit/design interface 
 *  is read from a file whose name must be specified by means of tool's command line parameters (see tool's help to \a -i command line parameter).
 *  Each module/circuit/design is assigned an independent set of the class objects (stored in 
 *  clsModule::mapPort / clsCircuit::mapPort / clsDesign::mapPort maps).
 */
class clsPort
{
  // --- properties
  string name;        //!< port name
  int node_type;      //!< port type (see enum tNode)
  int lbit;          //!< low-bit of the port
  int hbit;          //!< high-bit of the port
  bool orig;          //!< originality flag. Set to true for a bit originally present in the design

  t_TAdata ta_data;   //!< testability analysis related data \warning value valid only if testability analysis was run before reading the value

	public:
  map<int, clsBit> mapBit;    //!< map of all bits within port
	// --- constructor+destructor 
	/*!
	 *  @param[in] nm name of the port
	 *  @param[in] nt port type (see \a enum \a tNode)
	 *  @param[in] w port width 
	 *  @param[in] orig originality flag (implicitly set to true)
   */
  clsPort(string nm, int nt, int w, bool orig=true);          //!< width based constructor
	/*!
	 *  @param[in] nm name of the port
	 *  @param[in] nt port type (see \a enum \a tNode)
	 *  @param[in] hb high-bit of the port 
	 *  @param[in] lb low-bit of the port 
	 *  @param[in] orig originality flag (implicitly set to true)
   */
  clsPort(string nm, int nt, int hb, int lb, bool orig=true); //!< low/high bit based constructor
	~clsPort();  //!< destructor
	// --- cointainer-related methods
	bool operator<(clsPort const& c) const {return(name < c.name);}  //!< definition of the '<' operator over class objects
	bool operator!=(clsPort c) const {return(name != c.name);}       //!< definition of the '!=' operator over class objects
	bool operator*(clsPort const& c) const {return(this);}           //!< definition of the '*' operator over class objects
	// --- data-access methods
	string getName() const {return(name);}     //!< gets port's name 
	int getType() const {return(node_type);}   //!< gets port's type
	int getWidth() const {return(hbit-lbit+1);}//!< gets port's width
	int getLBit() const {return(lbit);}        //!< gets port's low-bit
	int getHBit() const {return(hbit);}        //!< gets port's high-bit
  /*! @param[in] bn number of the bit to be added to port 
  *   @param[in] nt type of the bit to be added to port (see ::tNode) 
  *   @param[in] orig originality flag (implicitly set to true)
  */
	bool addBit(int bn, int nt, bool orig=true); //!< adds bit to a port's clsPort::mapBit
  bool addBits(bool orig=true){bool res=false; int i; for(i=lbit; i<=hbit; i++) res|=addBit(i, node_type, orig); return(res);} //!< adds bits within port's width-range to a port's clsPort::mapBit
  clsBit *findBit(int bn){map<int, clsBit>::iterator it=mapBit.find(bn); if(it==mapBit.end()) return(NULL); else return(&(*it).second);} //!< searches for a bit within the port
  map<int, clsBit>::iterator getMapBitBegin(){return(mapBit.begin());}  //!< gets begining of port's clsPort::mapBit
  map<int, clsBit>::iterator getMapBitEnd(){return(mapBit.end());} //!< gets end of port's clsPort::mapBit
	t_TAdata getTAdata() const {return(ta_data);}  //!< gets port's testability-mark 
	void setTAdata(t_TAdata _ta_data) {ta_data = _ta_data;} //!< sets port's testability-mark 
	void clrTAdata(){ta_data.nbitsC=ta_data.nclksC=ta_data.nbitsO=ta_data.nclksO=0; 
  ta_data.multCprev=ta_data.multCctrl=ta_data.multOprev=ta_data.multOctrl=ta_data.con=ta_data.obs=ta_data.tst=0.0;} //!< clears port's testability-mark 

  int getNCon(); //!< gets number of controllable bits within the port \warning return value valid only if testability analysis was run before call of the function
  int getNObs(); //!< gets number of observable bits within the port \warning return value valid only if testability analysis was run before call of the function
  int getNTst(); //!< gets number of testable bits within the port \warning return value valid only if testability analysis was run before call of the function
  void evalTst();//!< evaluates testability of the port \warning return value valid only if testability analysis was run before call of the function

  void setOrig(bool v) { orig = v; }    //!< sets originality flag of the object to \a v value
  bool getOrig() const { return(orig); }//!< gets value of object's originality flag 
};

typedef map<string, clsPort>::value_type vt_port;
typedef map<string, int>::value_type vt_strint;

/******************************************************************************
*	clsModule class - interface header
******************************************************************************/
/*! \class clsModule
 *  \brief object of the class represents in-circuit module 
 *  
 *  information about modules (its name and type)
 *  is read from a file whose name must be specified by means of tool's command line parameters (see tool's help to \a -i command line parameter).
 *  Each circuit is assigned an independent set of the class objects (stored in 
 *  clsCircuit::mapMod map).
 */
class clsModule
{
  // --- properties
  string name;      //!< module name
  string type;      //!< name of module type
  int area;         //!< estimated actual area of the module
  int power;        //!< estimated actual wort-case power consumption of the module
  int area_orig;         //!< estimated original area of the module
  int power_orig;        //!< estimated original wort-case power consumption of the module
  map<string, clsPort> mapPort;   //!< map of all ports within module
  map<string, int> mapConsts;   //!< relation between constant name present in (general) module-type template and its value specified by particular module declaration
  set<tMTypeData, lt_mtdata> setOfTRANIS; //!< set of module's transparency data 
  bool orig;          //!< originality flag. Set to true for a bit originally present in the design

	public:
	// --- constructor+destructor 
	// --- constructor+destructor 
	/*!
	 *  @param[in] nm name of the module
	 *  @param[in] t module type 
	 *  @param[in] orig originality flag (implicitly set to true)
   */
  clsModule(string nm, string t, bool orig = true); //!< constructor
	~clsModule(); //!< destructor
	
	// --- cointainer-related methods
	bool operator<(clsModule const& c) const {return(name < c.name);} //!< definition of the '<' operator over class objects
	bool operator!=(clsModule c) const {return(name != c.name);}      //!< definition of the '!=' operator over class objects
	bool operator*(clsModule const& c) const {return(this);}           //!< definition of the '*' operator over class objects
	// --- data-access methods
	string getName() const {return(name);} //!< gets name of the module
	string getType() const {return(type);} //!< gets type of the module
  void setArea(int i){area = i;}        //!< sets area to the module
  int getArea(){return(area);}          //!< gets area of the module
  void setPower(int i){power = i;}      //!< sets power consumption to the module
  int getPower(){return(power);}        //!< gets power consumption of the module
  void setAreaOrig(int i){area_orig = i;}        //!< sets area to the module
  int getAreaOrig(){return(area_orig);}          //!< gets area of the module
  void setPowerOrig(int i){power_orig = i;}      //!< sets power consumption to the module
  int getPowerOrig(){return(power_orig);}        //!< gets power consumption of the module
	bool addPort(string nm, int nt, int w, bool orig=true);  //!< adds port with specified width to a module's clsModule::mapPort
	bool addPort(string nm, int nt, int hbit, int lbit, bool orig=true); //!< adds port (containing bits in hbit:lbit range ) to a module's clsModule::mapPort
  bool addInterface(clsModuleType *mtptr, bool orig=true); //!< adds interface to the module according to properties of clsModule::type
  bool bitsExist(string bits);  //!< function for checking existence of bits in module interface
  bool addTranis(string s, bool orig=true); //!< adds transparency data to the module
  set<tMTypeData, lt_mtdata>::iterator getSTranisBegin(){return(setOfTRANIS.begin());}  //!< gets begining of clsModule::setOfTRANIS
  set<tMTypeData, lt_mtdata>::iterator getSTranisEnd(){return(setOfTRANIS.end());} //!< gets end of clsModule::setOfTRANIS
  void eraseTranis(set<tMTypeData, lt_mtdata>::iterator t) {setOfTRANIS.erase(t);} //!< erases data stored in clsModule::setOfTRANIS
  bool addTranspData(clsModuleType *mtptr, bool orig=true); //!< adds transparency data to the module according to properties of clsModule::type

  clsPort *findPort(string nm){map<string, clsPort>::iterator it=mapPort.find(nm); if(it==mapPort.end()) return(NULL); else return(&(*it).second);} //!< searches for the port within module interface
  map<string, clsPort>::iterator getMapPortBegin(){return(mapPort.begin());} //!< gets begining of clsModule::mapPort
  map<string, clsPort>::iterator getMapPortEnd(){return(mapPort.end());}  //!< gets end of clsModule::mapPort
  void erasePort(map<string, clsPort>::iterator p) {mapPort.erase(p);}    //!< erases data stored in clsModule::mapPort
	bool addConst(string s_names, string s_values);                         //!< adds constant and its value to clsModule::mapConsts
  int evalExpr(string nm);      //!< evaluates given expression. Values to be assigned to non-numeric symbols utilized in the expression are taken from clsModule::mapConsts
  int getConstVal(string nm){map<string, int>::iterator it=mapConsts.find(nm); if(it==mapConsts.end()) return(-1); else return((*it).second);} //!< gets value of given constant
  map<string, int>::iterator getMapConstsBegin(){return(mapConsts.begin());} //!< gets beginning of clsModule::mapConsts
  map<string, int>::iterator getMapConstsEnd(){return(mapConsts.end());}      //!< gets end of clsModule::mapConsts

  void setOrig(bool v) { orig = v; }    //!< sets originality flag of the object to \a v value
  bool getOrig() const { return(orig); }//!< gets value of object's originality flag 

  bool changeTypeTo(string type); //!< changes clsModule::type to given type (including interface change, transparency-behavior change etc.). See dual clsModule::typeBack for more info \warning actually, mentioned functions are to be called in a pair, with no nesting
  bool typeBack();                //!< returns clsModule::type to its original setting (including interface change, transparency-behavior change etc.). See dual clsModule::changeTypeTo for more info \warning actually, mentioned functions are to be called in a pair, with no nesting
};

typedef map<string, clsModule>::value_type vt_mod;
class clsDesign;

/******************************************************************************
*	clsCircuit class - interface header
******************************************************************************/
/*! \class clsCircuit
 *  \brief object of the class represents in-circuit circuit 
 *  
 *  in the object, information about in-circuit modules, ports, wires etc. is stored.
 *  Actually, it is read from circuit-related file specified by means of tool's command 
 *  line parameters (see tool's help to \a -i command line parameter). 
 *  In the next (hierarchical) version of the tool, each module within a circuit 
 *  can represent another circuit - loaded from a file explicitly specified for such 
 *  a module. Map of 0-level circuits is in clsDesign::mapCir. 
 */
class clsCircuit
{
  // --- properties
  string name;      //!< circuit name
  map<string, clsPort> mapPort;   //!< map of in-circuit ports 
  map<string, clsModule> mapMod;    //!< map of in-circuit modules 
  set<clsWire> setOfWires;  //!< set of in-circuit wires used to decribe interconnections among module interfaces
  
  int area; //!< estimated worst-case area value (for update, call clsCircuit::getArea) of the circuit 
  int power; //!< estimated power-consumption value (for update, call clsCircuit::getPower) of the circuit
  
	public:
  double con; //!< controllability value of the circuit \warning value valid only if testability analysis was run before reading the value
  double obs; //!< observability value of the circuit \warning value valid only if testability analysis was run before reading the value
  double tst; //!< testability value of the circuit \warning value valid only if testability analysis was run before reading the value
  
  double rcon;  //!< ratio of controllable nodes within the circuit \warning return value valid only if testability analysis was run before reading the value
  double robs;  //!< ratio of observable nodes within the circuit \warning return value valid only if testability analysis was run before reading the value
  double rtst;  //!< ratio of testable nodes within the circuit \warning return value valid only if testability analysis was run before reading the value

	// --- constructor+destructor 
  clsCircuit(string nm);  //!< constructor
	~clsCircuit();           //!< destructor
	
	// --- cointainer-related methods
	bool operator<(clsCircuit const& c) const {return(name < c.name);} //!< definition of the '<' operator over class objects
	bool operator!=(clsCircuit c) const {return(name != c.name);}      //!< definition of the '!=' operator over class objects
	bool operator*(clsCircuit const& c) const {return(this);}          //!< definition of the '*' operator over class objects
	// --- data-access methods
	string getName() const {return(name);} //!< gets circuit name
	int getNAllBits();                     //!< gets sum of bits within in-design interfaces
	int getNBits();                        //!< gets sum of bits with data-flow control ability within in-design interfaces 
	int getNClks();                        //!< gets sum of clock-bits within in-design interfaces
	bool addPort(string nm, int nt, int w, bool orig=true); //!< adds a port with specified width to circuit interface
  clsPort *findPort(string nm){map<string, clsPort>::iterator it=mapPort.find(nm); if(it==mapPort.end()) return(NULL); else return(&(*it).second);} //!< searches for a port to be stored in clsCircui::mapPort
  map<string, clsPort>::iterator getMapPortBegin(){return(mapPort.begin());} //!< gets begining of clsCircuit::mapPort
  map<string, clsPort>::iterator getMapPortEnd(){return(mapPort.end());}  //!< gets end of clsCircuit::mapPort
  int getNallPorts();  //!< gets sum of bits within in-design interfaces
  double getAvgPortWidth();  //!<  gets average width of ports within the circuit
	bool addMod(string nm, string t, bool orig=true);  //!< adds module of specified type to the circuit
  clsModule *findMod(string nm){map<string, clsModule>::iterator it=mapMod.find(nm); if(it==mapMod.end()) return(NULL); else return(&(*it).second);}  //!< searches for a module to be stored in clsCircuit::mapMod
  map<string, clsModule>::iterator getMapModBegin(){return(mapMod.begin());} //!< gets begining of clsCircuit::mapMod
  map<string, clsModule>::iterator getMapModEnd(){return(mapMod.end());} //!< gets end of clsCircuit::mapMod
  int getNmodules() const {return(mapMod.size());} //!< gets number of in-circuit modules 
  bool addWire(clsWire w);  //!< adds a wire to clsCircuit::setOfWires
  bool connectPorts(string sm, string sp, string dm, string dp, bool reverse, bool orig=true); //!< connects two interfaces at the circuit-level
  set<clsWire>::iterator getSetOfWiresBegin(){return(setOfWires.begin());} //!< gets begining of clsCircuit::setOfWires
  set<clsWire>::iterator getSetOfWiresEnd(){return(setOfWires.end());} //!< gets end of clsCircuit::setOfWires
  void eraseWire(set<clsWire>::iterator w) {setOfWires.erase(w);} //!< erases data stored in clsCircuit::setOfWires
  int getNwires() const {return(setOfWires.size());} //!< gets number of in-circuit wires

  void evalArea(){area=0; map<string, clsModule>::iterator it=mapMod.begin(); for(;it!=mapMod.end();it++){ area += (*it).second.getArea(); } } //!< estimates area of the circuit
  int getArea(){ evalArea(); return(area); } //!< returns estimation of circuit area
  void evalPower(){power=0; map<string, clsModule>::iterator it=mapMod.begin(); for(;it!=mapMod.end();it++){ power += (*it).second.getPower(); } } //!< estimates peak power-consumption of the circuit
  int getPower(){ evalPower(); return(power); } //!< returns estimation of peak power consumption of the circuit
  
  //testability-analysis related methods
  int controlability_analysis(set<string> *marked_bits);  //!< starts controllability analysis
  int observability_analysis(set<string> *marked_bits); //!< starts observability analysis \warning before the analysis is started, clsCircuit::controlability_analysis must be finished
  int testability_analysis(); //!< starts testability analysis (controllability analysis first and observability analysis afterwards)
  void evalTst(); //!< according to testability analysis results, it completes evaluation of testability parameters of all in-circuit ports and circuit itself\warning return value valid only if testability analysis was run before call of the function

  // data-mine methods
  set<string> get_untestableBits(tmodeSel mode); //!< gets set of uncontrollable/unobservable/untestable bits within the circuit \warning return value valid only if testability analysis was run before call of the function
  set<string> get_badTsourceBits(int type); //!< not implemented yet \warning return value valid only if testability analysis was run before call of the function
  set<string> get_regNames(string fname); //!< gets register names specified in special file (see tool's command-line help)
  set<string> get_scanRegCandidateBits(set<string> regs); //!< returns set of in-register bits as candidates for selection into scan chains
  set<string> get_scanRegCandidates(set<string> regs); //!< returns set of registers as candidates for selection into scan chains

  // export methods
  void export_TAresults_txt(string fname);  //!< exports testability analysis results to a plain-text formated file (see help to tool's command-line option \a -o) \warning return value valid only if testability analysis was run before call of the function
  void export_TAresults_htm(string fname); //!< exports testability analysis results to a html-formated file (see help to tool's command-line option \a -o) \warning return value valid only if testability analysis was run before call of the function
  void export_TAresults_tex(string fname); //!< exports testability analysis results to a LaTEX-formated file (see help to tool's command-line option \a -o) \warning return value valid only if testability analysis was run before call of the function
  
  // dft-related methods
  set<string> get_scan_notation(string fname);  //!< gets scan notation from a file (see help to tool's command-line option \a -scan)
  bool implement_scan_notation(string fsname, string frname); //!< implements scan notation into circuit structure
  bool deimplement_changes(); //!< returns circuit structure into its original state
  
  void implementDft();  //!< used to implement all requested design-for-testability technique(s) in the future. Actually, only clsCircuit::implement_scan_notation is called in the function
};

typedef map<string, clsCircuit>::value_type vt_cir;

/******************************************************************************
*	clsDesign class - interface header
******************************************************************************/
/*! \class clsDesign
 *  \brief object of the class represents top-level component containing all data related to design being processed
 *  
 *  Actually, the object contains map of circuits (clsDesign::mapCircuit) implemented 
 *  in the design and map of templatized module-types (clsDesign::mapModType) used 
 *  to assign a type to a module.  
 *  In the future, set of wires representing interconnections between circuit
 *  interfaces will be added (see clsClass::setOfWires).   
 */
class clsDesign
{
  // --- properties
  string name;      //!< design name
  map<string, clsCircuit> mapCir;       //!< map of all circuits within design
  map<string, clsModuleType> mapModType; //!< map of all module-types used in design scope

  int area; //!< estimated worst-case area value (for update, call clsDesign::getArea) of the design
  int power; //!< estimated power-consumption value (for update, call clsDesign::getPower) of the design 
  
	public:
	// --- constructor+destructor 
  clsDesign(string nm); //!< constructor
	~clsDesign();  //!< destructor
	
	// --- data-access methods
	string getName() const {return(name);}  //!< gets design name 
	bool addCir(string nm);   //!< adds circuit to the design
  clsCircuit *findCir(string nm){map<string, clsCircuit>::iterator it=mapCir.find(nm); if(it==mapCir.end()) return(NULL); else return(&(*it).second);}  //!< serches for a circuit to be stored in clsDesign::mapCir
  map<string, clsCircuit>::iterator getMapCirBegin(){return(mapCir.begin());}  //!< returns beginning of clsDesign::mapCir
  map<string, clsCircuit>::iterator getMapCirEnd(){return(mapCir.end());} //!< returns end of clsDesign::mapCir
	bool addModType(string nm);  //!< adds module-type to the design
  clsModuleType *findModType(string nm); //!< searches for a module-type to be stored in clsDesign::mapModType
  void syncModWithType(clsModule *mptr); //!< synchronizes module with parameters (interface, constants, bit-widths, transparency parameters etc.) of its type
  void dump_info(); //!< sends information about design and its component to stdout

  void evalArea(){area=0; map<string, clsCircuit>::iterator it=mapCir.begin(); for(;it!=mapCir.end();it++){ area += (*it).second.getArea(); } } //!< estimates area of the design
  int getArea(){ evalArea(); return(area); }  //!< returns estimation of design area
  void evalPower(){power=0; map<string, clsCircuit>::iterator it=mapCir.begin(); for(;it!=mapCir.end();it++){ power += (*it).second.getPower(); } } //!< estimates peak power-consumption of the design
  int getPower(){ evalPower(); return(power); } //!< returns estimation of peak power consumption of the design

  // export methods
  void export_TAresults_txt(string fname); //!< not implemented yet
  
  // dft methods
  void implementDft(string cir="");  //!< function used to implement design-for-testability technique(s) into the design/circuit. 
  void deimplement_changes(string cir=""); //!< function used to return design/circuit structure to its original state
};

#endif
