/******************************************************************************
* TASTE (Testability Analysis SuiTE) version 1.00
* Copyright (c) 2008 Josef Strnadel, all rights reserved
* Brno University of Technology, Faculty of Information Technology
* -----------------------------------------------------------------------------
* This is a free software: you can redistribute it and/or modify it 
* under the terms of the latest version of the GNU Lesser General Public License 
* as published by the Free Software Foundation.
* 
* This software is distributed in the hope that it will be useful, 
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
* See the GNU General Public License and the GNU Lesser General Public License 
* for more details.
* 
* You should have received a copy of the GNU General Public License
* and GNU Lesser General Public License along with this software. 
* If not, see <http://www.gnu.org/licenses/>.
* -----------------------------------------------------------------------------
* filename:     fileio.cc
* version:      1.00
* started:      25. 4. 2007
* last revised: 21. 7. 2008
* language:     ANSI C++
* author:       Josef Strnadel
*               web:      http://www.fit.vutbr.cz/~strnadel/index.php.en
*               e-mail:   strnadel@fit.vutbr.cz
*               address:  Bozetechova 2, 61266 Brno, Czech Republic
*               phone:    +420 54114-1211
*               fax:      +420 54114-1270
* -----------------------------------------------------------------------------
* description:	io-functions
* -----------------------------------------------------------------------------
* notes:        -
* -----------------------------------------------------------------------------
* history of modifications:
*               -
* -----------------------------------------------------------------------------
* known bugs:   -
******************************************************************************/
//! \file fileio.cc io-functions module 

#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#include <stdio.h>
#include <stdlib.h>

#include "error.h"
#include "fileio.h"
#include "model.h"
#include "stringtokenizer/StringTokenizer.h"

/*!\brief end-of-line define  */
#define EOLN 10

/*!\brief line counter */
unsigned long line;

/*!\brief actual file name size limit */
char act_fname[MAX_STR_LEN];

extern bool debug_mode;
extern unsigned long nerrors;
extern clsDesign *dsgnPtr;

/*!\brief loader of module-type templates
*/
/*! on basis of information stored in fname file, information about module-type templates
 *  is stored into clsDesign object. Each module within a design is assigned a
 *  type.
 *  @param[in] fname name of file to be read from
 *  @param[out] dptr pointer to clsDesign object created according to data stored in the file
 *  \callgraph
 */
void loadTemplates(const char *fname, class clsDesign *dptr)
{
  int i, state, ntokens;
  char c, c1, c2;
  bool incomment=false;
  int sw=0;
  string str;         // line-storage
  string tempStr;
  ostringstream strm; // stream for line-reading

  string mType;
  clsModuleType *cmtptr=NULL;
  
  fstream fin(fname);
  if(fin==NULL) {warn("Library file '%s' not found - absence of the library can cause unexpected errors!\n"); return;};
  c1 = c2 = ' ';
  line=1;
  state = 0;

  strcpy(act_fname, fname);

  while(!fin.eof())
  {
    sw++;
    if(!debug_mode)
    {
      if(sw==1) cout << "\rLoading templates from '" << fname << "'... [|]";
      if(sw==2) cout << "\rLoading templates from '" << fname << "'... [/]";
      if(sw==3) cout << "\rLoading templates from '" << fname << "'... [-]";
      if(sw==4) {cout << "\rLoading templates from '" << fname << "'... [\\]"; sw=0;}
    }

    c=(char)(i=fin.get());
    c2 = c1;
    c1 = c;
    if((c1=='/')&&(c1=='/')) incomment=true;
    if(!incomment && !(c==EOLN)) strm << c;
    if(c==EOLN) // ------------------------ new line read
    {
      if(!incomment)
      {
        str=strm.str(); // read line stored in 'str'

        tempStr = str;
        StringTokenizer strtok = StringTokenizer(tempStr," ");
        ntokens = strtok.countTokens();
        
        for(int i = 0; i < ntokens; i++)
        {
           tempStr=strtok.nextToken();

          switch(state)
          {
            case 0:     // MODULE_TYPE-line expected
              if(i==0)
              {
                  if(tempStr != string("MODULE_TYPE")) error("'MODULE_TYPE' expected at the beginning of line %ld in '%s'!\n", line, fname);
              }
              if(i==1)
              {
                  mType = tempStr;
                  if(dptr)
                  {
                    dptr->addModType(mType);
                  }
                  else {error("dptr points to NULL! Insertion of module %s skipped (line %ld of '%s')\n", mType.c_str(), line, fname);}
                  state = 1;
              }
            break;

            case 1:     // INTERFACE-line expected
              if(i==0)
              {
                  if(tempStr != string("INTERFACE")) error("'INTERFACE' expected at the beginning of line %ld in '%s'!\n", line, fname);
              } 
              if(i>=1)
              { 
                if(dptr)
                {
                  cmtptr = dptr->findModType(mType);
                  if(!cmtptr) warn("Port '%s' not inserted into interface because module-type '%s' does not exist! See line %ld in '%s'!\n", tempStr.c_str(), mType.c_str(), line, fname);
                  else
                  {
                    cmtptr->addPort(tempStr);
                  }
                }
              }
              if(i==(ntokens-1))
              {
                  state=6;
              }
            break;

            case 6:     // IMPL-line expected
              if(i==0)
              {
                  if(tempStr != string("IMPL")) error("'IMPL' expected at the beginning of line %ld in '%s'!\n", line, fname);
              }
              if(i==1)  // get area-info
              {
                if(dptr)
                {
                  cmtptr = dptr->findModType(mType);
                  if(!cmtptr) warn("AREA '%s' not inserted because module-type '%s' does not exist! See line %ld in '%s'!\n", tempStr.c_str(), mType.c_str(), line, fname);
                  else
                  {
                    tempStr=str_after(tempStr, "area@");
                    cmtptr->setArea(tempStr);
                  }
                }
              }
              if(i==2)  // get power-info
              {
                if(dptr)
                {
                  cmtptr = dptr->findModType(mType);
                  if(!cmtptr) warn("POWER '%s' not insertedbecause module-type '%s' does not exist! See line %ld in '%s'!\n", tempStr.c_str(), mType.c_str(), line, fname);
                  else
                  {
                    tempStr=str_after(tempStr, "power@");
                    cmtptr->setPower(tempStr);
                  }
                }

                state = 2;
              }
            break;

            case 2:     // TRANI-line expected
              if(i==0)
              {
                  if(tempStr != string("TRANI")) error("'TRANI' expected at the beginning of line %ld in '%s'!\n", line, fname);
              } 
              if(i>=1)
              {
              }
              if(i==(ntokens-1))
              {
                  state=3;
              }
            break;

            case 3:     // TRANS-line expected
              if(i==0)
              {
                  if(tempStr != string("TRANS")) error("'TRANS' expected at the beginning of line %ld in '%s'!\n", line, fname);
              } 
              if(i>=1)
              {
              }
              if(i==(ntokens-1))
              {
                  state=4;
              }
            break;

            case 4:     // TRANIS-line expected
              if(i==0)
              {
                  if(tempStr != string("TRANIS")) error("'TRANIS' expected at the beginning of line %ld in '%s'!\n", line, fname);
              } 
              if(i>=1)
              {
                  if(dptr)
                  {
                      cmtptr = dptr->findModType(mType);
                      if(!cmtptr) warn("TRANIS '%s' not inserted because module-type '%s' does not exist! See line %ld in '%s'!\n", tempStr.c_str(), mType.c_str(), line, fname);
                      else
                      {
                        cmtptr->addTranis(tempStr);
                      }
                  }
                  else {error("dptr points to NULL! Finding of module %s skipped (line %ld of '%s')", mType.c_str(), line, fname);}
              }
              if(i==(ntokens-1))
              {
                  state=5;
              }
            break;

            case 5:     // TRND-line expected
              if(i==0)
              {
                  if(tempStr != string("TRND")) error("'TRND' expected at the beginning of line %ld in '%s'!\n", line, fname);
              } 
              if(i>=1)
              {
              }
              if(i==(ntokens-1))
              {
                  state=0;
              }
            break;
          }
        }
      }
      strm.str("");   // clear stream for new line to be read
      incomment=false;
      line++;
    }
  }
  cout << "\rLoading templates from '" << fname << "'... [OK]" << endl;
  fin.close();
}

/*!\brief netlist loader
 */
/*! on basis of information stored in fname file, corresponding clsDesign object 
 *  is created and its pointer returned by reference. 
 *  @param[in] f handler of file to be read from
 *  @param[in] fname name of file to be read from
 *  @param[out] dptr pointer to clsDesign object created according to data stored in the file
 */
void loadNetlist(FILE *f, const char *fname, class clsDesign *dptr)
{
	char cnm[MAX_STR_LEN];
	char inm[MAX_STR_LEN];
	char pnm[MAX_STR_LEN];
	char ptyp[MAX_STR_LEN];
	char pw[MAX_STR_LEN];
	char id[MAX_STR_LEN];
	unsigned long nbits, i, state, err;
   signed char c;
  int sw=0;
  clsCircuit *cptr=NULL;
  clsModule *mptr=NULL;

  strcpy(act_fname, fname);

   state = 0;     // reset automata state
   i=err=0;
 	 line=1;

  string src_port, dst_port;

	while((c=fgetc(f))!=EOF)
   {
    sw++;
    if(!debug_mode)
    {
      if(sw==1) cout << "\rLoading netlist from " << fname << "... [|]";
      if(sw==2) cout << "\rLoading netlist from " << fname << "... [/]";
      if(sw==3) cout << "\rLoading netlist from " << fname << "... [-]";
      if(sw==4) {cout << "\rLoading netlist from " << fname << "... [\\]"; sw=0;}
    }
    
      if(c==EOLN) line++;

      switch(state)
      {
         case 0: // on the 1st character of a line
			{
            // new-line -> init all necessary variables
            i=0; // string character counter            

            if(!isspace(c)) // skip spaces at the line beginning
            {
               if(c=='/') state=1;
               else
               { 
                  if(isalpha(c)) {id[i++]=c; state=3;}
                  else {error("Alpha-char expected at the beginning of line %ld!\n",line);}
                }
               }
            }
         break;

         case 1: // first '/'
         {
            if(c=='/') state=2;
            else {error("'/' expected (line %ld)\n", line);} //return(cptr)};
			}
         break;

		  case 2: // in-comment position
        {
           if(c==EOLN) state=0;
        }
        break;

        case 3: // read 1st ID at the beginning of the line
        {
           if(!isspace(c)) id[i++]=c;
           else 
           {
				  state=4;
              id[i]=0;
           }
        }
        break;

        case 4: // determine ID type
        {
           i=0;
           if(strcmp("CIR",id)==0) {state=5; cnm[i++]=c;} // goto start-read-cir-data state 
           else
           {
				  if(strcmp("ITM",id)==0) {state=50; inm[i++]=c;} // goto start-read-item-data state
				  else
              {
                 if(strcmp("LINK",id)==0) { state=100; inm[i++]=c;} // goto start-read-links-data state
                 else
                 {
                     error("Wrong ID at the begining of line %ld! One of these IDs expected: CIR, ITM, LINK\n", line);
                 }
              }
           }
        }
		  break;

        case 5: // start-read-cir-data state
        {
           if(!isspace(c)) // read circuit name
           {
              cnm[i++]=c;
           }
           else 
           {
              state=6; 
              cnm[i]=0; 
              i=0; 

              if(debug_mode) printf("Circuit name is: %s\n", cnm);
              dptr->addCir(cnm);
              if(!(cptr = dptr->findCir(cnm))) warn("Circuit '%s' does not exist in design '%s' dptr->getName(). See line %ld in '%s'.\n", line, fname);
          }
        }
        break;

        case 6: // read circuit port name
        {
           if(c!='(')
           {
              if(c==EOLN){state=0; i=0;}
				  if(!isspace(c)) pnm[i++]=c;
           }
           else {state=7; pnm[i]=0; i=0;}
		  }
        break;

        case 7: // read circuit port type
        {
           if(c!=',')
           {
              ptyp[i++]=c;
			  }
           else {state=8; ptyp[i]=0; i=0;}
        }
        break;

        case 8: // read circuit port bit-width
        {
           if(c!=')')
           {
              pw[i++]=c;
           }
           else // read next circuit port data 
           {
				      state=6;
              pw[i]=0; 
              i=0; 
              
              int ptype = 0;
    				  if(strcmp("in",ptyp)==0) {ptype=NODE_IN;}
              if(strcmp("out",ptyp)==0) {ptype=NODE_OUT;}
              if(strcmp("inout",ptyp)==0) {ptype=NODE_IN|NODE_OUT;}
              if(strcmp("sel",ptyp)==0) {ptype=NODE_CTRL;}
              if(strcmp("clk",ptyp)==0) {ptype=NODE_CLK;}

    				  nbits=(unsigned long)strtol(pw, NULL, 10);

              if(!ptype) warn("Type '%s' for port '%s' not supported (it should be one of: in, out, inout, sel, clk)! See line %ld in '%s'. Skipping insertion of the port.\n", ptyp, pnm, line, fname);
              else
              {
                if(cptr)
                {
                  cptr->addPort(pnm, ptype, nbits);
                }
                else warn("Insertion of in-circuit port '%s' failed because circuit '%s' does not exist in design '%s' dptr->getName(). See line %ld in '%s'.\n", pnm, line, fname);
          
                if(debug_mode) printf("Primary-port data are: name='%s', type='%s', nbits='%ld'\n", pnm, ptyp, nbits);
              }
            }
		  }
        break;

        case 50: // start-read-itm-data state
        {
           if(c!='(') // read itm name
           {
              inm[i++]=c;
			  }
           else {state=51; inm[i]=0; i=0;}
        }
        break;

        case 51: // read ITM type
        {
           if(c!=')')
           {
              id[i++]=c;
           }
           else 
           {
				  state=52;
              id[i]=0; 
              i=0;

              if(dptr)
              {
                if(dptr->findModType(id))
                {
                  if(cptr)
                  {
                    cptr->addMod(inm, id);  // add module to circuit
                    mptr = cptr->findMod(inm);
                    if(mptr) // actions according to module-type
                    {
                      dptr->syncModWithType(mptr); 
                    }
                    else warn("Attempt to insert interface to module '%s' failed! See line %ld of '%s'", inm, line, fname);
                  }
                  else warn("Insertion of in-circuit module '%s' failed because circuit '%s' does not exist in design '%s'! See line %ld in '%s'.\n", inm, dptr->getName().c_str(), line, fname);
                }
                else warn("Insertion of in-circuit module '%s' failed because module-type '%s' (or corresponding templatized type) not loaded for design '%s'! See line %ld in '%s'.\n", inm, id, dptr->getName().c_str(), line, fname);
              }
              else warn("Insertion of in-circuit module '%s' failed because design '%s' does not exist! See line %ld in '%s'.\n", dptr->getName().c_str(), inm, line, fname);
           }
        }
        break;

        case 52:
        {
           if(c==EOLN)
           {
              state=0; 
              i=0; 
              break;
           }

           if(!isspace(c))
           {
              state=53;
              i=0;
              pnm[i++]=c;
           }
        }
        break;

        case 53: // read item port name
        {
           if(c!='(')
           {
              if(c==EOLN){state=0; i=0;}
              pnm[i++]=c;
           }
           else {state=54; pnm[i]=0; i=0;}
        }
        break;

        case 54: // read item port type
        {
           if(c!=',')
           {
              ptyp[i++]=c;
           }
           else {state=55; ptyp[i]=0; i=0;}
        }
        break;

        case 55: // read item port bit-width
        {
           if(c!=')')
           {
				  pw[i++]=c;
           }
           else // read next item port data 
           {
              state=52; 
              pw[i]=0; 
              i=0; 
           }
        }
		  break;

        case 100: // read link data
        {
           if(c!='.') // read src item
           {
              inm[i++]=c;
           }
           else {state=101; inm[i]=0; i=0;}
        }
        break;

        case 101: // read src port
        {
           if(!isspace(c))
           {
              pnm[i++]=c;
           }
           else 
			  {
              pnm[i]=0; 
              i=0; 
              state=102; 
              
              sprintf(id,"%s%s%s",inm,".",pnm);
              src_port = id;
           }
        }
        break;

         case 102: // read '-'
         {
            if(c=='-') state=103;
				    else {error("'-' expected (line %ld)\n", line);}
         }
			break;

        case 103: // '>' expected
        {
           if(c=='>') state=104;
           else {error("'>' expected (line %ld)\n", line);}
        }
        break;

        case 104:
        {
			  if(!isspace(c))
           {
              state=105;
              inm[i++]=c;
           }
        }
        break;
 
        case 105: // read link data
        {
           if(c!='.') // read dst item
			  {
				  inm[i++]=c;
			  }
           else {state=106; inm[i]=0; i=0;}
        }
        break;

        case 106: // read dst port
        {
           if(!isspace(c))
           {
              pnm[i++]=c;
           }
           else 
           {
				  pnm[i]=0;
              i=0; 
              state=104; 

              sprintf(id,"%s%s%s",inm,".",pnm);
              dst_port = id;
              string from = string(src_port);
              string to = string(dst_port);
              string from_module="", from_port="", to_module="", to_port="";
              string cirName = string(cnm);
              unsigned int i;
              if((i=from.find(cirName))==string::npos) {from_module=str_before(from, ".");}
              from_port=str_after(from, ".");
              if((i=to.find(cirName))==string::npos) {to_module=str_before(to, ".");}
              to_port=str_after(to, ".");
              
              if(dptr) 
              { 
                if(cptr)
                {
                    cptr->connectPorts(from_module, from_port, to_module, to_port, false, true);
                }
                else warn("Insertion of wire '%s' failed because circuit '%s' does not exist in design '%s'! See line %ld in '%s'.\n", (from+"..."+to).c_str(), dptr->getName().c_str(), line, fname);
              }
           } 
			  if(c==EOLN){state=0; i=0; break;}
		  }
		  break;

		}
	}
   printf("\rLoading netlist from '%s' - circuit '%s' loaded... [OK]\n", fname, cnm);
}

