/*
 * Decompiled with CFR 0.152.
 */
package org.ndx.model.pcap;

import java.util.function.Consumer;
import org.apache.commons.codec.binary.Hex;
import org.ndx.model.Packet;
import org.ndx.model.parsers.applayer.DnsPcapParser;
import org.ndx.model.parsers.applayer.HttpPcapParser;
import org.ndx.model.parsers.applayer.SslHelper;
import org.ndx.model.pcap.BitConverter;
import org.ndx.model.pcap.PacketModel;
import org.ndx.model.pcap.PacketPayload;

public class PcapPacket
extends Packet {
    private static final long UNIX_BASE_TICKS = 621355968000000000L;
    private static final long TICKS_PER_MILLISECOND = 10000L;
    private static final int ETHERNET_HEADER_SIZE = 14;
    private static final int ETHERNET_TYPE_OFFSET = 12;
    private static final int ETHERNET_TYPE_IP = 2048;
    private static final int ETHERNET_TYPE_IPV6 = 34525;
    private static final int ETHERNET_TYPE_8021Q = 33024;
    private static final int SLL_HEADER_BASE_SIZE = 10;
    private static final int SLL_ADDRESS_LENGTH_OFFSET = 4;
    private static final int IPV6_HEADER_SIZE = 40;
    private static final int IP_VHL_OFFSET = 0;
    private static final int IP_TTL_OFFSET = 8;
    private static final int IP_TOTAL_LEN_OFFSET = 2;
    private static final int IPV6_PAYLOAD_LEN_OFFSET = 4;
    private static final int IPV6_HOPLIMIT_OFFSET = 7;
    private static final int IP_PROTOCOL_OFFSET = 9;
    private static final int IPV6_NEXTHEADER_OFFSET = 6;
    private static final int IP_SRC_OFFSET = 12;
    private static final int IPV6_SRC_OFFSET = 8;
    private static final int IP_DST_OFFSET = 16;
    private static final int IPV6_DST_OFFSET = 24;
    private static final int IP_ID_OFFSET = 4;
    private static final int IPV6_ID_OFFSET = 4;
    private static final int IP_FLAGS_OFFSET = 6;
    private static final int IPV6_FLAGS_OFFSET = 3;
    private static final int IP_FRAGMENT_OFFSET = 6;
    private static final int IPV6_FRAGMENT_OFFSET = 2;
    private static final int PROTOCOL_HEADER_SRC_PORT_OFFSET = 0;
    private static final int PROTOCOL_HEADER_DST_PORT_OFFSET = 2;
    private static final int PROTOCOL_HEADER_TCP_SEQ_OFFSET = 4;
    private static final int PROTOCOL_HEADER_TCP_ACK_OFFSET = 8;
    private static final int TCP_HEADER_DATA_OFFSET = 12;

    @Override
    public void parsePacket(PacketModel.RawFrame frame) {
        this.parsePacket(frame, this::processTcpUdpPayload);
    }

    public void parsePacket(PacketModel.RawFrame frame, Consumer<PacketPayload> processPayload) {
        byte[] packetData = frame.getData().toByteArray();
        int snapLen = 65535;
        int frameNumber = frame.getFrameNumber();
        this.put("frame_len", frame.getFrameLength());
        this.put("ts", this.convertTimeStamp(frame.getTimeStamp()));
        this.put("number", frameNumber);
        int ipStart = this.findIPStart(frame.getLinkTypeValue(), packetData);
        if (ipStart == -1) {
            return;
        }
        int ipProtocolHeaderVersion = this.getInternetProtocolHeaderVersion(packetData, ipStart);
        this.put("ip_version", ipProtocolHeaderVersion);
        if (ipProtocolHeaderVersion == 4 || ipProtocolHeaderVersion == 6) {
            int totalLength;
            int ipHeaderLen = this.getInternetProtocolHeaderLength(packetData, ipProtocolHeaderVersion, ipStart);
            if (ipProtocolHeaderVersion == 4) {
                this.buildInternetProtocolV4Packet(packetData, ipStart);
                totalLength = BitConverter.convertShort(packetData, ipStart + 2);
            } else {
                this.buildInternetProtocolV6Packet(packetData, ipStart);
                ipHeaderLen += this.buildInternetProtocolV6ExtensionHeaderFragment(packetData, ipStart);
                int payloadLength = BitConverter.convertShort(packetData, ipStart + 4);
                totalLength = payloadLength + 40;
            }
            this.put("ip_header_length", ipHeaderLen);
            if (((Boolean)this.get("fragment")).booleanValue()) {
                LOG.info(PcapPacket.getLogPrefix(frameNumber) + "IP fragment detected - fragmented packets are not supported.");
            } else {
                String protocol = (String)this.get("protocol");
                int payloadDataStart = ipStart + ipHeaderLen;
                int payloadLength = totalLength - ipHeaderLen;
                byte[] packetPayload = this.readPayload(packetData, payloadDataStart, payloadLength, snapLen);
                if ("UDP".equals(protocol) || "TCP".equals(protocol)) {
                    packetPayload = this.buildTcpAndUdpPacket(packetData, ipProtocolHeaderVersion, ipStart, ipHeaderLen, totalLength, snapLen);
                }
                if ("TCP".equals(protocol)) {
                    this.put("tcp_payload", packetPayload != null ? Hex.encodeHexString(packetPayload) : "");
                }
                this.put("len", packetPayload != null ? packetPayload.length : 0);
                this.processPacketPayload(packetPayload, processPayload);
            }
        }
    }

    private void processTcpUdpPayload(PacketPayload packetPayload) {
        byte[] payload = packetPayload.getPayload();
        if (payload == null || payload.length == 0) {
            this.put("app_protocol", Packet.AppLayerProtocols.UNKNOWN);
            return;
        }
        if (this.tryParseDnsProtocol(payload)) {
            return;
        }
        if (this.tryParseEmailProtocol()) {
            return;
        }
        if (this.tryParseHttpProtocol(payload)) {
            return;
        }
        if (this.tryParseSslProtocol()) {
            return;
        }
        this.put("app_protocol", Packet.AppLayerProtocols.UNKNOWN);
    }

    private boolean tryParseSslProtocol() {
        Integer dst;
        Integer src = (Integer)this.get("src_port");
        Packet.ProtocolsOverSsl sslProtocol = SslHelper.detectSslProtocol(src, dst = (Integer)this.get("dst_port"));
        if (sslProtocol != Packet.ProtocolsOverSsl.UNKNOWN) {
            this.put("app_protocol", Packet.AppLayerProtocols.SSL);
            this.put("ssl_protocol", sslProtocol);
            return true;
        }
        return false;
    }

    private boolean tryParseHttpProtocol(byte[] payload) {
        HttpPcapParser httpParser = new HttpPcapParser();
        try {
            httpParser.parse(payload);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
        this.put("app_protocol", Packet.AppLayerProtocols.HTTP);
        this.putAll(httpParser);
        return true;
    }

    private boolean tryParseEmailProtocol() {
        Integer src = (Integer)this.get("src_port");
        Integer dst = (Integer)this.get("dst_port");
        if (src == null || dst == null) {
            return false;
        }
        Packet.AppLayerProtocols protocol = Packet.AppLayerProtocols.UNKNOWN;
        if (src == 110 || dst == 110) {
            protocol = Packet.AppLayerProtocols.POP3;
        } else if (src == 143 || dst == 143) {
            protocol = Packet.AppLayerProtocols.IMAP;
        } else if (src == 25 || dst == 25 || src == 587 || dst == 587) {
            protocol = Packet.AppLayerProtocols.SMTP;
        }
        if (protocol == Packet.AppLayerProtocols.UNKNOWN) {
            return false;
        }
        this.put("app_protocol", protocol);
        return true;
    }

    private boolean tryParseDnsProtocol(byte[] payload) {
        Integer src = (Integer)this.get("src_port");
        Integer dst = (Integer)this.get("dst_port");
        if (src != null && dst != null && (src == 53 || dst == 53)) {
            try {
                DnsPcapParser dnsParser = new DnsPcapParser();
                dnsParser.parse(payload);
                this.put("app_protocol", Packet.AppLayerProtocols.DNS);
                this.putAll(dnsParser);
                return true;
            }
            catch (IllegalArgumentException e) {
                LOG.warn(PcapPacket.getLogPrefix((Integer)this.get("number")) + "Malformed DNS packet.");
                return false;
            }
        }
        return false;
    }

    private long convertTimeStamp(long timeStamp) {
        return (timeStamp - 621355968000000000L) / 10000L;
    }

    private void processPacketPayload(byte[] payload, Consumer<PacketPayload> processPayload) {
        if (processPayload != null) {
            processPayload.accept(new PacketPayload(this, payload));
        }
    }

    private int findIPStart(int linkType, byte[] packet) {
        switch (linkType) {
            case 0: {
                return 4;
            }
            case 1: {
                int start = 14;
                int etherType = BitConverter.convertShort(packet, 12);
                if (etherType == 33024) {
                    etherType = BitConverter.convertShort(packet, 16);
                    start += 4;
                }
                if (etherType != 2048 && etherType != 34525) break;
                return start;
            }
            case 12: {
                return 0;
            }
            case 108: {
                return 4;
            }
            case 113: {
                int start = 10;
                int sllAddressLength = BitConverter.convertShort(packet, 4);
                return start += sllAddressLength;
            }
        }
        return -1;
    }

    private int getInternetProtocolHeaderLength(byte[] packet, int ipProtocolHeaderVersion, int ipStart) {
        if (ipProtocolHeaderVersion == 4) {
            return (packet[ipStart + 0] & 0xF) * 4;
        }
        if (ipProtocolHeaderVersion == 6) {
            return 40;
        }
        return -1;
    }

    private int getInternetProtocolHeaderVersion(byte[] packet, int ipStart) {
        return packet[ipStart + 0] >> 4 & 0xF;
    }

    private int getTcpHeaderLength(byte[] packet, int tcpStart) {
        int dataOffset = tcpStart + 12;
        return (packet[dataOffset] >> 4 & 0xF) * 4;
    }

    private void buildInternetProtocolV4Packet(byte[] packetData, int ipStart) {
        long id = BitConverter.convertShort(packetData, ipStart + 4);
        this.put("id", id);
        int flags = packetData[ipStart + 6] & 0xE0;
        this.put("ip_flags_df", (flags & 0x40) != 0);
        this.put("ip_flags_mf", (flags & 0x20) != 0);
        long fragmentOffset = (BitConverter.convertShort(packetData, ipStart + 6) & 0x1FFF) * 8;
        this.put("fragment_offset", fragmentOffset);
        if ((flags & 0x20) != 0 || fragmentOffset != 0L) {
            this.put("fragment", true);
            this.put("last_fragment", (flags & 0x20) == 0 && fragmentOffset != 0L);
        } else {
            this.put("fragment", false);
        }
        int ttl = packetData[ipStart + 8] & 0xFF;
        this.put("ttl", ttl);
        byte protocol = packetData[ipStart + 9];
        this.put("protocol", PcapPacket.convertProtocolIdentifier(protocol));
        String src = BitConverter.convertAddress(packetData, ipStart + 12, 4);
        this.put("src", src);
        String dst = BitConverter.convertAddress(packetData, ipStart + 16, 4);
        this.put("dst", dst);
    }

    private void buildInternetProtocolV6Packet(byte[] packetData, int ipStart) {
        int ttl = packetData[ipStart + 7] & 0xFF;
        this.put("ttl", ttl);
        byte protocol = packetData[ipStart + 6];
        this.put("protocol", PcapPacket.convertProtocolIdentifier(protocol));
        String src = BitConverter.convertAddress(packetData, ipStart + 8, 16);
        this.put("src", src);
        String dst = BitConverter.convertAddress(packetData, ipStart + 24, 16);
        this.put("dst", dst);
    }

    private int buildInternetProtocolV6ExtensionHeaderFragment(byte[] packetData, int ipStart) {
        if ("Fragment".equals(this.get("protocol"))) {
            long id = BitConverter.convertUnsignedInt(packetData, ipStart + 40 + 4);
            this.put("id", id);
            int flags = packetData[ipStart + 40 + 3] & 7;
            this.put("ipv6_flags_m", (flags & 1) != 0);
            long fragmentOffset = BitConverter.convertShort(packetData, ipStart + 40 + 2) & 0xFFF8;
            this.put("fragment_offset", fragmentOffset);
            this.put("fragment", true);
            this.put("last_fragment", (flags & 1) == 0 && fragmentOffset != 0L);
            byte protocol = packetData[ipStart + 40];
            this.put("protocol", PcapPacket.convertProtocolIdentifier(protocol));
            return 8;
        }
        this.put("fragment", false);
        return 0;
    }

    private byte[] buildTcpAndUdpPacket(byte[] packetData, int ipProtocolHeaderVersion, int ipStart, int ipHeaderLen, int totalLength, int snapLen) {
        int tcpOrUdpHeaderSize;
        this.put("src_port", BitConverter.convertShort(packetData, ipStart + ipHeaderLen + 0));
        this.put("dst_port", BitConverter.convertShort(packetData, ipStart + ipHeaderLen + 2));
        String protocol = (String)this.get("protocol");
        if ("UDP".equals(protocol)) {
            int cksum;
            tcpOrUdpHeaderSize = 8;
            if (ipProtocolHeaderVersion == 4 && (cksum = this.getUdpChecksum(packetData, ipStart, ipHeaderLen)) >= 0) {
                this.put("udpsum", cksum);
            }
            int udpLen = this.getUdpLength(packetData, ipStart, ipHeaderLen);
            this.put("udp_length", udpLen);
            this.put("payload_len", udpLen);
        } else if ("TCP".equals(protocol)) {
            tcpOrUdpHeaderSize = this.getTcpHeaderLength(packetData, ipStart + ipHeaderLen);
            this.put("tcp_header_length", tcpOrUdpHeaderSize);
            this.put("tcp_seq", BitConverter.convertUnsignedInt(packetData, ipStart + ipHeaderLen + 4));
            this.put("tcp_ack", BitConverter.convertUnsignedInt(packetData, ipStart + ipHeaderLen + 8));
            int flags = BitConverter.convertShort(new byte[]{packetData[ipStart + ipHeaderLen + 12], packetData[ipStart + ipHeaderLen + 12 + 1]}) & 0x1FF;
            this.put("tcp_flag_ns", (flags & 0x100) != 0);
            this.put("tcp_flag_cwr", (flags & 0x80) != 0);
            this.put("tcp_flag_ece", (flags & 0x40) != 0);
            this.put("tcp_flag_urg", (flags & 0x20) != 0);
            this.put("tcp_flag_ack", (flags & 0x10) != 0);
            this.put("tcp_flag_psh", (flags & 8) != 0);
            this.put("tcp_flag_rst", (flags & 4) != 0);
            this.put("tcp_flag_syn", (flags & 2) != 0);
            this.put("tcp_flag_fin", (flags & 1) != 0);
            int tcpLen = totalLength - (tcpOrUdpHeaderSize + ipHeaderLen);
            this.put("payload_len", tcpLen);
        } else {
            return null;
        }
        int payloadDataStart = ipStart + ipHeaderLen + tcpOrUdpHeaderSize;
        int payloadLength = totalLength - ipHeaderLen - tcpOrUdpHeaderSize;
        return this.readPayload(packetData, payloadDataStart, payloadLength, snapLen);
    }

    private int getUdpChecksum(byte[] packetData, int ipStart, int ipHeaderLen) {
        if (packetData[ipStart + ipHeaderLen + 6] == 0 && packetData[ipStart + ipHeaderLen + 7] == 0) {
            return -1;
        }
        byte[] data = new byte[packetData.length - ipStart - ipHeaderLen + 12];
        int sum = 0;
        System.arraycopy(packetData, ipStart + 12, data, 0, 4);
        System.arraycopy(packetData, ipStart + 16, data, 4, 4);
        data[8] = 0;
        data[9] = 17;
        System.arraycopy(packetData, ipStart + ipHeaderLen + 4, data, 10, 2);
        System.arraycopy(packetData, ipStart + ipHeaderLen, data, 12, packetData.length - ipStart - ipHeaderLen);
        for (int i = 0; i < data.length; ++i) {
            int j = data[i];
            if (j < 0) {
                j += 256;
            }
            sum += j << (i % 2 == 0 ? 8 : 0);
        }
        sum = (sum >> 16) + (sum & 0xFFFF);
        sum += sum >> 16;
        return ~sum & 0xFFFF;
    }

    private int getUdpLength(byte[] packetData, int ipStart, int ipHeaderLen) {
        return BitConverter.convertShort(packetData, ipStart + ipHeaderLen + 4);
    }

    private byte[] readPayload(byte[] packetData, int payloadDataStart, int payloadLength, int snapLen) {
        Integer frameNumber = (Integer)this.get("number");
        if (payloadLength < 0) {
            LOG.warn(PcapPacket.getLogPrefix(frameNumber) + "Malformed packet - negative payload length. Returning empty payload.");
            return new byte[0];
        }
        if (payloadDataStart > packetData.length) {
            LOG.warn(PcapPacket.getLogPrefix(frameNumber) + "Payload start (" + payloadDataStart + ") is larger than packet data (" + packetData.length + "). Returning empty payload.");
            return new byte[0];
        }
        if (payloadDataStart + payloadLength > packetData.length) {
            if (payloadDataStart + payloadLength <= snapLen) {
                LOG.warn(PcapPacket.getLogPrefix(frameNumber) + "Payload length field value (" + payloadLength + ") is larger than available packet data (" + (packetData.length - payloadDataStart) + "). Packet may be corrupted. Returning only available data.");
            }
            payloadLength = packetData.length - payloadDataStart;
        }
        byte[] data = new byte[payloadLength];
        System.arraycopy(packetData, payloadDataStart, data, 0, payloadLength);
        return data;
    }
}

