/**
 *  Author: Martin Čadík <cadik@fit.vutbr.cz>
 *          Lionel Baboud
 *          Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#include "CamUtil.h"

Matrix4f CamUtil::transfoC2Niso(float hFOV, int w, int h, float zNear, float zFar) {
    return transfoC2Niso(hFOV, float(w) / float(h), zNear, zFar);
    }

Matrix4f CamUtil::transfoC2Niso(float hFOV, float aspectRatio, float zNear, float zFar) {
    const float ax = tanf(hFOV * M_PI);
    const float ay = ax / aspectRatio;
    return proj(ax, ay, zNear, zFar);
    }

Matrix4f CamUtil::transfoC2N(float hFOV, float vFOV, float zNear, float zFar) {
    const float ax = tanf(hFOV * M_PI);
    const float ay = tanf(vFOV * M_PI);
    return proj(ax, ay, zNear, zFar);
    }

Matrix4f CamUtil::proj(float ax, float ay, float zNear, float zFar) {
    const float cx = 1.0f / ax;
    const float cy = 1.0f / ay;
    const float z0 = -zNear;
    const float z1 = -zFar;
    const float az = (z0 + z1) / (z0 - z1);
    const float bz = (1 - az) * z0;
    
    Matrix4f m;
    m <<    cx, 0, 0, 0,
            0, cy, 0, 0,
            0, 0, az, bz,
            0, 0, -1, 0;
    return m;
    }

float CamUtil::horizFOV(float focalLength, float sensorWidth) {
    return 2 * atanf(sensorWidth / (2 * focalLength));
    }


Matrix4f CamUtil::transfoW2C(float yaw, float pitch, float roll) {
    return transfoW2C(double(yaw), double(pitch), double(roll)).cast<float>();
    }

void CamUtil::rotationAngles(Matrix4f W2C, float &yaw, float &pitch, float &roll) {
    double yawD, pitchD, rollD;
    rotationAngles(W2C.cast<double>(), yawD, pitchD, rollD);
    yaw   = static_cast<float>(yawD);
    pitch = static_cast<float>(pitchD);
    roll  = static_cast<float>(rollD);
    }

Matrix4d CamUtil::transfoW2C(double yaw, double pitch, double roll) {

    Matrix4d eye;
    eye << 0,1,0,0,
           0,0,1,0,
           1,0,0,0,
           0,0,0,1;
    Matrix4d eyeinv = eye.inverse(); // so that the camera viewing direction is along -z
    return
        CamUtil::rotationZ(-roll ) *
        CamUtil::rotationX(-pitch) *
        CamUtil::rotationY(-yaw  ) *
        eyeinv;
    }

void printProj3D(Matrix4d p)
{
    for (int i = 0; i < 4; ++i)
    {
        for (int j = 0; j < 4; ++j)
        {
            std::cout << p(i, j) << " ";
        }
        std::cout << std::endl;
    }
    std::cout << std::endl;
}

void CamUtil::rotationAngles(Matrix4d W2C, double &yaw, double &pitch, double &roll) {
    Matrix4d R2W;
    R2W << 0,0,0,0,
           0,1,0,0,
           0,0,1,0,
           1,0,0,1;

    Matrix4d R2C = W2C * R2W;

    printProj3D(R2C);

    Vector4d A = R2C.inverse() * Vector4d(0,0,1,1);
    yaw = atan2(A.x(), A.z());
    //R2C = R2C * Proj3d::rotation(yaw, Vec3d(0,1,0), true);
    R2C = R2C * CamUtil::rotationY(yaw);

    A = R2C.inverse() * Vector4d(0,0,1,1);
    pitch = -atan2(A.y(), A.z());
    //R2C = R2C * Proj3d::rotation(pitch, Vec3d(1,0,0), true);
    R2C = R2C * CamUtil::rotationX(pitch);
    
    A = R2C.inverse() * Vector4d(1,0,0,1);
    roll = atan2(A.y(), A.x());
    //R2C = R2C * Proj3d::rotation(roll, Vec3d(0,0,1), true);   // for debug : should give identity
    //R2C = R2C * Proj3d::rotationZ(roll);   // for debug : should give identity
    }

Matrix4d CamUtil::rotationX(double theta)
{
    const double c = cos(theta);
    const double s = sin(theta);
    Matrix4d rot;
    rot <<  1 , 0 , 0 ,  0,
            0 , c ,-s ,  0,
            0 , s , c ,  0,
            0 , 0 , 0 ,  1;
    return rot;
}

Matrix4d CamUtil::rotationY(double theta)
{
    const double c = cos(theta);
    const double s = sin(theta);
    Matrix4d rot;
    rot <<  c , 0 , s ,  0,
            0 , 1 , 0 ,  0,
            -s , 0 , c , 0,
            0 , 0 , 0 ,  1;
    return rot;
}

Matrix4d CamUtil::rotationZ(double theta)
{
    const double c = cos(theta);
    const double s = sin(theta);
    Matrix4d rot;
    rot <<  c ,-s , 0 ,  0,
            s , c , 0 ,  0,
            0 , 0 , 1 ,  0,
            0 , 0 , 0 ,  1;
    return rot;
}

