/**
 *  DatabaseEntry.h
 *
 *  Created on: Nov 14, 2014
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2014  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 *
 * DatabaseEntry is parent class for all entries in BagOfWords database
 * (BOWDatabase). Each database entry has to override distance() method to
 * evaluate the distance from other DatabaseEntries.
 */

#ifndef PROJECTS_RELIEF_PANO_SRC_HORIZON_LOCATE_HLOC_DATABASEENTRY_H_
#define PROJECTS_RELIEF_PANO_SRC_HORIZON_LOCATE_HLOC_DATABASEENTRY_H_

#include <vector>
#include <map>
#include <cstdio>
#include <memory>

#include <QDomDocument>
#include <QDebug>

#include "LocalFeature.h"
#include "Persistable.h"
#include "Contourlett.h"
#include "ContourlettExtractor.h"

#include <libxml/xmlreader.h>

class DatabaseEntry : public Persistable {
public:

    enum FeaType{
        ONLY_2_5,
        ONLY_10,
        BOTH
    };

	///Key of the entry - e. g. image name
	QString key;

	DatabaseEntry()
	{
		histogramSum = 0;
	}

    void extractFeatures(ContourlettExtractor &ce, FeaType ft)
    {
        if (ft == FeaType::ONLY_2_5 || ft == FeaType::BOTH)
        {
            ce.extract(hFOV, 2.5, features);
        }
        if (ft == FeaType::ONLY_10 || ft == FeaType::BOTH)
        {
            ce.extract(hFOV, 10, features);
        }
     }

    DatabaseEntry(QString filePath, float fov, FeaType ft)
	: key(filePath)
	{
		hFOV = fov;
		ContourlettExtractor ce(filePath);
        extractFeatures(ce, ft);
		histogramSum = 0;
		this->calculateHistogram();
		this->calculateHistogramSum();
		//this->printHistogram();
	}

    DatabaseEntry(ContourlettExtractor &ce, QString filePath, float fov, FeaType ft)
    : key(filePath)
    {
        hFOV = fov;
        extractFeatures(ce, ft);
        histogramSum = 0;
        this->calculateHistogram();
        this->calculateHistogramSum();
    }

	///Normalized histogram
	std::map<int, double> histogram;

	double getTf(int feature_id)
	{
		//TODO should not be needed
		if (histogramSum == 0)
			return 0;
		std::map<int, double>::iterator it = histogram.find(feature_id);
		if (it != histogram.end())
		{
			return it->second / histogramSum;
		}
		return 0;
	}

	double getCount(int feature_id)
	{
		std::map<int, double>::iterator it = histogram.find(feature_id);
		if (it != histogram.end())
		{
			return it->second;
		}
		return 0;
	}

	bool containsFeatureId(int feature_id)
	{
		return (histogram.find(feature_id) != histogram.end());
	}

	void printHistogram()
	{
		for (std::map<int, double>::iterator it = histogram.begin();
						it != histogram.end(); ++it)
		{
			printf("bin: %d, value: %f \n", it->first, it->second);
		}
	}

	virtual QDomElement toXMLElement(QDomDocument &doc)
	{
		QDomElement de = doc.createElement("DatabaseEntry");
		de.setAttribute("key", key);
		for (std::vector<std::shared_ptr<LocalFeature> >::iterator it = features.begin();
				it != features.end(); ++it)
		{
			de.appendChild((*it)->toXMLElement(doc));
		}
		return de;
	}

    virtual void saveToXmlStream(QXmlStreamWriter &xml)
    {
        xml.writeStartElement("DatabaseEntry");
        xml.writeAttribute("key", key);
        for (std::vector<std::shared_ptr<LocalFeature> >::iterator it = features.begin();
                it != features.end(); ++it)
        {
            (*it)->saveToXmlStream(xml);
        }
        xml.writeEndElement();
    }

	virtual void initFromXMLElement(QDomElement &e)
	{
		key = e.attributeNode("key").value();
		//TODO delete features
		QDomNode n = e.firstChild();
		while (!n.isNull())
		{
			QDomElement c_dom = n.toElement();
			if (!c_dom.isNull())
			{
				//Contourlett obtained, create it
				Contourlett *c = new Contourlett;
				c->initFromXMLElement(c_dom);
				features.push_back(std::shared_ptr<Contourlett>(c));
			}
			n = n.nextSibling();
		}

		this->calculateHistogram();
		this->calculateHistogramSum();
	}

	virtual void initWithXmlStreamReader(QXmlStreamReader &xml)
	{
		while (!((xml.isStartElement()) && (xml.name() == "DatabaseEntry")))
		{
			if (xml.hasError())
			{
				qDebug() << "error: " << xml.errorString();
				break;
			}
			//qDebug() << "start: " << xml.name() << "token type " << xml.tokenType();
			xml.readNext();
		}
		
		this->key = xml.attributes().value("", "key").toString();
		
		while (!xml.atEnd() && !xml.hasError())
		{
			xml.readNext();
			//qDebug() << "xml name: " << xml.name();
			if (xml.isStartElement() && xml.name() == "c_id")
			{
				//qDebug() << "init contourlett";
				Contourlett *c = new Contourlett;
				c->initWithXmlStreamReader(xml);
				//printf("%s, %d\n", this->key.toStdString().c_str(), c->featureId());
				features.push_back(std::shared_ptr<Contourlett>(c));
			}
			else break;
		}
	}

    virtual int initWithXmlStreamReader(xmlTextReaderPtr &reader)
    {
        const xmlChar *name, *_key;

        _key = xmlTextReaderGetAttribute(reader, (xmlChar *)"key");
        this->key = QString((const char *)_key);
        xmlFree((void *)_key);
        _key = NULL;

        int ret = 1;
        while (ret == 1)
        {
            ret = xmlTextReaderRead(reader);
            name = xmlTextReaderLocalName(reader);
            if (name != NULL && strcmp((const char *)name, "c_id") == 0)
            {
                Contourlett *c = new Contourlett;
                ret = c->initWithXmlStreamReader(reader);
                features.push_back(std::shared_ptr<Contourlett>(c));

                xmlFree((void *)name);
                name = NULL;
            }
            else break;
        }
        return ret;
    }

	std::vector< std::shared_ptr<LocalFeature> >::const_iterator
	featuresBegin()
	{
		return features.begin();
	}

	std::vector< std::shared_ptr<LocalFeature> >::const_iterator
	featuresEnd()
	{
		return features.end();
	}

	int size()
	{
		return features.size();
	}

public:
	float hFOV;


private:


	///Vector of local features
	std::vector< std::shared_ptr<LocalFeature> > features;

	int histogramSum;

	virtual void calculateHistogram()
	{
		for (std::vector< std::shared_ptr<LocalFeature> >::iterator it = features.begin();
				it != features.end(); ++it)
		{
			histogram[(*it)->featureId()]++;
		}
	}

	virtual void calculateHistogramSum()
	{
		int sum = 0;
		for (std::map<int, double>::iterator it = histogram.begin();
				it != histogram.end(); ++it)
		{
			sum += it->second;
		}
		histogramSum = sum;
	}

	virtual void normalizeHistogram()
	{
		for (std::map<int, double>::iterator it = histogram.begin();
					it != histogram.end(); ++it)
		{
			histogram[it->first] /= histogramSum;
		}

	}


};

#endif /* PROJECTS_RELIEF_PANO_SRC_HORIZON_LOCATE_HLOC_DATABASEENTRY_H_ */
