#include "VideoReader.h"
#include "ImageReader.h"
#include "CameraReader.h"
#include "EmptyReader.h"
#include "NoneReader.h"
#include "main.h"
#include "WaveletOpenmpSep.h"
#include "WaveletOpenclSep.h"
#include "WaveletOpenclComb.h"
#include "MainConfig.h"
#include <sstream>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include "Debug.h"
#include <omp.h>


char *read_file(char *filename)
  {
    std::ifstream file(filename, std::fstream::binary | std::fstream::in);
    if(!file.is_open()) return NULL;
    
    file.seekg(0, std::ios_base::end);
    size_t file_length = (size_t)file.tellg();
    file.seekg(0, std::ios_base::beg);
    
    char *data = new char[file_length + 1];
    
    file.read(data,file_length);
    data[file_length] = '\0';
    
    if(file.fail()) return NULL;
    
    file.close();
    
    return data;
  }

size_t ceil_align(size_t num, size_t block)
  {
    return ((num + block - 1)/block) * block;
  }

double get_time(cl_event proc_event)
  {
    cl_ulong start, end;
    clGetEventProfilingInfo(proc_event, CL_PROFILING_COMMAND_START, sizeof(cl_ulong), (void *)&start, NULL);
    clGetEventProfilingInfo(proc_event, CL_PROFILING_COMMAND_END, sizeof(cl_ulong), (void *)&end, NULL);
    return (end - start)/1000000.0;
  }



void print_device_info(cl_uint device_num, cl_device_id id, FILE *file)
{
    size_t par_size;
    cl_platform_id platform;
    cl_uint subdevices_count;
    cl_uint compute_units_count;

    clGetDeviceInfo(id, CL_DEVICE_NAME, 0, NULL, &par_size);
    char *device_name = new char[par_size + 1];
    clGetDeviceInfo(id, CL_DEVICE_NAME, par_size, (void *)device_name, NULL);
    device_name[par_size] = '\0';
    clGetDeviceInfo(id, CL_DEVICE_PARTITION_MAX_SUB_DEVICES, sizeof(cl_uint), (void*)&subdevices_count, NULL);
    
    clGetDeviceInfo(id, CL_DEVICE_PLATFORM, sizeof(cl_platform_id), (void *)(&platform), NULL);

    clGetDeviceInfo(id, CL_DEVICE_MAX_COMPUTE_UNITS, sizeof(cl_uint), (void *)(&compute_units_count), NULL);

    clGetPlatformInfo(platform, CL_PLATFORM_NAME, 0, NULL, &par_size);
    char *platform_name = new char[par_size + 1];
    clGetPlatformInfo(platform, CL_PLATFORM_NAME, par_size, (void *)platform_name, NULL);
    platform_name[par_size] = '\0';

    fprintf(file, "\nDevice info:\n" \
                      " Device id:       %d\n" \
                      " Subdevices count %d\n" \
                      " Multiprocessors: %d\n" \
                      " Platform name:   %s\n" \
                      " Device name:     %s\n", device_num, subdevices_count, compute_units_count, platform_name, device_name);
}

void print_detection_info(e_output_type output_type, det_output timer, int frame_count)
  {
    fprintf(stderr, "\nDetection info:\n" \
                      "  Processed %d frames.\n" \
                      "\nDetection time info:\n" \
                      "  Average detection time   %9.4f ms  consists of\n" \
                      "    Average preprocess time  %9.4f ms\n" \
                      "    Average process time     %9.4f ms\n" \
                      "    Average postprocess time %9.4f ms\n", frame_count, timer.getTotalTime(), timer.getTotalPreprocessTime(), timer.getTotalProcessTime(), timer.getTotalPostprocessTime());
    fprintf(stderr, "\nDetection preprocess info:\n");
    for(unsigned int i = 0; i < timer.pre.size(); i++)
      {
        fprintf(stderr, "  Average kernel rank %3d time %9.4f ms\n", i, timer.pre[i]);
      }
    fprintf(stderr, "\nDetection process info:\n");
    for(unsigned int i = 0; i < timer.proc.size(); i++)
      {
        fprintf(stderr, "  Average kernel rank %3d time %9.4f ms\n", i, timer.proc[i]);
      }
    fprintf(stderr, "\nDetection postprocess info:\n");
    for(unsigned int i = 0; i < timer.post.size(); i++)
      {
        fprintf(stderr, "  Average kernel rank %3d time %9.4f ms\n", i, timer.post[i]);
      }
    switch(output_type)
      {
        case CONF_OUTPUT_TYPE_CSV:
          fprintf(stderr, "\nBenchmark Data:\n");
          fprintf(stdout, "%d\n%.4f\n%.4f;%.4f;%.4f\n", frame_count, timer.getTotalTime(), timer.getTotalPreprocessTime(), timer.getTotalProcessTime(), timer.getTotalPostprocessTime());
          for(unsigned int i = 0; i < timer.pre.size(); i++)
            {
              if(i != 0) fprintf(stdout, ";");
              fprintf(stdout, "%.4f", timer.pre[i]);
            }
          fprintf(stdout, "\n");
          for(unsigned int i = 0; i < timer.proc.size(); i++)
            {
              if(i != 0) fprintf(stdout, ";");
              fprintf(stdout, "%.4f", timer.proc[i]);
            }
          fprintf(stdout, "\n");
          for(unsigned int i = 0; i < timer.post.size(); i++)
            {
              if(i != 0) fprintf(stdout, ";");
              fprintf(stdout, "%.4f", timer.post[i]);
            }
          fprintf(stdout, "\n");
        break;
        default:
        break;
      }
  }

int main(int argc, char *argv[])
  {
		std::stringstream error;
		int width, height;
		IplImage *v_input;
		IplImage *v_output;
		void *bgra_frame;
		MainConfig m_conf;
    if(!m_conf.loadFromArguments(argc, argv, &error))
      {
        fprintf(stderr, "\nExiting program: Cannot get basic program parameters.\n");
        fprintf(stderr, error.str().c_str());
        exit(1);
      }
    if(error.str().compare("") != 0)
      {
        fprintf(stderr, "\nProgram parameters info:\n");
        fprintf(stderr, error.str().c_str());
        error.clear();
      }

    if(m_conf.print_devices)
      {
        WaveletOpencl::printDevicesInfo(&error, stdout);
      }
		VideoReader video_reader;
    CameraReader cam_reader;
    ImageReader image_reader;
    EmptyReader empty_reader;
    NoneReader none_reader;
    
    std::vector<FileReaderInterface *> temp_reader;
    FileReaderInterface *reader = NULL;
    if(m_conf.benchmark_proc || m_conf.memless_exec)
      {
        if((!m_conf.in_show) && (!m_conf.out_show))
          {
            temp_reader.push_back(&none_reader);
          }
        else
          {
            temp_reader.push_back(&empty_reader);
          }
      }
    else if(m_conf.cam_use)
      {
        temp_reader.push_back(&cam_reader);
      }
    else
      {
        temp_reader.push_back(&image_reader);
        temp_reader.push_back(&video_reader);
      }
    for(size_t i = 0; i < temp_reader.size(); i++)
      {
        if(temp_reader[i]->openFile(m_conf.in_file, width, height, true, &error))
          {
            reader = temp_reader[i];
            break;
          }
      }
    if(reader == NULL)
      {
        fprintf(stderr, "\nFile loader info:\n");
        fprintf(stderr, error.str().c_str());
        fprintf(stderr, "\nExiting program: File reader isnt at valid state.\n");
        exit(3);
      }
    rect image_size(width, height);
    Wavelet *wavelet;
    WaveletOpenclSep *wavelet_opencl_sep;
    WaveletOpenclComb *wavelet_opencl_comb;
    WaveletOpenmpSep *wavelet_openmp_sep;
    switch(m_conf.engine)
      {
        case ENGINE_TYPE_OPENCL_SEP:
            wavelet_opencl_sep = new WaveletOpenclSep(image_size, &(m_conf.opencl_sep_par));
            wavelet = (Wavelet*)wavelet_opencl_sep;
        break;
        case ENGINE_TYPE_OPENCL_COMB:
            wavelet_opencl_comb = new WaveletOpenclComb(image_size, &(m_conf.opencl_comb_par));
            wavelet = (Wavelet*)wavelet_opencl_comb;
        break;
        case ENGINE_TYPE_OPENMP_SEP:
            wavelet_openmp_sep = new WaveletOpenmpSep(image_size, &(m_conf.openmp_sep_par));
            wavelet = (Wavelet*)wavelet_openmp_sep;
        break;
      }
    wavelet->printError();
    if(!wavelet->isValid())
      {
        fprintf(stderr, "\nDetector info:\n");
				fprintf(stderr, error.str().c_str());
        wavelet->printError();
        fprintf(stderr, "\nExiting program: Detector isnt at valid state.\n");
				exit(4);
      }
		if(m_conf.in_show)
			{
				cvNamedWindow("Input video", CV_WINDOW_AUTOSIZE);
        v_input = cvCreateImageHeader(cvSize((int)image_size.w, (int)image_size.h), IPL_DEPTH_8U, 4);
			}
		if(m_conf.out_show)
			{
				cvNamedWindow("Output video", CV_WINDOW_AUTOSIZE);
        v_output = cvCreateImageHeader(cvSize((int)image_size.w, (int)image_size.h), IPL_DEPTH_8U, 4);
			}
    det_output timer;
    det_output_stat timers;
    bool end = false;
    char key;
    int i;
    CvFont font;
    cvInitFont(&font,CV_FONT_HERSHEY_SIMPLEX, 0.4, 0.4, 0, 1);
    std::stringstream frame_info;
    const char * key_info = "Quit:q";
    bool process_stopped;
    bool first;
    double start_time = omp_get_wtime();
    for(i = 0; reader->getFrame(&bgra_frame, &error); i++)
			{
        timer.reset();
        if(i == m_conf.frame_count)
          {
            break;
          }
				if(m_conf.in_show)
					{
            v_input->imageData = (char *)bgra_frame;
						cvShowImage("Input video", v_input);
						if(cvWaitKey(1) >= 0)
							{
								break;
							}
					}
        if(!wavelet->getFrame((uchar4 *)bgra_frame, &timer))
          {
            wavelet->printError();
            fprintf(stderr, "\nProcess frame info:\n");
				    fprintf(stderr, error.str().c_str());
            fprintf(stderr, "\nExiting program: Cannot process frame.\n");
            exit(5);
          }
        timers.output.push_back(timer);
				if(m_conf.out_show)
					{
            frame_info.str(std::string());
            switch(m_conf.engine)
              {
                case ENGINE_TYPE_OPENCL_SEP:
                case ENGINE_TYPE_OPENCL_COMB:
                  if(m_conf.memless_exec || m_conf.benchmark_proc) frame_info << " Process time:" << timer.getTotalProcessTime();
                  else frame_info <<  " Copy time:" << (timer.pre[0] + timer.post[1]) << " Preprocess time:" << timer.pre[1] << " Process time:" << timer.getTotalProcessTime() << " Postprocess time:" << timer.post[0] << " Total time:" << timer.getTotalTime();
                break;
                case ENGINE_TYPE_OPENMP_SEP:
                case ENGINE_TYPE_OPENMP_COMB:
                  frame_info << " Preprocess time:" << timer.getTotalPreprocessTime() << " Process time:" << timer.getTotalProcessTime() << " Postprocess time:" << timer.getTotalPostprocessTime() << " Total time:" << timer.getTotalTime();
                break;
              }
            v_output->imageData = (char *)bgra_frame;
            cvPutText(v_output,key_info,cvPoint(10,15), &font, cvScalar(0,255,0));
            cvPutText(v_output,frame_info.str().c_str(),cvPoint(10,30), &font, cvScalar(0,255,0));
						cvShowImage("Output video", v_output);
            process_stopped = true;
            first = true;
            while(process_stopped)
              {
                if(first)
                  {
                    process_stopped = false;
                    first = false;
                  }
						    if((key = cvWaitKey(1) & 255) >= 0)
							    {
                    switch(key)
                      {
                        case 'q':
                          i++;
                          end = true;
                        break;
                        case ' ':
                          process_stopped = !process_stopped;
                        break;
                      }
							    }
                if(end)
                  {
                    break;
                  }
              }
					}
        if(end)
          {
            break;
          }
			}
    double end_time = omp_get_wtime();
    if(error.str().compare("") != 0)
      {
        fprintf(stderr, "\nProgram info:\n");
        fprintf(stderr, error.str().c_str());
      }
    det_output avg_timer = timers.getMedianData();
    print_detection_info(m_conf.output_type, avg_timer, i);
    fprintf(stderr, "\nAvg frame time with CPU overhead: %.4f ms\n", ((end_time - start_time)*1000.0) / ((double)(m_conf.frame_count)));
  }
