#ifndef TRANSFORM_H
#define TRANSFORM_H

#include "vec2.h"
#include "imageptr.h"
#include "range2.h"
#include "codeblock.h"

struct transform_t;

/**
 * @brief Thread-specific transform data.
 */
struct worker_t {
	/** auxiliary y-buffer */
	float *buffer_y;

	/** local (private) fragment of auxiliary x-buffer, due to the code-block prolog processing */
	float *buffer_x;

	/** code-block pointer, the code-block coordinates at which the transformation will continue */
	struct vec2_t cb_global;

	/** worker needs to access the data inside the transform */
	struct transform_t *transform;

	/** the persistent codeblock or pool of codeblocks */
	struct codeblock_t *codeblock;
};

/**
 * @brief Transform context.
 */
struct transform_t {
	/** the next scale of the transform */
	struct transform_t *next_level;

	/** Total decomposition levels requested. */
	int N;

	/** This decomposition level. */
	int n;

	/** the upper left hand sample of the tile-component (inclusive) */
	struct vec2_t tc0;
	/** the lower right hand sample of the tile-component (exclusive) */
	struct vec2_t tc1;
	/** the size of the component */
	struct vec2_t size;

	/** the code-block size (exponent) */
	struct vec2_t cb_exp;
	/** code-block size (in the transform domain) */
	struct vec2_t cb_size1;
	/** code-block size (in the original domain) */
	struct vec2_t cb_size2;

	/** the first core of the tile-component */
	struct vec2_t start;
	/** the first core after the tile-component */
	struct vec2_t stop;

	/** first codeblock */
	struct vec2_t tc0_cb;
	struct vec2_t tc1_cb;

	/** first fast (SIMD accelerated) codeblock */
	struct vec2_t regular_tc0_cb;
	struct vec2_t regular_tc1_cb;

	/** super size (frame size) */
	struct vec2_t super;

	/** auxiliary y-buffer */
	float *buffer_x;

	/** super size (frame size) for the LL band */
	struct vec2_t super_ll;

	struct imageptr_t *debug;

	/** LL band, output from this level, input of the next level */
	struct imageptr_t *llband;
	struct imageptr_t *ll_viewport;

	/** the number of transform lines to process next time */
	int cb_step;

	/** the range [inclusive, exclusive) of input lines to process next time */
	struct range2_t lines;

	/** number of threads for strip processing */
	int threads;

	/** thread-specific data */
	struct worker_t *workers;

	/** number of codeblocks processed at once (size of the pool) */
	int cb_pool;

	/** callback function to be invoked on each codeblock produced */
	codeblock_callback_t codeblock_callback;
	/** pointer to be passed into the callback function */
	void *codeblock_params;
};

/**
 * @brief Allocate the thread-specific data.
 */
int worker_create(
	struct worker_t *worker,
	struct transform_t *transform
);

/**
 * @brief Free the thread-specific data.
 */
void worker_destroy(
	struct worker_t *worker
);

/**
 * @brief Get the thread-specific data corresponding to the current thread.
 */
struct worker_t *transform_get_worker(
	struct transform_t *transform
);

/**
 * @brief Get the thread-specific data corresponding to the current thread.
 */
const struct worker_t *transform_get_worker_const(
	const struct transform_t *transform
);

/**
 * @brief Get the thread-specific data corresponding to the entire transform.
 */
struct worker_t *transform_get_master_worker(
	struct transform_t *transform
);

/**
 * @brief Get the thread-specific data corresponding to the entire transform.
 */
const struct worker_t *transform_get_master_worker_const(
	const struct transform_t *transform
);

/**
 * @brief Is the transform already completed?
 *
 * @return non-zero if true, zero if false
 */
int transform_finished(
	const struct transform_t *transform
);

/**
 * @brief Get the lines required to transform a strip of the code-blocks, eventually two such strips (due to mirroring at the edges).
 *
 * @return The caller should prepare the input lines requested in return value.
 */
struct range2_t transform_prepare_strip(
	struct transform_t *transform
);

/**
 * @brief Transform the strip of the codeblocks.
 *
 * @param data_lines pointer to the input tile
 * @param data_offset user-specific data offset
 */
void transform_process_strip(
	struct transform_t *transform,
	const struct imageptr_t *data_lines,
	const struct vec2_t data_offset
);

/**
 * @brief Create the transform object.
 *
 * @param[in] tc0 the upper left hand sample of the tile-component
 * @param[in] tc1 the lower right hand sample of the tile-component
 * @param[in] cb_exp the code-block size
 * @param[in] n start at this decomposition level
 * @param[in] N the total number of decomposition levels
 */
struct transform_t *transform_create(
	const struct vec2_t tc0,
	const struct vec2_t tc1,
	const struct vec2_t cb_exp,
	int n,
	int N,
	codeblock_callback_t codeblock_callback,
	void *codeblock_params
);

/**
 * @brief Dump the transforms into files (for debugging purposes).
 */
void transform_dump(
	const struct transform_t *transform
);

/**
 * @brief Destroy (free) the transform object.
 */
void transform_destroy(
	struct transform_t *transform
);

/**
 * @brief Transform the tile stored in memory.
 *
 * High-level function to transform the tile-component completely stored in the memory.
 * Does not do anything useful with the data.
 *
 * @param[in] tc0 the upper left hand sample of the tile-component
 * @param[in] tc1 the lower right hand sample of the tile-component
 * @param[in] cb_exp the code-block size
 * @param[in] N the total number of decomposition levels
 * @param[in] data_lines pointer to the input tile
 * @param[in] imageptr the tile-component
 */
int transform_process_tile(
	const struct vec2_t tc0,
	const struct vec2_t tc1,
	const struct vec2_t cb_exp,
	int N,
	const struct imageptr_t *imageptr
);

/**
 * @brief Loop over the tile-component stored in memory.
 *
 * @param[in] imageptr the tile-component
 */
void transform_loop_tc(
	struct transform_t *transform,
	const struct imageptr_t *imageptr
);

#endif
