#ifndef INTMATH_H
#define INTMATH_H

#include <assert.h>
#include "system.h"

/**
 * @brief Closest even integer not larger than argument.
 *
 * @note Works properly also for negative numbers.
 */
UNUSED_FUNCTION
static
int floor2(
	int x
)
{
	return x & ~1;
}

/**
 * @brief Closest multiple of 4 not larger than argument.
 */
UNUSED_FUNCTION
static
int floor4(
	int x
)
{
	return x & ~3;
}

/**
 * @brief Closest multiple of 8 not larger than argument.
 */
UNUSED_FUNCTION
static
int floor8(
	int x
)
{
	return x & ~7;
}

/**
 * @brief Closest even integer not less than argument.
 *
 * @note Works properly also for negative numbers.
 */
UNUSED_FUNCTION
static
int ceil2(
	int x
)
{
	return floor2(x+1);
}

/**
 * @brief Closest multiple of 4 not less than argument.
 */
UNUSED_FUNCTION
static
int ceil4(
	int x
)
{
	return floor4(x+3);
}

/**
 * @brief Closest multiple of 8 not less than argument.
 */
UNUSED_FUNCTION
static
int ceil8(
	int x
)
{
	return floor8(x+7);
}

/**
 * @brief Smallest integral value not less than @f$ x / 2^n @f$.
 *
 * @note i.e. ceil( x / 2^n )
 */
UNUSED_FUNCTION
static
int ceil_div_pow2(
	int x,
	int n
)
{
	assert( n >= 0 );

	// 0.999...
	const int c = (1<<n) - 1;

	return (x + c) >> n;
}

/**
 * @brief ceil( x / 2 )
 */
UNUSED_FUNCTION
static
int ceil_div_pow2_1(
	int x
)
{
	return (x + 1) >> 1;
}

/**
 * @brief floor( x / 2^n )
 */
UNUSED_FUNCTION
static
int floor_div_pow2(
	int x,
	int n
)
{
	assert( n >= 0 );

	return x >> n;
}

/**
 * @brief floor( x / 2 )
 */
UNUSED_FUNCTION
static
int floor_div_pow2_1(
	int x
)
{
	return x >> 1;
}

/**
 * @brief Minimum of two integers.
 */
UNUSED_FUNCTION
static
int min(
	int a,
	int b
)
{
	return a < b ? a : b;
}

/**
 * @brief Maximum of two integers.
 */
UNUSED_FUNCTION
static
int max(
	int a,
	int b
)
{
	return a > b ? a : b;
}

/**
 * @brief Is the coordinate at the code-block boundary?
 *
 * @param[in] x target coordinate in the original domain
 * @param[in] cb_exp base 2 logarithm of the code-block size in the reduced/transform domain
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int is_codeblock(
	int x,
	int cb_exp
)
{
	// the code-block size in the original domain
	const int cb_size2 = 2 << cb_exp;

	// mask, lower bits
	const int m = cb_size2 - 1;

	return 0 == ( x & m );
}

/**
 * @brief Floor the coordinate to the code-block boundary.
 *
 * @param x target coordinate in the original domain
 * @param cb_exp base 2 logarithm of the code-block size in the reduced/transform domain
 *
 * @returns target coordinate in the original domain
 */
UNUSED_FUNCTION
static
int floor_codeblock(
	int x,
	int cb_exp
)
{
	// the code-block size in the original domain
	const int cb_size2 = 2 << cb_exp;

	// mask, upper bits
	const unsigned m = ~(cb_size2 - 1);

	return x & m;
}

/**
 * @brief Ceil the coordinate to the code-block boundary.
 *
 * @param x target coordinate in the original domain
 * @param cb_exp base 2 logarithm of the code-block size in the reduced/transform domain
 *
 * @returns target coordinate in the original domain
 */
UNUSED_FUNCTION
static
int ceil_codeblock(
	int x,
	int cb_exp
)
{
	// the code-block size in the original domain
	const int cb_size2 = 2 << cb_exp;

	// mask, upper bits
	const unsigned m = ~(cb_size2 - 1);

	// 0.999...
	const int c = cb_size2 - 1;

	return (x + c) & m;
}

/**
 * @brief Ceil the coordinate to the code-block boundary (in the reduced domain).
 *
 * @param x target coordinate in the reduced domain
 * @param cb_exp base 2 logarithm of the code-block size in the reduced/transform domain
 *
 * @returns target coordinate in the reduced domain
 */
UNUSED_FUNCTION
static
int ceil_codeblock1(
	int x,
	int cb_exp
)
{
	// the code-block size in the original domain
	const int cb_size1 = 1 << cb_exp;

	// mask, upper bits
	const unsigned m = ~(cb_size1 - 1);

	// 0.999...
	const int c = cb_size1 - 1;

	return (x + c) & m;
}

/**
 * @brief Symmetric periodic extension of coordinates.
 *
 * @param[in] x the coordinate
 * @param[in] size the size of the signal
 *
 * @return the extended coordinate
 */
UNUSED_FUNCTION
static
int coord_extr(
	int x,
	int size
)
{
	assert( size > 0 && "the signal shall contain at least some data" );

	if( 1 == size )
	{
		return 0;
	}

	while( x < 0 || x >= size )
	{
		if( x < 0 )
			x = 0 - x;
		else
			x = 2*(size-1) - x;
	}

	assert( x >= 0 && x <= (size-1) );

	return x;
}

// NOTE: does not allow the coordinate to go outside of +4 border extended signal support
UNUSED_FUNCTION
static
int coord_extr_limited(
	int x,
	int size
)
{
	assert( size > 0 && "the signal shall contain at least some data" );

	if( 1 == size )
	{
		return 0;
	}

	// FIXME: is this really necessary?
#if 1
	// -4 -3 -2 -1 | 0 ...
	if( x < -4 )
		x = 0;
	// ... size-1 | size size+1 size+2 size+3
	if( x > size+3 )
		x = size-1;
#endif

	// NOTE: infinite loop when size=0
	while( x < 0 || x >= size )
	{
		if( x < 0 )
			x = 0 - x;
		else
			x = 2*(size-1) - x;
	}

	assert( x >= 0 && x <= (size-1) );

	return x;
}

/**
 * @brief Does the coordinate fall into the valid image area? (relative to the 0)
 *
 * @param[in] x the coordinate
 * @param[in] size the size
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int coord_valid(
	int x,
	int size
)
{
	assert( size > 0 && "the signal shall contain at least some data" );

	return x >= 0 && x < size;
}

/**
 * @brief Is valid size-2 core coordinate?
 *
 * @note even coordinates are valid
 *
 * @param[in] x the coordinate
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int is_core2(
	int x
)
{
	return 1 & ~x;
}

/**
 * @brief Is valid size-4 core coordinate?
 *
 * @note even coordinates are valid
 *
 * @param[in] x the coordinate
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int is_core4(
	int x
)
{
	return 3 == (3 & ~x);
}

/**
 * @brief Is valid size-8 core coordinate?
 *
 * @note even coordinates are valid
 *
 * @param[in] x the coordinate
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int is_core8(
	int x
)
{
	return 7 == (7 & ~x);
}

/**
 * @returns (int)ceil(x/(double)y)
 */
UNUSED_FUNCTION
static
int ceil_div(
	int x,
	int y
)
{
	return (x + y - 1) / y;
}

/**
 * @returns (int)floor(x/(double)y)
 */
UNUSED_FUNCTION
static
int floor_div(
	int x,
	int y
)
{
	return x / y;
}

#endif
