/*
 * File:     resampling.c, compiled by gcc
 * Date:     2009-02-02 21:03:33 
 * Author:   Bronislav Pribyl, xpriby12@stud.fit.vutbr.cz
 * Project:  Separable image resampling
 * Encoding: utf-8
 * Version:  1.5
 * Desc:     Resamples input picture according to given parameters
 */

#include "FIR.h"
#include "squares.h"
#include "bit-handling.h"

#include <digilib/Image.h>
#include <digilib/ImageFil.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <stdbool.h>
#include <assert.h>

//image dimensions
int IMG_WIDTH = 0;
int IMG_HEIGHT = 0;


/**
 *
 */
int getConvolvedXPixel(ImageStruct *img, int fir_length, const int fir[][fir_length], long int offset[IMG_WIDTH * IMG_HEIGHT], int destRow, int destCol, int srcRow, int srcCol)
{
	float sum = 0;
	int variant = firVariant(offset[IMG_WIDTH * destRow + destCol]);
	
	for(int i = 0; i < fir_length; i++){
		int xCoord = (srcCol + i - 3 + ImageXSize(img)) % ImageXSize(img);
		int yCoord = (srcRow + ImageYSize(img)) % ImageYSize(img);
		sum += (fir[variant][i]/(float)FIR_ONE) * Image8Pixel(img, xCoord, yCoord);
	}
	
	if(sum < 0) sum = 0;
	if(sum > 255) sum = 255;

	return (int)sum;
}


/**
 * Returns value of pixel made by convolution of column stripe FIR filter with corresponding pixels
 * @param img Pointer to image structure
 * @param fir_length Number of coeficients of FIR filter (most right array size)
 * @param fir FIR filter; 2D array of coefficients
 * @param topLeftX X-coordinate of starting pixel
 * @param topLeftY Y-coordinate of starting pixel
 * @param sx X-offset of source pixel
 * @param sy Y-offset of source pixel (bit representation)
 * @return Returns value of resultant pixel
 */
int getConvolvedYPixel(ImageStruct *img, int fir_length, const int fir[][fir_length], int topLeftX, int topLeftY, int sx, int sy, long int offset)
{
	float SY = sy + var2float(offset) + 0.5;
	float sum = 0;
	int variant = firVariant(offset);
	
	for(int i = 0; i < fir_length; i++){
		int xCoord = (topLeftX + sx + ImageXSize(img)) % ImageXSize(img);
		int yCoord = (topLeftY + (int)roundl(SY) + i - (fir_length / 2) + ImageYSize(img)) % ImageYSize(img);
		sum += (fir[variant][i]/(float)FIR_ONE) * Image8Pixel(img, xCoord, yCoord);
	}
	
	if(sum < 0) sum = 0;
	if(sum > 255) sum = 255;

	return (int)sum;
}


/**
 * Horizontally resamples image.
 * @param src Pointer to structure of source image
 * @param dest Pointer to structure of destination image
 * @param coords Pointer to structure of square coefficients
 */
void resample_1d_x(ImageStruct *src, ImageStruct *dest, TCoords *coords)
{
	long int *x_offset;
	if((x_offset = calloc(IMG_WIDTH * IMG_HEIGHT, sizeof(long int))) == NULL) {
		fprintf(stderr, "Not enough memory.\n");
		exit(EXIT_FAILURE);
	}

	//prepare array of offsets
	for(int sqRow = 0; sqRow < coords->ySquares; sqRow++){
		for(int sqCol = 0; sqCol < coords->xSquares; sqCol++){

			//input parameters of interpolation algorithm
			TPoint2D S0 = coords->squares[sqRow * coords->xSquares + sqCol].S0;
			TPoint2D DC0 = coords->squares[sqRow * coords->xSquares + sqCol].DC0;
			TPoint2D DR = coords->squares[sqRow * coords->xSquares + sqCol].DR;
			TPoint2D DDC = coords->squares[sqRow * coords->xSquares + sqCol].DDC;
			//variables of interpolation algorithm
			long int SoR;//Start of Row
			long int DC;//Differece - Cell
			long int S;//Pointer, Position, ...
			
			//coordinates of top left pixel of the square
			int topLeftX = sqCol * coords->sqSize;
			int topLeftY = sqRow * coords->sqSize;
			
			//interpolation algorithm ------------------------------------------------
			/*___ Init ___*/
			SoR = S0.x;
			DC = DC0.x;
			for(int i = 0; i < coords->sqSize; i++){
			/*___ InitX ___*/
				S = SoR;
				for(int j = 0; j < coords->sqSize; j++){
			/*___ StepX ___*/
					x_offset[IMG_WIDTH * (topLeftY + i) + (topLeftX + j)] = S;
					S += DC;
					CLEAN_VAR(S);
				}
			/*___ StepY ___*/
				DC += DDC.x;
				CLEAN_VAR(DC);
				SoR += DR.x;
				CLEAN_VAR(SoR);
			}
			//------------------------------------------------------------------------
		}
	}
	
	
	// resample whole image
	bool overflow;
	long int one  = float2var(1.0, &overflow);
	assert(!overflow);
	long int reg_offset = 0;
	long int offset_rel_cur;
	long int offset_rel_sum_cur = 0;
	long int reg_offset_rel_sum = 0;
	bool diff_is_1 = 0;
	bool diff_is_2 = 0;
	
	for (int j = 0; j < IMG_HEIGHT; j++) {
		int px2fetch = (int)(0.0 + var2float(x_offset[IMG_WIDTH*j]));
		int srcRow = j;
		int srcCol = px2fetch;
	
		for (int i = 0; i < IMG_WIDTH; i++) {
			reg_offset = x_offset[IMG_WIDTH*j + (((i-1) < 0) ? 0 : (i-1))];
			offset_rel_cur = x_offset[IMG_WIDTH*j + i] - reg_offset + one;
			CLEAN_VAR(offset_rel_cur);
			
			if (i == 0) { // row start
				offset_rel_sum_cur = x_offset[IMG_WIDTH*j];
				reg_offset_rel_sum = offset_rel_sum_cur;
			} else { // row continue
				offset_rel_sum_cur += offset_rel_cur;
				CLEAN_VAR(offset_rel_sum_cur);
			}
			
			diff_is_1 = getBit(offset_rel_sum_cur, POSITION, 16) ^ getBit(reg_offset_rel_sum, POSITION, 16);
			diff_is_2 = !(getBit(offset_rel_sum_cur, POSITION, 16) ^ getBit(reg_offset_rel_sum, POSITION, 16)) && (getBit(offset_rel_sum_cur, POSITION, 17) ^ getBit(reg_offset_rel_sum, POSITION, 17));
   
   		if (diff_is_2) {
				srcCol += 2;
			} else if (diff_is_1) {
				srcCol++;
			}
			
			Image8Pixel(dest, i, j) = getConvolvedXPixel(src, FIR_LENGTH, FIR_LANCZOS, x_offset, j, i, srcRow, srcCol);
			
			
			reg_offset_rel_sum = offset_rel_sum_cur;
		}
	}
	
	free(x_offset);
	return;
}


/**
 * Vertically resamples image.
 * @param src Pointer to structure of source image
 * @param dest Pointer to structure of destination image
 * @param coords Pointer to structure of square coefficients
 */
void resample_1d_y(ImageStruct *src, ImageStruct *dest, TCoords *coords)
{
	//resample each square
	for(int sqRow = 0; sqRow < coords->ySquares; sqRow++){
		for(int sqCol = 0; sqCol < coords->xSquares; sqCol++){

			//input parameters of interpolation algorithm
			TPoint2D S0   = coords->squares[sqRow * coords->xSquares + sqCol].S0;
			TPoint2D DC0 = coords->squares[sqRow * coords->xSquares + sqCol].DC0;
			TPoint2D DR = coords->squares[sqRow * coords->xSquares + sqCol].DR;
			TPoint2D DDC = coords->squares[sqRow * coords->xSquares + sqCol].DDC;
			//variables of interpolation algorithm
			long int SoR;//Start of Row
			long int DC;//Differece - Cell
			long int S;//Pointer, Position, ...
			
			//coordinates of top left pixel of the square
			int topLeftX = sqCol * coords->sqSize;
			int topLeftY = sqRow * coords->sqSize;
			
			//interpolation algorithm ------------------------------------------------
			/*___ Init ___*/
			SoR = S0.y;
			DC = DC0.y;
			for(int i = 0; i < coords->sqSize; i++){
			/*___ InitX ___*/
				S = SoR;
				for(int j = 0; j < coords->sqSize; j++){
			/*___ StepX ___*/
					Image8Pixel(dest, topLeftX+j, topLeftY+i) = getConvolvedYPixel(src, FIR_LENGTH, FIR_LANCZOS, topLeftX, topLeftY, j, i, S);
					S += DC;
					CLEAN_VAR(S);
				}
			/*___ StepY ___*/
				DC += DDC.y;
				CLEAN_VAR(DC);
				SoR += DR.y;
				CLEAN_VAR(SoR);
			}
			//------------------------------------------------------------------------
		}
	}
	
	return;
}


/**
 * MAIN
 */
int main(int argc, char *argv[])
{
	ImageStruct *source;
	ImageStruct *xResampled;
	ImageStruct *product;
	TCoords coords;
	

	//3 parameters required (input image, output image, coefficient set)
	if(argc != 4){
		fprintf(stderr, "resampling <input_img> <output_img> <square_coefs_file>\n");
		exit(EXIT_FAILURE);
	}

	//initialize square coeffitiens from file
	coordsInitialize(&coords, argv[3]);

	
	//create image from input file
	source = CreateImageFile(argv[1], 0, 0, 0);
	if(!source){
		fprintf(stderr, "File %s could not be opened.\n", argv[1]);
		exit(EXIT_FAILURE);
	}
	
	//find out the dimensions of the image
	IMG_WIDTH = ImageXSize(source);
	IMG_HEIGHT = ImageYSize(source);
		
	//check, if there is correct number of squares
	if(ImageXSize(source) != (coords.sqSize*coords.xSquares)){
		fprintf(stderr, "Width of image %s has to be %d pixels.\n", argv[1], coords.sqSize*coords.xSquares);
		exit(EXIT_FAILURE);
	}
	if(ImageYSize(source) != (coords.sqSize*coords.ySquares)){
		fprintf(stderr, "Height of image %s has to be %d pixels.\n", argv[1], coords.sqSize*coords.ySquares);
		exit(EXIT_FAILURE);
	}

	//create new image for horizontal resampling
	xResampled = NewImage8(ImageXSize(source), ImageYSize(source));
	//resample horizontally
	resample_1d_x(source, xResampled, &coords);
	
	//create new image for vertical resampling
	product = NewImage8(ImageXSize(source), ImageYSize(source));
	//resample vertically
	resample_1d_y(xResampled, product, &coords);

	
	//save result do output file
	if(!SaveImageFile(argv[2], 0, product, NULL, NULL)){
		fprintf(stderr, "Cannot save image.\n");
	}
	
	//delete all image structures
	DeleteImage(source);
	DeleteImage(xResampled);
	DeleteImage(product);

	exit(EXIT_SUCCESS);
}

/* end of file resampling.c */

