/*
								+--------------------------------+
								|								 |
								|	  ***	OpenGL 4.0	 ***	 |
								|								 |
								|  Copyright  -tHE SWINe- 2011	 |
								|								 |
								|		   OpenGL40.cpp			 |
								|								 |
								+--------------------------------+
*/

/**
 *	@file gl4/OpenGL40.cpp
 *	@author -tHE SWINe-
 *	@brief OpenGL 4.0 and extensions entry points
 */

#include "../NewFix.h"
#include "../CallStack.h"
#include "OpenGL40.h"
#include "../StlUtils.h"

#if defined(_WIN32) || defined(_WIN64)
#include <windows.h>
#include <GL/wglext.h>
#else // _WIN32 || _WIN64
#ifdef __APPLE__
// note that this works even if /usr/X11R6/include is not present on your
// system. in case it is, you can remove this branch and compile / link with:
//		-I /usr/X11R6/include -L/usr/X11R6/lib -lGL -lX11
// otherwise use:
//		-framework GLUT -framework OpenGL

#include <stdlib.h>
#include <string.h>
#include <AvailabilityMacros.h>

#ifdef MAC_OS_X_VERSION_10_3

#include <dlfcn.h>

void *glXGetProcAddress(const void *p_s_name)
{
	static void *p_opengl_dll = 0;
	if(!p_opengl_dll)
		p_opengl_dll = dlopen("/System/Library/Frameworks/OpenGL.framework/Versions/Current/OpenGL", RTLD_LAZY);
	return (p_opengl_dll)? dlsym(p_opengl_dll, (const char*)p_s_name) : 0;
}

#else // MAC_OS_X_VERSION_10_3

#include <mach-o/dyld.h>

void *glXGetProcAddress(const void *p_s_name)
{
	static const struct mach_header *p_opengl_dll = 0;
	if(!p_opengl_dll) {
		p_opengl_dll = NSAddImage("/System/Library/Frameworks/OpenGL.framework/Versions/Current/OpenGL",
			NSADDIMAGE_OPTION_RETURN_ON_ERROR);
	}
	// load the OpenGL library

	const char *_p_s_name = (const char*)p_s_name;
	char *p_s_mangled_name = new(std::nothrow) char[strlen(_p_s_name) + 2];
	if(!p_s_mangled_name)
		return 0;
	*p_s_mangled_name = '_';
	strcpy(p_s_mangled_name + 1, _p_s_name);
	// mangle the function name in accordance with cdecl

	NSSymbol p_symbol = (p_opengl_dll)? NSLookupSymbolInImage(p_opengl_dll, p_s_mangled_name,
		NSLOOKUPSYMBOLINIMAGE_OPTION_BIND | NSLOOKUPSYMBOLINIMAGE_OPTION_RETURN_ON_ERROR) : 0;
	delete[] p_s_mangled_name;
	return (p_symbol)? NSAddressOfSymbol(p_symbol) : 0;
	// get proc addr
}

#endif // MAC_OS_X_VERSION_10_3

#else // __APPLE__
#include <GL/glx.h>
//#include <GL/glxext.h> // my glxext.h doesn't cope nicely with some unices; gotta get another version

#ifndef GLX_VERSION_1_4

/**
 *	@def glXGetProcAddress
 *	@brief get proc address on linux platform
 *	@param[in] x is name of the function to retrieve
 */
#define glXGetProcAddress(x) (*glXGetProcAddressARB)((const GLubyte*)x)

#endif // !GLX_VERSION_1_4
#endif // __APPLE__

/**
 *	@def wglGetProcAddress
 *	@brief windows get proc address lookalike on linux platform
 *	@param[in] x is name of the function to retrieve
 */
#define wglGetProcAddress(x) (*glXGetProcAddress)((GLubyte*)(x))
#endif // _WIN32 || _WIN64

/**
 *	@def INIT_FUNC_POINTERS
 *	@brief code for function pointer initialization
 */
#ifdef __GL_NULL_FUNCTION_POINTERS
#define INIT_FUNC_POINTERS = 0
#else // __GL_NULL_FUNCTION_POINTERS
#define INIT_FUNC_POINTERS 
#endif // __GL_NULL_FUNCTION_POINTERS

/*
 *								=== CGL4ExtensionHandler ===
 */

bool CGL4ExtensionHandler::b_Support_OpenGL(int n_min_major, int n_min_minor)
{
	int n_major = n_OpenGL_Version_Major(),
		n_minor = n_OpenGL_Version_Minor();
	if(n_major > n_min_major)
		return true;
	if(n_major == n_min_major && n_minor >= n_min_minor)
		return true;
	return false;
}

int CGL4ExtensionHandler::n_OpenGL_Version()
{
	int n_major = n_OpenGL_Version_Major(),
		n_minor = n_OpenGL_Version_Minor();
	// new simpler code using OpenGL 3.0 query

	/*const char *p_s_version;
	p_s_version = (const char*)glGetString(GL_VERSION);
	if(!p_s_version) {
#if defined(_WIN32) || defined(_WIN64)
		MessageBoxA(0, "OpenGL extension string is 0!", "CGL4ExtensionHandler", MB_OK);
#endif // _WIN32 || _WIN64
		return -1;
	}

#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
	if(sscanf_s(p_s_version, "%d.%d", &n_major, &n_minor) != 2) {
#else // _MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	if(sscanf(p_s_version, "%d.%d", &n_major, &n_minor) != 2) {
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
#if defined(_WIN32) || defined(_WIN64)
		MessageBoxA(0, "OpenGL version string malformed!", "CGL4ExtensionHandler", MB_OK);
#endif // _WIN32 || _WIN64
		//_ASSERTE(0); // error - OpenGL version string malformed!
		return -1;
	}
	//_ASSERTE(n_minor >= 0 && n_minor < 10);*/
	// old code, using the GL_VERSION string (still works)

	return n_major * 10 + n_minor;
}

int CGL4ExtensionHandler::n_OpenGL_Version_Major()
{
	int n_major;
	glGetIntegerv(GL_MAJOR_VERSION, &n_major);
	return n_major;
}

int CGL4ExtensionHandler::n_OpenGL_Version_Minor()
{
	int n_minor;
	glGetIntegerv(GL_MINOR_VERSION, &n_minor);
	return n_minor;
}

bool CGL4ExtensionHandler::b_SupportedExtension(const char *p_s_extension_name)
{
	if(!GLFuncPtrName(glGetStringi) && n_GetGL30FuncPointers())
		return false;
	// requires OpenGL 3.0 functions

	int n_extension_num;
	glGetIntegerv(GL_NUM_EXTENSIONS, &n_extension_num);
	for(int i = 0; i < n_extension_num; ++ i) {
		const char *p_s_ext_name = (const char*)glGetStringi(GL_EXTENSIONS, i);
		if(!p_s_ext_name)
			continue; // ?
		if(!strcmp(p_s_ext_name, p_s_extension_name))
			return true; // extension was found
	}
	// new OpenGL 3 extensions scheme to avoid buffer overruns in old software

	return false;
}

bool CGL4ExtensionHandler::Get_ExtensionString(std::string &r_s_extension_list)
{
	if(!GLFuncPtrName(glGetStringi) && n_GetGL30FuncPointers())
		return false;
	// requires OpenGL 3.0 functions

	r_s_extension_list.erase();

	int n_extension_num;
	glGetIntegerv(GL_NUM_EXTENSIONS, &n_extension_num);
	for(int i = 0; i < n_extension_num; ++ i) {
		const char *p_s_ext_name = (const char*)glGetStringi(GL_EXTENSIONS, i);
		if(!p_s_ext_name)
			continue; // ?
		try {
			if(i)
				r_s_extension_list += " ";
			r_s_extension_list += p_s_ext_name;
		} catch(std::bad_alloc&) {
			return false;
		}
	}
	// new OpenGL 3 extensions scheme to avoid buffer overruns in old software

	return true;
}

#if defined(_WIN32) || defined(_WIN64)

bool CGL4ExtensionHandler::b_SupportedPlatformExtension(const char *p_s_extension_name)
{
#ifndef WGL_WGLEXT_PROTOTYPES
	PFNWGLGETEXTENSIONSSTRINGARBPROC wglGetExtensionsStringARB;
	if(!(wglGetExtensionsStringARB =
	   (PFNWGLGETEXTENSIONSSTRINGARBPROC)wglGetProcAddress("wglGetExtensionsStringARB")))
		return false;
	// get wglGetExtensionsStringARB() if not linked statically
#endif // !WGL_WGLEXT_PROTOTYPES

	const char *p_s_extensions = (const char*)wglGetExtensionsStringARB(wglGetCurrentDC());
	// get WGL extensions string

	const char *p_s_search_str = p_s_extensions;
	do {
		if((p_s_search_str = strstr(p_s_search_str, p_s_extension_name))) {
			if(p_s_search_str > p_s_extensions && !isspace(*(p_s_search_str - 1)))
				continue;
			if(strlen(p_s_search_str + strlen(p_s_extension_name)) &&
			   !isspace(*(p_s_search_str + strlen(p_s_extension_name))))
				continue;

			return true;
		}
	} while(p_s_search_str && strlen(++ p_s_search_str));
	// look for the extension

	return false;
}

#else // _WIN32 || _WIN64

bool CGL4ExtensionHandler::b_SupportedPlatformExtension(const char *p_s_extension_name)
{
#ifdef GLX_VERSION_1_2
	const char *p_s_extensions;
	p_s_extensions = (const char*)glXGetClientString(glXGetCurrentDisplay(), GLX_EXTENSIONS); // @todo - debug this
	// glXGetClientString() is in GLX 1.1 and later (static linking)
	// glXGetCurrentDisplay() is in GLX 1.2 and later (static linking)
	// GLX_EXTENSIONS is in GLX 1.1
	// get GLX extensions string

	/*p_s_extensions = (const char*)glXQueryExtensionsString(glXGetCurrentDisplay(),
		DefaultScreen(glXGetCurrentDisplay()));*/
	// this could also work

	const char *p_s_search_str = p_s_extensions;
	do {
		if((p_s_search_str = strstr(p_s_search_str, p_s_extension_name))) {
			if(p_s_search_str > p_s_extensions && !isspace(*(p_s_search_str - 1)))
				continue;
			if(strlen(p_s_search_str + strlen(p_s_extension_name)) &&
			   !isspace(*(p_s_search_str + strlen(p_s_extension_name))))
				continue;

			return true;
		}
	} while(p_s_search_str && strlen(++ p_s_search_str));
	// look for the extension

#else // GLX_VERSION_1_2
#ifdef __APPLE__
	// can't expect anything from a Mac
#else // __APPLE__
#error("GLX 1.2 is needed for glXGetClientString(GLX_EXTENSIONS) to work")
#endif // __APPLE__
#endif // GLX_VERSION_1_2

	return false;
}

#endif // _WIN32 || _WIN64

int CGL4ExtensionHandler::n_GetAllOpenGLCoreFuncPointers()
{
	return n_GetGL12FuncPointers() +
		n_GetGL13FuncPointers() +
		n_GetGL14FuncPointers() +
		n_GetGL15FuncPointers() +
		n_GetGL20FuncPointers() +
		n_GetGL21FuncPointers() +
		n_GetGL30FuncPointers() +
		n_GetGL31FuncPointers() +
		n_GetGL32FuncPointers() +
		n_GetGL33FuncPointers() +
		n_GetGL40FuncPointers() +
		n_GetGL41FuncPointers() +
		n_GetGL42FuncPointers() +
		n_GetGL43FuncPointers() +
		n_GetGL44FuncPointers() +
		n_GetGL45FuncPointers();
}

int CGL4ExtensionHandler::n_GetOpenGLCoreFuncPointers(int n_major, int n_minor)
{
	int n_failed_function_num = 0;
	if(n_major > 1 || (n_major == 1 && n_minor >= 2)) n_failed_function_num += n_GetGL12FuncPointers();
	if(n_major > 1 || (n_major == 1 && n_minor >= 3)) n_failed_function_num += n_GetGL13FuncPointers();
	if(n_major > 1 || (n_major == 1 && n_minor >= 4)) n_failed_function_num += n_GetGL14FuncPointers();
	if(n_major > 1 || (n_major == 1 && n_minor >= 5)) n_failed_function_num += n_GetGL15FuncPointers();
	if(n_major > 2 || (n_major == 2 && n_minor >= 0)) n_failed_function_num += n_GetGL20FuncPointers();
	if(n_major > 2 || (n_major == 2 && n_minor >= 1)) n_failed_function_num += n_GetGL21FuncPointers();
	if(n_major > 3 || (n_major == 3 && n_minor >= 0)) n_failed_function_num += n_GetGL30FuncPointers();
	if(n_major > 3 || (n_major == 3 && n_minor >= 1)) n_failed_function_num += n_GetGL31FuncPointers();
	if(n_major > 3 || (n_major == 3 && n_minor >= 2)) n_failed_function_num += n_GetGL32FuncPointers();
	if(n_major > 3 || (n_major == 3 && n_minor >= 3)) n_failed_function_num += n_GetGL33FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 0)) n_failed_function_num += n_GetGL40FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 1)) n_failed_function_num += n_GetGL41FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 2)) n_failed_function_num += n_GetGL42FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 3)) n_failed_function_num += n_GetGL43FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 4)) n_failed_function_num += n_GetGL44FuncPointers();
	if(n_major > 4 || (n_major == 4 && n_minor >= 5)) n_failed_function_num += n_GetGL45FuncPointers();

	return n_failed_function_num;
}

int CGL4ExtensionHandler::n_GetAllSupportedExtensionsFuncPointers()
{
	return n_GetGL12FuncPointers() +

		n_GetGL13FuncPointers() +

		n_GetGL14FuncPointers() +

		n_GetGL15FuncPointers() +

		n_GetGL20FuncPointers() +

		n_GetGL21FuncPointers() +

		n_GetGL30FuncPointers() +
		//n_GetARBDepthBufferFloatFuncPointers() +
		//n_GetARBFramebufferObjectFuncPointers() +
		//n_GetARBFramebufferSRGBFuncPointers() +
		//n_GetARBHalfFloatVertexFuncPointers() +
		//n_GetARBMapBufferRangeFuncPointers() +
		//n_GetARBTextureCompressionRGtcFuncPointers() +
		//n_GetARBTextureRGFuncPointers() +
		//n_GetARBVertexArrayObjectFuncPointers() +

		n_GetGL31FuncPointers() +
		//n_GetARBUniformBufferObjectFuncPointers() +
		//n_GetARBDrawInstancedFuncPointers() +
		//n_GetARBCopyBufferFuncPointers() +
		//n_GetARBTextureBufferObjectFuncPointers() + // this is fused in n_GetGL31FuncPointers()

		n_GetGL32FuncPointers() +
		//n_GetARBDepthClampFuncPointers() +
		//n_GetARBDrawElementsBaseVertexFuncPointers() +
		//n_GetARBFragmentCoordConventionsFuncPointers() +
		//n_GetARBProvokingVertexFuncPointers() +
		//n_GetARBSeamlessCubeMapFuncPointers() +
		//n_GetARBSyncFuncPointers() +
		//n_GetARBTextureMultisampleFuncPointers() +
		//n_GetARBVertexArrayBGRAFuncPointers() +

		n_GetGL33FuncPointers() +
		//n_GetARBShaderBitEncodingFuncPointers() +
		//n_GetARBBlendFuncExtendedFuncPointers() +
		//n_GetARBExplicitAttribLocationFuncPointers() +
		//n_GetARBOcclusionQuery2FuncPointers() +
		//n_GetARBSamplerObjectsFuncPointers() +
		//n_GetARBTextureRGB10A2uiFuncPointers() +
		//n_GetARBTextureSwizzleFuncPointers() +
		//n_GetARBTimerQueryFuncPointers() +
		//n_GetARBInstancedArraysFuncPointers() +
		//n_GetARBVertexType2101010RevFuncPointers() +

		n_GetGL40FuncPointers() +
		//n_GetARBTextureQueryLodFuncPointers() +
		//n_GetARBDrawBuffersBlendFuncPointers() +
		//n_GetARBDrawIndirectFuncPointers() +
		//n_GetARBGPUShader5FuncPointers() +
		//n_GetARBGPUShaderFP64FuncPointers() +
		//n_GetARBSampleShadingFuncPointers() +
		//n_GetARBShaderSubroutineFuncPointers() +
		//n_GetARBTessellationShaderFuncPointers() +
		//n_GetARBTextureBufferObjectRGB32FuncPointers() +
		//n_GetARBTextureGatherFuncPointers() +
		//n_GetARBTextureCubeMapArrayFuncPointers() +
		//n_GetARBTransformFeedback2FuncPointers() +
		//n_GetARBTransformFeedback3FuncPointers() +

		n_GetGL41FuncPointers() +
		//n_GetARBES2CompatibilityFuncPointers() +
		//n_GetARBGetProgramBinaryFuncPointers() +
		//n_GetARBSeparateShaderObjectsFuncPointers() +
		//n_GetARBShaderPrecisionFuncPointers() +
		//n_GetARBVertexAttrib64bitFuncPointers() +
		//n_GetARBViewportArrayFuncPointers() +

		n_GetGL42FuncPointers() +
		//n_GetARBTextureCompressionBptcFuncPointers() +
		//n_GetARBCompressedTexturePixelStorageFuncPointers() +
		//n_GetARBShaderAtomicCountersFuncPointers() +
		//n_GetARBTextureStorageFuncPointers() +
		//n_GetARBTransformFeedbackInstancedFuncPointers() +
		//n_GetARBBaseInstanceFuncPointers() +
		//n_GetARBShaderImageLoadStoreFuncPointers() +
		//n_GetARBConservativeDepthFuncPointers() +
		//n_GetARBShadingLanguage_420packFuncPointers() +
		//n_GetARBInternalformatQueryFuncPointers() +
		//n_GetARBMapBufferAlignmentFuncPointers() +
		//n_GetARBShadingLanguagePackingFuncPointers() +

		n_GetGL43FuncPointers() +
		//n_GetARBArraysOfArraysFuncPointers() +
		//n_GetARBES3CompatibilityFuncPointers() +
		//n_GetARBClearBufferObjectFuncPointers() +
		//n_GetARBComputeShaderFuncPointers() +
		//n_GetARBCopyImageFuncPointers() +
		//n_GetARBDebugGroupFuncPointers() +
		//n_GetARBDebugLabelFuncPointers() +
		//n_GetARBDebugOutput2FuncPointers() +
		//n_GetARBDebugOutputFuncPointers() +
		//n_GetARBExplicitUniformLocationFuncPointers() +
		//n_GetARBFragmentLayerViewportFuncPointers() +
		//n_GetARBFramebufferNoAttachmentsFuncPointers() +
		//n_GetARBInternalformatQuery2FuncPointers() +
		//n_GetARBInvalidateSubdataFuncPointers() +
		//n_GetARBMultiDrawIndirectFuncPointers() +
		//n_GetARBProgramInterfaceQueryFuncPointers() +
		//n_GetARBRobustBufferAccessBehaviorFuncPointers() +
		//n_GetARBShaderImageSizeFuncPointers() +
		//n_GetARBShaderStorageBufferObjectFuncPointers() +
		//n_GetARBStencilTexturingFuncPointers() +
		//n_GetARBTextureBufferRangeFuncPointers() +
		//n_GetARBTextureQueryLevelsFuncPointers() +
		//n_GetARBTextureStorageMultisampleFuncPointers() +
		//n_GetARBTextureViewFuncPointers() +
		//n_GetARBVertexAttribBindingFuncPointers() +
		//n_GetKHRTextureCompressionASTC_HDRFuncPointers() +
		//n_GetARBRobustnessIsolationFuncPointers() +

		n_GetGL44FuncPointers() +
		//n_GetARBBufferStorageFuncPointers() +
		//n_GetARBClearTextureFuncPointers() +
		//n_GetARBEnhancedLayoutsFuncPointers() +
		//n_GetARBMultiBindFuncPointers() +
		//n_GetARBQueryBufferObjectFuncPointers() +
		//n_GetARBTextureMirrorClampToEdgeFuncPointers() +
		//n_GetARBTextureStencil8FuncPointers() +
		//n_GetARBVertexType10f11f11fRevFuncPointers() +

		n_GetARBBindlessTextureFuncPointers() +
		n_GetARBSparseTextureFuncPointers() +
		n_GetARBComputeVariableGroupSizeFuncPointers() +
		n_GetARBIndirectParametersFuncPointers() +
		n_GetARBSeamlessCubemapPerTextureFuncPointers() +
		n_GetARBShaderDrawParametersFuncPointers() +
		n_GetARBShaderGroupVoteFuncPointers() +
	// new GL 4.4 exts

		n_GetGL45FuncPointers() +

		n_GetEXTDrawBuffers2FuncPointers() +
		//n_GetEXTTimerQueryFuncPointers() + // removed, ARB is core in 3.3
		n_GetEXTGPUProgramParametersFuncPointers() +
		n_GetEXTBindableUniformFuncPointers() +
		n_GetARBShaderTextureLodFuncPointers() +
		n_GetS3_S3TCFuncPointers() +
		n_GetEXTTextureCompressionLATCFuncPointers() +
		n_GetEXTTextureCompressionS3TCFuncPointers() +
		n_GetEXTTextureFilterAnisotropicFuncPointers() +
		n_GetEXTABGRFuncPointers() +
		n_GetARBShadingLanguageIncludeFuncPointers() +
		n_GetARBCLEventFuncPointers() +
		n_GetARBDebugOutputFuncPointers() +
		//n_GetKHRRobustnessFuncPointers() + // gl45
		n_GetARBShaderStencilExportFuncPointers() +
		n_GetNVCommandListFuncPointers();
	// remaining older ARB exts (might need some pruning)
}

/*
 *								--- OpenGL 1.2 ---
 */

#ifdef __GENERATE_GL_12__

void (GLApi *GLFuncPtrName(glBlendColor))(GLclampf, GLclampf, GLclampf, GLclampf) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlendEquation))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawRangeElements))(GLenum, GLuint, GLuint, GLsizei, GLenum, const GLvoid*) INIT_FUNC_POINTERS;

// it is unclear whether this is deprecated (it seems to be)
/*void (GLApi *GLFuncPtrName(glColorTable))(GLenum, GLenum, GLsizei, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glColorTableParameterfv))(GLenum, GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glColorTableParameteriv))(GLenum, GLenum, const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyColorTable))(GLenum, GLenum, GLint, GLint, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetColorTable))(GLenum, GLenum, GLenum, GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetColorTableParameterfv))(GLenum, GLenum, GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetColorTableParameteriv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glColorSubTable))(GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyColorSubTable))(GLenum, GLsizei, GLint, GLint, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionFilter1D))(GLenum, GLenum, GLsizei, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionFilter2D))(GLenum, GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionParameterf))(GLenum, GLenum, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionParameterfv))(GLenum, GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionParameteri))(GLenum, GLenum, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glConvolutionParameteriv))(GLenum, GLenum, const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyConvolutionFilter1D))(GLenum, GLenum, GLint, GLint, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyConvolutionFilter2D))(GLenum, GLenum, GLint, GLint, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetConvolutionFilter))(GLenum, GLenum, GLenum, GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetConvolutionParameterfv))(GLenum, GLenum, GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetConvolutionParameteriv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSeparableFilter))(GLenum, GLenum, GLenum, GLvoid *, GLvoid *, GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSeparableFilter2D))(GLenum, GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid *, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetHistogram))(GLenum, GLboolean, GLenum, GLenum, GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetHistogramParameterfv))(GLenum, GLenum, GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetHistogramParameteriv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetMinmax))(GLenum, GLboolean, GLenum, GLenum, GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetMinmaxParameterfv))(GLenum, GLenum, GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetMinmaxParameteriv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glHistogram))(GLenum, GLsizei, GLenum, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMinmax))(GLenum, GLenum, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glResetHistogram))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glResetMinmax))(GLenum) INIT_FUNC_POINTERS;*/

void (GLApi *GLFuncPtrName(glTexImage3D))(GLenum, GLint, GLint, GLsizei, GLsizei, GLsizei, GLint, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexSubImage3D))(GLenum, GLint, GLint, GLint, GLint, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyTexSubImage3D))(GLenum, GLint, GLint, GLint, GLint, GLint, GLint, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_12__

bool GLEH_OPENGL_1_2 = false;

int CGL4ExtensionHandler::n_GetGL12FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_12__

	if(!(GLFuncPtrName(glBlendColor) = (void(GLApi*)(GLclampf, GLclampf, GLclampf, GLclampf))wglGetProcAddress("glBlendColor"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBlendEquation) = (void(GLApi*)(GLenum))wglGetProcAddress("glBlendEquation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDrawRangeElements) = (void(GLApi*)(GLenum, GLuint, GLuint, GLsizei, GLenum, const GLvoid*))wglGetProcAddress("glDrawRangeElements"))) ++ n_failed_function_num;

	// it is unclear whether this is deprecated (it seems to be)
	/*if(!(GLFuncPtrName(glColorTable) = (void(GLApi*)(GLenum, GLenum, GLsizei, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glColorTable"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glColorTableParameterfv) = (void(GLApi*)(GLenum, GLenum, const GLfloat*))wglGetProcAddress("glColorTableParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glColorTableParameteriv) = (void(GLApi*)(GLenum, GLenum, const GLint*))wglGetProcAddress("glColorTableParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCopyColorTable) = (void(GLApi*)(GLenum, GLenum, GLint, GLint, GLsizei))wglGetProcAddress("glCopyColorTable"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetColorTable) = (void(GLApi*)(GLenum, GLenum, GLenum, GLvoid*))wglGetProcAddress("glGetColorTable"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetColorTableParameterfv) = (void(GLApi*)(GLenum, GLenum, GLfloat*))wglGetProcAddress("glGetColorTableParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetColorTableParameteriv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetColorTableParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glColorSubTable) = (void(GLApi*)(GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glColorSubTable"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCopyColorSubTable) = (void(GLApi*)(GLenum, GLsizei, GLint, GLint, GLsizei))wglGetProcAddress("glCopyColorSubTable"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionFilter1D) = (void(GLApi*)(GLenum, GLenum, GLsizei, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glConvolutionFilter1D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionFilter2D) = (void(GLApi*)(GLenum, GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glConvolutionFilter2D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionParameterf) = (void(GLApi*)(GLenum, GLenum, GLfloat))wglGetProcAddress("glConvolutionParameterf"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionParameterfv) = (void(GLApi*)(GLenum, GLenum, const GLfloat*))wglGetProcAddress("glConvolutionParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionParameteri) = (void(GLApi*)(GLenum, GLenum, GLint))wglGetProcAddress("glConvolutionParameteri"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glConvolutionParameteriv) = (void(GLApi*)(GLenum, GLenum, const GLint*))wglGetProcAddress("glConvolutionParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCopyConvolutionFilter1D) = (void(GLApi*)(GLenum, GLenum, GLint, GLint, GLsizei))wglGetProcAddress("glCopyConvolutionFilter1D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCopyConvolutionFilter2D) = (void(GLApi*)(GLenum, GLenum, GLint, GLint, GLsizei, GLsizei))wglGetProcAddress("glCopyConvolutionFilter2D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetConvolutionFilter) = (void(GLApi*)(GLenum, GLenum, GLenum, GLvoid*))wglGetProcAddress("glGetConvolutionFilter"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetConvolutionParameterfv) = (void(GLApi*)(GLenum, GLenum, GLfloat*))wglGetProcAddress("glGetConvolutionParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetConvolutionParameteriv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetConvolutionParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetSeparableFilter) = (void(GLApi*)(GLenum, GLenum, GLenum, GLvoid *, GLvoid *, GLvoid*))wglGetProcAddress("glGetSeparableFilter"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSeparableFilter2D) = (void(GLApi*)(GLenum, GLenum, GLsizei, GLsizei, GLenum, GLenum, const GLvoid *, const GLvoid*))wglGetProcAddress("glSeparableFilter2D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetHistogram) = (void(GLApi*)(GLenum, GLboolean, GLenum, GLenum, GLvoid*))wglGetProcAddress("glGetHistogram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetHistogramParameterfv) = (void(GLApi*)(GLenum, GLenum, GLfloat*))wglGetProcAddress("glGetHistogramParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetHistogramParameteriv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetHistogramParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetMinmax) = (void(GLApi*)(GLenum, GLboolean, GLenum, GLenum, GLvoid*))wglGetProcAddress("glGetMinmax"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetMinmaxParameterfv) = (void(GLApi*)(GLenum, GLenum, GLfloat*))wglGetProcAddress("glGetMinmaxParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetMinmaxParameteriv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetMinmaxParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glHistogram) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLboolean))wglGetProcAddress("glHistogram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMinmax) = (void(GLApi*)(GLenum, GLenum, GLboolean))wglGetProcAddress("glMinmax"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glResetHistogram) = (void(GLApi*)(GLenum))wglGetProcAddress("glResetHistogram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glResetMinmax) = (void(GLApi*)(GLenum))wglGetProcAddress("glResetMinmax"))) ++ n_failed_function_num;*/

	if(!(GLFuncPtrName(glTexImage3D) = (void(GLApi*)(GLenum, GLint, GLint, GLsizei, GLsizei, GLsizei, GLint, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glTexImage3D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glTexSubImage3D) = (void(GLApi*)(GLenum, GLint, GLint, GLint, GLint, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const GLvoid*))wglGetProcAddress("glTexSubImage3D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCopyTexSubImage3D) = (void(GLApi*)(GLenum, GLint, GLint, GLint, GLint, GLint, GLint, GLsizei, GLsizei))wglGetProcAddress("glCopyTexSubImage3D"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_12__

	GLEH_OPENGL_1_2 = !n_failed_function_num && b_Support_OpenGL(1, 2);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 1.2 ---
 */

/*
 *								--- OpenGL 1.3 ---
 */

#ifdef __GENERATE_GL_13__

// functions below are deprecated
/*void (GLApi *GLFuncPtrName(glLoadTransposeMatrixf))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glLoadTransposeMatrixd))(const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultTransposeMatrixf))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultTransposeMatrixd))(const GLdouble*) INIT_FUNC_POINTERS;*/

void (GLApi *GLFuncPtrName(glActiveTexture))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSampleCoverage))(GLclampf, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexImage3D))(GLenum, GLint, GLenum, GLsizei, GLsizei, GLsizei, GLint, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexImage2D))(GLenum, GLint, GLenum, GLsizei, GLsizei, GLint, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexImage1D))(GLenum, GLint, GLenum, GLsizei, GLint, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexSubImage3D))(GLenum, GLint, GLint, GLint, GLint, GLsizei, GLsizei, GLsizei, GLenum, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexSubImage2D))(GLenum, GLint, GLint, GLint, GLsizei, GLsizei, GLenum, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTexSubImage1D))(GLenum, GLint, GLint, GLsizei, GLenum, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetCompressedTexImage))(GLenum, GLint, GLvoid*) INIT_FUNC_POINTERS;

// functions below are deprecated
/*void (GLApi *GLFuncPtrName(glClientActiveTexture))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1d))(GLenum, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1dv))(GLenum, const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1f))(GLenum, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1fv))(GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1i))(GLenum, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1iv))(GLenum, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1s))(GLenum, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord1sv))(GLenum, const GLshort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2d))(GLenum, GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2dv))(GLenum, const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2f))(GLenum, GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2fv))(GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2i))(GLenum, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2iv))(GLenum, const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2s))(GLenum, GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord2sv))(GLenum, const GLshort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3d))(GLenum, GLdouble, GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3dv))(GLenum, const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3f))(GLenum, GLfloat, GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3fv))(GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3i))(GLenum, GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3iv))(GLenum, const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3s))(GLenum, GLshort, GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord3sv))(GLenum, const GLshort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4d))(GLenum, GLdouble, GLdouble, GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4dv))(GLenum, const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4f))(GLenum, GLfloat, GLfloat, GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4fv))(GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4i))(GLenum, GLint, GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4iv))(GLenum, const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4s))(GLenum, GLshort, GLshort, GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiTexCoord4sv))(GLenum, const GLshort*) INIT_FUNC_POINTERS;*/

#endif // __GENERATE_GL_13__

bool GLEH_OPENGL_1_3 = false;

int CGL4ExtensionHandler::n_GetGL13FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_13__

	// functions below are deprecated
	/*if(!(GLFuncPtrName(glLoadTransposeMatrixf) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glLoadTransposeMatrixf"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glLoadTransposeMatrixd) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glLoadTransposeMatrixd"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultTransposeMatrixf) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glMultTransposeMatrixf"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultTransposeMatrixd) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glMultTransposeMatrixd"))) ++ n_failed_function_num;*/

	if(!(GLFuncPtrName(glActiveTexture) = (void(GLApi*)(GLenum))wglGetProcAddress("glActiveTexture"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSampleCoverage) = (void(GLApi*)(GLclampf, GLboolean))wglGetProcAddress("glSampleCoverage"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexImage3D) = (void(GLApi*)(GLenum, GLint, GLenum, GLsizei, GLsizei, GLsizei, GLint, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexImage3D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexImage2D) = (void(GLApi*)(GLenum, GLint, GLenum, GLsizei, GLsizei, GLint, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexImage2D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexImage1D) = (void(GLApi*)(GLenum, GLint, GLenum, GLsizei, GLint, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexImage1D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexSubImage3D) = (void(GLApi*)(GLenum, GLint, GLint, GLint, GLint, GLsizei, GLsizei, GLsizei, GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexSubImage3D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexSubImage2D) = (void(GLApi*)(GLenum, GLint, GLint, GLint, GLsizei, GLsizei, GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexSubImage2D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompressedTexSubImage1D) = (void(GLApi*)(GLenum, GLint, GLint, GLsizei, GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glCompressedTexSubImage1D"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetCompressedTexImage) = (void(GLApi*)(GLenum, GLint, GLvoid*))wglGetProcAddress("glGetCompressedTexImage"))) ++ n_failed_function_num;

	// functions below are deprecated
	/*if(!(GLFuncPtrName(glClientActiveTexture) = (void(GLApi*)(GLenum))wglGetProcAddress("glClientActiveTexture"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1d) = (void(GLApi*)(GLenum, GLdouble))wglGetProcAddress("glMultiTexCoord1d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1dv) = (void(GLApi*)(GLenum, const GLdouble*))wglGetProcAddress("glMultiTexCoord1dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1f) = (void(GLApi*)(GLenum, GLfloat))wglGetProcAddress("glMultiTexCoord1f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1fv) = (void(GLApi*)(GLenum, const GLfloat*))wglGetProcAddress("glMultiTexCoord1fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1i) = (void(GLApi*)(GLenum, GLint))wglGetProcAddress("glMultiTexCoord1i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1iv) = (void(GLApi*)(GLenum, const GLint *))wglGetProcAddress("glMultiTexCoord1iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1s) = (void(GLApi*)(GLenum, GLshort))wglGetProcAddress("glMultiTexCoord1s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord1sv) = (void(GLApi*)(GLenum, const GLshort*))wglGetProcAddress("glMultiTexCoord1sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2d) = (void(GLApi*)(GLenum, GLdouble, GLdouble))wglGetProcAddress("glMultiTexCoord2d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2dv) = (void(GLApi*)(GLenum, const GLdouble*))wglGetProcAddress("glMultiTexCoord2dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2f) = (void(GLApi*)(GLenum, GLfloat, GLfloat))wglGetProcAddress("glMultiTexCoord2f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2fv) = (void(GLApi*)(GLenum, const GLfloat*))wglGetProcAddress("glMultiTexCoord2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2i) = (void(GLApi*)(GLenum, GLint, GLint))wglGetProcAddress("glMultiTexCoord2i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2iv) = (void(GLApi*)(GLenum, const GLint*))wglGetProcAddress("glMultiTexCoord2iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2s) = (void(GLApi*)(GLenum, GLshort, GLshort))wglGetProcAddress("glMultiTexCoord2s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord2sv) = (void(GLApi*)(GLenum, const GLshort*))wglGetProcAddress("glMultiTexCoord2sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3d) = (void(GLApi*)(GLenum, GLdouble, GLdouble, GLdouble))wglGetProcAddress("glMultiTexCoord3d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3dv) = (void(GLApi*)(GLenum, const GLdouble*))wglGetProcAddress("glMultiTexCoord3dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3f) = (void(GLApi*)(GLenum, GLfloat, GLfloat, GLfloat))wglGetProcAddress("glMultiTexCoord3f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3fv) = (void(GLApi*)(GLenum, const GLfloat*))wglGetProcAddress("glMultiTexCoord3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3i) = (void(GLApi*)(GLenum, GLint, GLint, GLint))wglGetProcAddress("glMultiTexCoord3i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3iv) = (void(GLApi*)(GLenum, const GLint*))wglGetProcAddress("glMultiTexCoord3iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3s) = (void(GLApi*)(GLenum, GLshort, GLshort, GLshort))wglGetProcAddress("glMultiTexCoord3s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord3sv) = (void(GLApi*)(GLenum, const GLshort*))wglGetProcAddress("glMultiTexCoord3sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4d) = (void(GLApi*)(GLenum, GLdouble, GLdouble, GLdouble, GLdouble))wglGetProcAddress("glMultiTexCoord4d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4dv) = (void(GLApi*)(GLenum, const GLdouble*))wglGetProcAddress("glMultiTexCoord4dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4f) = (void(GLApi*)(GLenum, GLfloat, GLfloat, GLfloat, GLfloat))wglGetProcAddress("glMultiTexCoord4f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4fv) = (void(GLApi*)(GLenum, const GLfloat*))wglGetProcAddress("glMultiTexCoord4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4i) = (void(GLApi*)(GLenum, GLint, GLint, GLint, GLint))wglGetProcAddress("glMultiTexCoord4i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4iv) = (void(GLApi*)(GLenum, const GLint*))wglGetProcAddress("glMultiTexCoord4iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4s) = (void(GLApi*)(GLenum, GLshort, GLshort, GLshort, GLshort))wglGetProcAddress("glMultiTexCoord4s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiTexCoord4sv) = (void(GLApi*)(GLenum, const GLshort*))wglGetProcAddress("glMultiTexCoord4sv"))) ++ n_failed_function_num;*/

#endif // __GENERATE_GL_13__

	GLEH_OPENGL_1_3 = !n_failed_function_num && b_Support_OpenGL(1, 3);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 1.3 ---
 */

/*
 *								--- OpenGL 1.4 ---
 */

#ifdef __GENERATE_GL_14__

void (GLApi *GLFuncPtrName(glBlendFuncSeparate))(GLenum, GLenum, GLenum, GLenum) INIT_FUNC_POINTERS;

// functions below are deprecated
/*void (GLApi *GLFuncPtrName(glFogCoordf))(GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFogCoordfv))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFogCoordd))(GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFogCoorddv))(const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFogCoordPointer))(GLenum, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;*/

void (GLApi *GLFuncPtrName(glMultiDrawArrays))(GLenum, GLint *, GLsizei *, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiDrawElements))(GLenum, const GLsizei *, GLenum, const GLvoid* *, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPointParameterf))(GLenum, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPointParameterfv))(GLenum, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPointParameteri))(GLenum, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPointParameteriv))(GLenum, const GLint*) INIT_FUNC_POINTERS;

// functions below are deprecated
/*void (GLApi *GLFuncPtrName(glSecondaryColor3b))(GLbyte, GLbyte, GLbyte) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3bv))(const GLbyte*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3d))(GLdouble, GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3dv))(const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3f))(GLfloat, GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3fv))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3i))(GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3iv))(const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3s))(GLshort, GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3sv))(const GLshort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3ub))(GLubyte, GLubyte, GLubyte) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3ubv))(const GLubyte*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3ui))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3uiv))(const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3us))(GLushort, GLushort, GLushort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColor3usv))(const GLushort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSecondaryColorPointer))(GLint, GLenum, GLsizei, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2d))(GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2dv))(const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2f))(GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2fv))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2i))(GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2iv))(const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2s))(GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos2sv))(const GLshort*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3d))(GLdouble, GLdouble, GLdouble) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3dv))(const GLdouble*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3f))(GLfloat, GLfloat, GLfloat) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3fv))(const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3i))(GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3iv))(const GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3s))(GLshort, GLshort, GLshort) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWindowPos3sv))(const GLshort*) INIT_FUNC_POINTERS;*/

#endif // __GENERATE_GL_14__

bool GLEH_OPENGL_1_4 = false;

int CGL4ExtensionHandler::n_GetGL14FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_14__

	if(!(GLFuncPtrName(glBlendFuncSeparate) = (void(GLApi*)(GLenum, GLenum, GLenum, GLenum))wglGetProcAddress("glBlendFuncSeparate"))) ++ n_failed_function_num;

	// functions below are deprecated
	/*if(!(GLFuncPtrName(glFogCoordf) = (void(GLApi*)(GLfloat))wglGetProcAddress("glFogCoordf"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glFogCoordfv) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glFogCoordfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glFogCoordd) = (void(GLApi*)(GLdouble))wglGetProcAddress("glFogCoordd"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glFogCoorddv) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glFogCoorddv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glFogCoordPointer) = (void(GLApi*)(GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glFogCoordPointer"))) ++ n_failed_function_num;*/

	if(!(GLFuncPtrName(glMultiDrawArrays) = (void(GLApi*)(GLenum, GLint *, GLsizei *, GLsizei))wglGetProcAddress("glMultiDrawArrays"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMultiDrawElements) = (void(GLApi*)(GLenum, const GLsizei *, GLenum, const GLvoid* *, GLsizei))wglGetProcAddress("glMultiDrawElements"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glPointParameterf) = (void(GLApi*)(GLenum, GLfloat))wglGetProcAddress("glPointParameterf"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glPointParameterfv) = (void(GLApi*)(GLenum, const GLfloat*))wglGetProcAddress("glPointParameterfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glPointParameteri) = (void(GLApi*)(GLenum, GLint))wglGetProcAddress("glPointParameteri"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glPointParameteriv) = (void(GLApi*)(GLenum, const GLint*))wglGetProcAddress("glPointParameteriv"))) ++ n_failed_function_num;

	// functions below are deprecated
	/*if(!(GLFuncPtrName(glSecondaryColor3b) = (void(GLApi*)(GLbyte, GLbyte, GLbyte))wglGetProcAddress("glSecondaryColor3b"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3bv) = (void(GLApi*)(const GLbyte*))wglGetProcAddress("glSecondaryColor3bv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3d) = (void(GLApi*)(GLdouble, GLdouble, GLdouble))wglGetProcAddress("glSecondaryColor3d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3dv) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glSecondaryColor3dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3f) = (void(GLApi*)(GLfloat, GLfloat, GLfloat))wglGetProcAddress("glSecondaryColor3f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3fv) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glSecondaryColor3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3i) = (void(GLApi*)(GLint, GLint, GLint))wglGetProcAddress("glSecondaryColor3i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3iv) = (void(GLApi*)(const GLint*))wglGetProcAddress("glSecondaryColor3iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3s) = (void(GLApi*)(GLshort, GLshort, GLshort))wglGetProcAddress("glSecondaryColor3s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3sv) = (void(GLApi*)(const GLshort*))wglGetProcAddress("glSecondaryColor3sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3ub) = (void(GLApi*)(GLubyte, GLubyte, GLubyte))wglGetProcAddress("glSecondaryColor3ub"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3ubv) = (void(GLApi*)(const GLubyte*))wglGetProcAddress("glSecondaryColor3ubv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3ui) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glSecondaryColor3ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3uiv) = (void(GLApi*)(const GLuint*))wglGetProcAddress("glSecondaryColor3uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3us) = (void(GLApi*)(GLushort, GLushort, GLushort))wglGetProcAddress("glSecondaryColor3us"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColor3usv) = (void(GLApi*)(const GLushort*))wglGetProcAddress("glSecondaryColor3usv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glSecondaryColorPointer) = (void(GLApi*)(GLint, GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glSecondaryColorPointer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2d) = (void(GLApi*)(GLdouble, GLdouble))wglGetProcAddress("glWindowPos2d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2dv) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glWindowPos2dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2f) = (void(GLApi*)(GLfloat, GLfloat))wglGetProcAddress("glWindowPos2f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2fv) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glWindowPos2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2i) = (void(GLApi*)(GLint, GLint))wglGetProcAddress("glWindowPos2i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2iv) = (void(GLApi*)(const GLint*))wglGetProcAddress("glWindowPos2iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2s) = (void(GLApi*)(GLshort, GLshort))wglGetProcAddress("glWindowPos2s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos2sv) = (void(GLApi*)(const GLshort*))wglGetProcAddress("glWindowPos2sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3d) = (void(GLApi*)(GLdouble, GLdouble, GLdouble))wglGetProcAddress("glWindowPos3d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3dv) = (void(GLApi*)(const GLdouble*))wglGetProcAddress("glWindowPos3dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3f) = (void(GLApi*)(GLfloat, GLfloat, GLfloat))wglGetProcAddress("glWindowPos3f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3fv) = (void(GLApi*)(const GLfloat*))wglGetProcAddress("glWindowPos3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3i) = (void(GLApi*)(GLint, GLint, GLint))wglGetProcAddress("glWindowPos3i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3iv) = (void(GLApi*)(const GLint*))wglGetProcAddress("glWindowPos3iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3s) = (void(GLApi*)(GLshort, GLshort, GLshort))wglGetProcAddress("glWindowPos3s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glWindowPos3sv) = (void(GLApi*)(const GLshort*))wglGetProcAddress("glWindowPos3sv"))) ++ n_failed_function_num;*/

#endif // __GENERATE_GL_14__

	GLEH_OPENGL_1_4 = !n_failed_function_num && b_Support_OpenGL(1, 4);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 1.4 ---
 */

/*
 *								--- OpenGL 1.5 ---
 */

#ifdef __GENERATE_GL_15__

void (GLApi *GLFuncPtrName(glGenQueries))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteQueries))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsQuery))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBeginQuery))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEndQuery))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryiv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryObjectiv))(GLuint, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryObjectuiv))(GLuint, GLenum, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindBuffer))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteBuffers))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenBuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsBuffer))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBufferData))(GLenum, GLsizeiptr, const GLvoid *, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBufferSubData))(GLenum, GLintptr, GLsizeiptr, const GLvoid*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBufferSubData))(GLenum, GLintptr, GLsizeiptr, GLvoid*) INIT_FUNC_POINTERS;
GLvoid* (GLApi *GLFuncPtrName(glMapBuffer))(GLenum, GLenum) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glUnmapBuffer))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBufferParameteriv))(GLenum, GLenum, GLint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBufferPointerv))(GLenum, GLenum, GLvoid**) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_15__

bool GLEH_OPENGL_1_5 = false;

int CGL4ExtensionHandler::n_GetGL15FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_15__

	if(!(GLFuncPtrName(glGenQueries) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenQueries"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDeleteQueries) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteQueries"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glIsQuery) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsQuery"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBeginQuery) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glBeginQuery"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glEndQuery) = (void(GLApi*)(GLenum))wglGetProcAddress("glEndQuery"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetQueryiv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetQueryiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetQueryObjectiv) = (void(GLApi*)(GLuint, GLenum, GLint*))wglGetProcAddress("glGetQueryObjectiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetQueryObjectuiv) = (void(GLApi*)(GLuint, GLenum, GLuint*))wglGetProcAddress("glGetQueryObjectuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBindBuffer) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glBindBuffer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDeleteBuffers) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteBuffers"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGenBuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenBuffers"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glIsBuffer) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsBuffer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBufferData) = (void(GLApi*)(GLenum, GLsizeiptr, const GLvoid *, GLenum))wglGetProcAddress("glBufferData"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBufferSubData) = (void(GLApi*)(GLenum, GLintptr, GLsizeiptr, const GLvoid*))wglGetProcAddress("glBufferSubData"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetBufferSubData) = (void(GLApi*)(GLenum, GLintptr, GLsizeiptr, GLvoid*))wglGetProcAddress("glGetBufferSubData"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glMapBuffer) = (GLvoid*(GLApi*)(GLenum, GLenum))wglGetProcAddress("glMapBuffer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUnmapBuffer) = (GLboolean(GLApi*)(GLenum))wglGetProcAddress("glUnmapBuffer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetBufferParameteriv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetBufferParameteriv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetBufferPointerv) = (void(GLApi*)(GLenum, GLenum, GLvoid**))wglGetProcAddress("glGetBufferPointerv"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_15__

	GLEH_OPENGL_1_5 = !n_failed_function_num && b_Support_OpenGL(1, 5);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 1.5 ---
 */

/*
 *								--- OpenGL 2.0 ---
 */

#ifdef __GENERATE_GL_20__

void (GLApi *GLFuncPtrName(glDeleteShader))(GLuint shader) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDetachShader))(GLuint program, GLuint shader) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glCreateShader))(GLenum type) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glShaderSource))(GLuint shader, GLsizei count, const GLchar* *string, const GLint *length) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompileShader))(GLuint shader) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glCreateProgram))(void) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glAttachShader))(GLuint program, GLuint shader) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glLinkProgram))(GLuint program) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUseProgram))(GLuint program) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteProgram))(GLuint program) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glValidateProgram))(GLuint program) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1f))(GLint location, GLfloat v0) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2f))(GLint location, GLfloat v0, GLfloat v1) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3f))(GLint location, GLfloat v0, GLfloat v1, GLfloat v2) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4f))(GLint location, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1i))(GLint location, GLint v0) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2i))(GLint location, GLint v0, GLint v1) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3i))(GLint location, GLint v0, GLint v1, GLint v2) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4i))(GLint location, GLint v0, GLint v1, GLint v2, GLint v3) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1fv))(GLint location, GLsizei count, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2fv))(GLint location, GLsizei count, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3fv))(GLint location, GLsizei count, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4fv))(GLint location, GLsizei count, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1iv))(GLint location, GLsizei count, const GLint *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2iv))(GLint location, GLsizei count, const GLint *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3iv))(GLint location, GLsizei count, const GLint *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4iv))(GLint location, GLsizei count, const GLint *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix2fv))(GLint location, GLsizei count, GLboolean transpose, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3fv))(GLint location, GLsizei count, GLboolean transpose, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4fv))(GLint location, GLsizei count, GLboolean transpose, const GLfloat *value) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindAttribLocation))(GLuint program, GLuint index, const GLchar *name) INIT_FUNC_POINTERS;
GLint (GLApi *GLFuncPtrName(glGetAttribLocation))(GLuint program, const GLchar *name) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1d))(GLuint index, GLdouble x) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1dv))(GLuint index, const GLdouble *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1f))(GLuint index, GLfloat x) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1fv))(GLuint index, const GLfloat *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1s))(GLuint index, GLshort x) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib1sv))(GLuint index, const GLshort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2d))(GLuint index, GLdouble x, GLdouble y) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2dv))(GLuint index, const GLdouble *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2f))(GLuint index, GLfloat x, GLfloat y) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2fv))(GLuint index, const GLfloat *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2s))(GLuint index, GLshort x, GLshort y) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib2sv))(GLuint index, const GLshort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3d))(GLuint index, GLdouble x, GLdouble y, GLdouble z) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3dv))(GLuint index, const GLdouble *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3f))(GLuint index, GLfloat x, GLfloat y, GLfloat z) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3fv))(GLuint index, const GLfloat *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3s))(GLuint index, GLshort x, GLshort y, GLshort z) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib3sv))(GLuint index, const GLshort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nbv))(GLuint index, const GLbyte *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Niv))(GLuint index, const GLint *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nsv))(GLuint index, const GLshort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nub))(GLuint index, GLubyte x, GLubyte y, GLubyte z, GLubyte w) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nubv))(GLuint index, const GLubyte *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nuiv))(GLuint index, const GLuint *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4Nusv))(GLuint index, const GLushort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4bv))(GLuint index, const GLbyte *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4d))(GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4dv))(GLuint index, const GLdouble *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4f))(GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4fv))(GLuint index, const GLfloat *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4iv))(GLuint index, const GLint *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4s))(GLuint index, GLshort x, GLshort y, GLshort z, GLshort w) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4sv))(GLuint index, const GLshort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4ubv))(GLuint index, const GLubyte *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4uiv))(GLuint index, const GLuint *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttrib4usv))(GLuint index, const GLushort *v) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribPointer))(GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, const GLvoid *pointer) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEnableVertexAttribArray))(GLuint index) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDisableVertexAttribArray))(GLuint index) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribdv))(GLuint index, GLenum pname, GLdouble *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribfv))(GLuint index, GLenum pname, GLfloat *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribiv))(GLuint index, GLenum pname, GLint *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribPointerv))(GLuint index, GLenum pname, GLvoid* *pointer) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsShader))(GLuint shader) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsProgram))(GLuint program) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetShaderiv))(GLuint program, GLenum pname, GLint *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramiv))(GLuint program, GLenum pname, GLint *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetAttachedShaders))(GLuint program, GLsizei maxCount, GLsizei *count, GLuint *shaders) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetShaderInfoLog))(GLuint shader, GLsizei bufSize, GLsizei *length, GLchar *infoLog) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramInfoLog))(GLuint program, GLsizei bufSize, GLsizei *length, GLchar *infoLog) INIT_FUNC_POINTERS;
GLint (GLApi *GLFuncPtrName(glGetUniformLocation))(GLuint program, const GLchar *name) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveUniform))(GLuint program, GLuint index, GLsizei bufSize, GLsizei *length, GLsizei *size, GLenum *type, GLchar *name) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetUniformfv))(GLuint program, GLint location, GLfloat *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetUniformiv))(GLuint program, GLint location, GLint *params) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetShaderSource))(GLuint shader, GLsizei bufSize, GLsizei *length, GLchar *source) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveAttrib))(GLuint program, GLuint index, GLsizei bufSize, GLsizei *length, GLsizei *size, GLenum *type, GLchar *name) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawBuffers))(GLsizei n, const GLenum *bufs) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glStencilFuncSeparate))(GLenum face, GLenum func, GLint ref, GLuint mask) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glStencilOpSeparate))(GLenum face, GLenum fail, GLenum zfail, GLenum zpass) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glStencilMaskSeparate))(GLenum face, GLuint mask) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlendEquationSeparate))(GLenum modeRGB, GLenum modeAlpha) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_20__

bool GLEH_OPENGL_2_0 = false;

int CGL4ExtensionHandler::n_GetGL20FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_20__

	if(!(GLFuncPtrName(glDeleteShader) = (void(GLApi*)(GLuint))wglGetProcAddress("glDeleteShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDetachShader) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glDetachShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCreateShader) = (GLuint(GLApi*)(GLenum))wglGetProcAddress("glCreateShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glShaderSource) = (void(GLApi*)(GLuint, GLsizei, const GLchar**, const GLint*))wglGetProcAddress("glShaderSource"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCompileShader) = (void(GLApi*)(GLuint))wglGetProcAddress("glCompileShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glCreateProgram) = (GLuint(GLApi*)(void))wglGetProcAddress("glCreateProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glAttachShader) = (void(GLApi*)(GLuint, GLuint shader))wglGetProcAddress("glAttachShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glLinkProgram) = (void(GLApi*)(GLuint))wglGetProcAddress("glLinkProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUseProgram) = (void(GLApi*)(GLuint))wglGetProcAddress("glUseProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDeleteProgram) = (void(GLApi*)(GLuint))wglGetProcAddress("glDeleteProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glValidateProgram) = (void(GLApi*)(GLuint))wglGetProcAddress("glValidateProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1f) = (void(GLApi*)(GLint, GLfloat))wglGetProcAddress("glUniform1f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2f) = (void(GLApi*)(GLint, GLfloat, GLfloat))wglGetProcAddress("glUniform2f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3f) = (void(GLApi*)(GLint, GLfloat, GLfloat, GLfloat))wglGetProcAddress("glUniform3f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4f) = (void(GLApi*)(GLint, GLfloat, GLfloat, GLfloat, GLfloat))wglGetProcAddress("glUniform4f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1i) = (void(GLApi*)(GLint, GLint))wglGetProcAddress("glUniform1i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2i) = (void(GLApi*)(GLint, GLint, GLint))wglGetProcAddress("glUniform2i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3i) = (void(GLApi*)(GLint, GLint, GLint, GLint))wglGetProcAddress("glUniform3i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4i) = (void(GLApi*)(GLint, GLint, GLint, GLint, GLint))wglGetProcAddress("glUniform4i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1fv) = (void(GLApi*)(GLint, GLsizei, const GLfloat*))wglGetProcAddress("glUniform1fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2fv) = (void(GLApi*)(GLint, GLsizei, const GLfloat*))wglGetProcAddress("glUniform2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3fv) = (void(GLApi*)(GLint, GLsizei, const GLfloat*))wglGetProcAddress("glUniform3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4fv) = (void(GLApi*)(GLint, GLsizei, const GLfloat*))wglGetProcAddress("glUniform4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1iv) = (void(GLApi*)(GLint, GLsizei, const GLint*))wglGetProcAddress("glUniform1iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2iv) = (void(GLApi*)(GLint, GLsizei, const GLint*))wglGetProcAddress("glUniform2iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3iv) = (void(GLApi*)(GLint, GLsizei, const GLint*))wglGetProcAddress("glUniform3iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4iv) = (void(GLApi*)(GLint, GLsizei, const GLint*))wglGetProcAddress("glUniform4iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix2fv) = (void(GLApi*)(GLint, GLsizei, GLboolean transpose, const GLfloat*))wglGetProcAddress("glUniformMatrix2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix3fv) = (void(GLApi*)(GLint, GLsizei, GLboolean transpose, const GLfloat*))wglGetProcAddress("glUniformMatrix3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix4fv) = (void(GLApi*)(GLint, GLsizei, GLboolean transpose, const GLfloat*))wglGetProcAddress("glUniformMatrix4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBindAttribLocation) = (void(GLApi*)(GLuint, GLuint, const GLchar *name))wglGetProcAddress("glBindAttribLocation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetAttribLocation) = (GLint(GLApi*)(GLuint, const GLchar *name))wglGetProcAddress("glGetAttribLocation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1d) = (void(GLApi*)(GLuint, GLdouble))wglGetProcAddress("glVertexAttrib1d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1dv) = (void(GLApi*)(GLuint, const GLdouble*))wglGetProcAddress("glVertexAttrib1dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1f) = (void(GLApi*)(GLuint, GLfloat))wglGetProcAddress("glVertexAttrib1f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1fv) = (void(GLApi*)(GLuint, const GLfloat*))wglGetProcAddress("glVertexAttrib1fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1s) = (void(GLApi*)(GLuint, GLshort))wglGetProcAddress("glVertexAttrib1s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib1sv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttrib1sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2d) = (void(GLApi*)(GLuint, GLdouble, GLdouble))wglGetProcAddress("glVertexAttrib2d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2dv) = (void(GLApi*)(GLuint, const GLdouble*))wglGetProcAddress("glVertexAttrib2dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2f) = (void(GLApi*)(GLuint, GLfloat, GLfloat))wglGetProcAddress("glVertexAttrib2f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2fv) = (void(GLApi*)(GLuint, const GLfloat*))wglGetProcAddress("glVertexAttrib2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2s) = (void(GLApi*)(GLuint, GLshort, GLshort))wglGetProcAddress("glVertexAttrib2s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib2sv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttrib2sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3d) = (void(GLApi*)(GLuint, GLdouble, GLdouble, GLdouble))wglGetProcAddress("glVertexAttrib3d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3dv) = (void(GLApi*)(GLuint, const GLdouble*))wglGetProcAddress("glVertexAttrib3dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3f) = (void(GLApi*)(GLuint, GLfloat, GLfloat, GLfloat z))wglGetProcAddress("glVertexAttrib3f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3fv) = (void(GLApi*)(GLuint, const GLfloat*))wglGetProcAddress("glVertexAttrib3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3s) = (void(GLApi*)(GLuint, GLshort, GLshort, GLshort z))wglGetProcAddress("glVertexAttrib3s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib3sv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttrib3sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nbv) = (void(GLApi*)(GLuint, const GLbyte*))wglGetProcAddress("glVertexAttrib4Nbv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Niv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttrib4Niv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nsv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttrib4Nsv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nub) = (void(GLApi*)(GLuint, GLubyte, GLubyte, GLubyte, GLubyte))wglGetProcAddress("glVertexAttrib4Nub"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nubv) = (void(GLApi*)(GLuint, const GLubyte*))wglGetProcAddress("glVertexAttrib4Nubv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nuiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttrib4Nuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4Nusv) = (void(GLApi*)(GLuint, const GLushort*))wglGetProcAddress("glVertexAttrib4Nusv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4bv) = (void(GLApi*)(GLuint, const GLbyte*))wglGetProcAddress("glVertexAttrib4bv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4d) = (void(GLApi*)(GLuint, GLdouble, GLdouble, GLdouble, GLdouble))wglGetProcAddress("glVertexAttrib4d"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4dv) = (void(GLApi*)(GLuint, const GLdouble*))wglGetProcAddress("glVertexAttrib4dv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4f) = (void(GLApi*)(GLuint, GLfloat, GLfloat, GLfloat, GLfloat))wglGetProcAddress("glVertexAttrib4f"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4fv) = (void(GLApi*)(GLuint, const GLfloat*))wglGetProcAddress("glVertexAttrib4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4iv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttrib4iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4s) = (void(GLApi*)(GLuint, GLshort, GLshort, GLshort, GLshort))wglGetProcAddress("glVertexAttrib4s"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4sv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttrib4sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4ubv) = (void(GLApi*)(GLuint, const GLubyte*))wglGetProcAddress("glVertexAttrib4ubv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4uiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttrib4uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttrib4usv) = (void(GLApi*)(GLuint, const GLushort*))wglGetProcAddress("glVertexAttrib4usv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribPointer) = (void(GLApi*)(GLuint, GLint, GLenum, GLboolean, GLsizei, const GLvoid*))wglGetProcAddress("glVertexAttribPointer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glEnableVertexAttribArray) = (void(GLApi*)(GLuint index))wglGetProcAddress("glEnableVertexAttribArray"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDisableVertexAttribArray) = (void(GLApi*)(GLuint index))wglGetProcAddress("glDisableVertexAttribArray"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribdv) = (void(GLApi*)(GLuint, GLenum, GLdouble*))wglGetProcAddress("glGetVertexAttribdv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribfv) = (void(GLApi*)(GLuint, GLenum, GLfloat*))wglGetProcAddress("glGetVertexAttribfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribiv) = (void(GLApi*)(GLuint, GLenum, GLint*))wglGetProcAddress("glGetVertexAttribiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribPointerv) = (void(GLApi*)(GLuint, GLenum, GLvoid**))wglGetProcAddress("glGetVertexAttribPointerv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glIsShader) = (GLboolean(GLApi*)(GLuint shader))wglGetProcAddress("glIsShader"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glIsProgram) = (GLboolean(GLApi*)(GLuint program))wglGetProcAddress("glIsProgram"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetShaderiv) = (void(GLApi*)(GLuint, GLenum, GLint*))wglGetProcAddress("glGetShaderiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetProgramiv) = (void(GLApi*)(GLuint, GLenum, GLint*))wglGetProcAddress("glGetProgramiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetAttachedShaders) = (void(GLApi*)(GLuint, GLsizei maxCount, GLsizei *count, GLuint*))wglGetProcAddress("glGetAttachedShaders"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetShaderInfoLog) = (void(GLApi*)(GLuint shader, GLsizei bufSize, GLsizei*, GLchar*))wglGetProcAddress("glGetShaderInfoLog"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetProgramInfoLog) = (void(GLApi*)(GLuint, GLsizei bufSize, GLsizei*, GLchar*))wglGetProcAddress("glGetProgramInfoLog"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetUniformLocation) = (GLint(GLApi*)(GLuint, const GLchar *name))wglGetProcAddress("glGetUniformLocation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetActiveUniform) = (void(GLApi*)(GLuint, GLuint, GLsizei bufSize, GLsizei*, GLsizei*, GLenum*, GLchar*))wglGetProcAddress("glGetActiveUniform"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetUniformfv) = (void(GLApi*)(GLuint, GLint, GLfloat*))wglGetProcAddress("glGetUniformfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetUniformiv) = (void(GLApi*)(GLuint, GLint, GLint*))wglGetProcAddress("glGetUniformiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetShaderSource) = (void(GLApi*)(GLuint, GLsizei bufSize, GLsizei*, GLchar*))wglGetProcAddress("glGetShaderSource"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetActiveAttrib) = (void(GLApi*)(GLuint, GLuint, GLsizei bufSize, GLsizei*, GLsizei*, GLenum*, GLchar*))wglGetProcAddress("glGetActiveAttrib"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDrawBuffers) = (void(GLApi*)(GLsizei, const GLenum*))wglGetProcAddress("glDrawBuffers"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glStencilFuncSeparate) = (void(GLApi*)(GLenum, GLenum, GLint, GLuint))wglGetProcAddress("glStencilFuncSeparate"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glStencilOpSeparate) = (void(GLApi*)(GLenum, GLenum, GLenum, GLenum))wglGetProcAddress("glStencilOpSeparate"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glStencilMaskSeparate) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glStencilMaskSeparate"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBlendEquationSeparate) = (void(GLApi*)(GLenum, GLenum))wglGetProcAddress("glBlendEquationSeparate"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_20__

	GLEH_OPENGL_2_0 = !n_failed_function_num && b_Support_OpenGL(2, 0);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 2.0 ---
 */

/*
 *								--- OpenGL 2.1 ---
 */

#ifdef __GENERATE_GL_21__

void (GLApi *GLFuncPtrName(glUniformMatrix2x3fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3x2fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix2x4fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4x2fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3x4fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4x3fv))(GLint, GLsizei, GLboolean, const GLfloat*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_21__

bool GLEH_OPENGL_2_1 = false;

int CGL4ExtensionHandler::n_GetGL21FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_21__

	if(!(GLFuncPtrName(glUniformMatrix2x3fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix2x3fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix3x2fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix3x2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix2x4fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix2x4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix4x2fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix4x2fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix3x4fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix3x4fv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniformMatrix4x3fv) = (void(GLApi*)(GLint, GLsizei, GLboolean, const GLfloat*))wglGetProcAddress("glUniformMatrix4x3fv"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_21__

	GLEH_OPENGL_2_1 = !n_failed_function_num && b_Support_OpenGL(2, 1);

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 2.1 ---
 */

/*
 *								--- OpenGL 3.0 ---
 */

#ifdef __GENERATE_GL_30__

void (GLApi *GLFuncPtrName(glColorMaski))(GLuint, GLboolean, GLboolean, GLboolean, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBooleani_v))(GLenum, GLuint, GLboolean *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetIntegeri_v))(GLenum, GLuint, GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEnablei))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDisablei))(GLenum, GLuint) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsEnabledi))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBeginTransformFeedback))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEndTransformFeedback))(void) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindBufferRange))(GLenum, GLuint, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindBufferBase))(GLenum, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTransformFeedbackVaryings))(GLuint, GLsizei, const GLchar* *, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTransformFeedbackVarying))(GLuint, GLuint, GLsizei, GLsizei *, GLsizei *, GLenum *, GLchar *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClampColor))(GLenum, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBeginConditionalRender))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEndConditionalRender))(void) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribIPointer))(GLuint, GLint, GLenum, GLsizei, const GLvoid *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribIiv))(GLuint, GLenum, GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribIuiv))(GLuint, GLenum, GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI1i))(GLuint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI2i))(GLuint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI3i))(GLuint, GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4i))(GLuint, GLint, GLint, GLint, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI1ui))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI2ui))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI3ui))(GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4ui))(GLuint, GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI1iv))(GLuint, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI2iv))(GLuint, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI3iv))(GLuint, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4iv))(GLuint, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI1uiv))(GLuint, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI2uiv))(GLuint, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI3uiv))(GLuint, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4uiv))(GLuint, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4bv))(GLuint, const GLbyte *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4sv))(GLuint, const GLshort *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4ubv))(GLuint, const GLubyte *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribI4usv))(GLuint, const GLushort *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetUniformuiv))(GLuint, GLint, GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindFragDataLocation))(GLuint, GLuint, const GLchar *) INIT_FUNC_POINTERS;
GLint (GLApi *GLFuncPtrName(glGetFragDataLocation))(GLuint, const GLchar *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1ui))(GLint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2ui))(GLint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3ui))(GLint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4ui))(GLint, GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1uiv))(GLint, GLsizei, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2uiv))(GLint, GLsizei, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3uiv))(GLint, GLsizei, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4uiv))(GLint, GLsizei, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexParameterIiv))(GLenum, GLenum, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexParameterIuiv))(GLenum, GLenum, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTexParameterIiv))(GLenum, GLenum, GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTexParameterIuiv))(GLenum, GLenum, GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearBufferiv))(GLenum, GLint, const GLint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearBufferuiv))(GLenum, GLint, const GLuint *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearBufferfv))(GLenum, GLint, const GLfloat *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearBufferfi))(GLenum, GLint, GLfloat, GLint) INIT_FUNC_POINTERS;
const GLubyte *(GLApi *GLFuncPtrName(glGetStringi))(GLenum, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_30__

bool GLEH_OPENGL_3_0 = false;

int CGL4ExtensionHandler::n_GetGL30FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_30__

	if(!(GLFuncPtrName(glColorMaski) = (void(GLApi*)(GLuint, GLboolean, GLboolean, GLboolean, GLboolean))wglGetProcAddress("glColorMaski"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetBooleani_v) = (void(GLApi*)(GLenum, GLuint, GLboolean*))wglGetProcAddress("glGetBooleani_v"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetIntegeri_v) = (void(GLApi*)(GLenum, GLuint, GLint*))wglGetProcAddress("glGetIntegeri_v"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glEnablei) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glEnablei"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glDisablei) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glDisablei"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glIsEnabledi) = (GLboolean(GLApi*)(GLenum, GLuint))wglGetProcAddress("glIsEnabledi"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBeginTransformFeedback) = (void(GLApi*)(GLenum))wglGetProcAddress("glBeginTransformFeedback"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glEndTransformFeedback) = (void(GLApi*)(void))wglGetProcAddress("glEndTransformFeedback"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBindBufferRange) = (void(GLApi*)(GLenum, GLuint, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glBindBufferRange"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBindBufferBase) = (void(GLApi*)(GLenum, GLuint, GLuint))wglGetProcAddress("glBindBufferBase"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glTransformFeedbackVaryings) = (void(GLApi*)(GLuint, GLsizei, const GLchar**, GLenum))wglGetProcAddress("glTransformFeedbackVaryings"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetTransformFeedbackVarying) = (void(GLApi*)(GLuint, GLuint, GLsizei, GLsizei*, GLsizei*, GLenum*, GLchar*))wglGetProcAddress("glGetTransformFeedbackVarying"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glClampColor) = (void(GLApi*)(GLenum, GLenum))wglGetProcAddress("glClampColor"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBeginConditionalRender) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glBeginConditionalRender"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glEndConditionalRender) = (void(GLApi*)(void))wglGetProcAddress("glEndConditionalRender"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribIPointer) = (void(GLApi*)(GLuint, GLint, GLenum, GLsizei, const GLvoid*))wglGetProcAddress("glVertexAttribIPointer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribIiv) = (void(GLApi*)(GLuint, GLenum, GLint*))wglGetProcAddress("glGetVertexAttribIiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetVertexAttribIuiv) = (void(GLApi*)(GLuint, GLenum, GLuint*))wglGetProcAddress("glGetVertexAttribIuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI1i) = (void(GLApi*)(GLuint, GLint))wglGetProcAddress("glVertexAttribI1i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI2i) = (void(GLApi*)(GLuint, GLint, GLint))wglGetProcAddress("glVertexAttribI2i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI3i) = (void(GLApi*)(GLuint, GLint, GLint, GLint))wglGetProcAddress("glVertexAttribI3i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4i) = (void(GLApi*)(GLuint, GLint, GLint, GLint, GLint))wglGetProcAddress("glVertexAttribI4i"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI1ui) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glVertexAttribI1ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI2ui) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glVertexAttribI2ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI3ui) = (void(GLApi*)(GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glVertexAttribI3ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4ui) = (void(GLApi*)(GLuint, GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glVertexAttribI4ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI1iv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttribI1iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI2iv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttribI2iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI3iv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttribI3iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4iv) = (void(GLApi*)(GLuint, const GLint*))wglGetProcAddress("glVertexAttribI4iv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI1uiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttribI1uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI2uiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttribI2uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI3uiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttribI3uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4uiv) = (void(GLApi*)(GLuint, const GLuint*))wglGetProcAddress("glVertexAttribI4uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4bv) = (void(GLApi*)(GLuint, const GLbyte*))wglGetProcAddress("glVertexAttribI4bv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4sv) = (void(GLApi*)(GLuint, const GLshort*))wglGetProcAddress("glVertexAttribI4sv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4ubv) = (void(GLApi*)(GLuint, const GLubyte*))wglGetProcAddress("glVertexAttribI4ubv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glVertexAttribI4usv) = (void(GLApi*)(GLuint, const GLushort*))wglGetProcAddress("glVertexAttribI4usv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetUniformuiv) = (void(GLApi*)(GLuint, GLint, GLuint*))wglGetProcAddress("glGetUniformuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glBindFragDataLocation) = (void(GLApi*)(GLuint, GLuint, const GLchar*))wglGetProcAddress("glBindFragDataLocation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetFragDataLocation) = (GLint(GLApi*)(GLuint, const GLchar*))wglGetProcAddress("glGetFragDataLocation"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1ui) = (void(GLApi*)(GLint, GLuint))wglGetProcAddress("glUniform1ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2ui) = (void(GLApi*)(GLint, GLuint, GLuint))wglGetProcAddress("glUniform2ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3ui) = (void(GLApi*)(GLint, GLuint, GLuint, GLuint))wglGetProcAddress("glUniform3ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4ui) = (void(GLApi*)(GLint, GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glUniform4ui"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform1uiv) = (void(GLApi*)(GLint, GLsizei, const GLuint*))wglGetProcAddress("glUniform1uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform2uiv) = (void(GLApi*)(GLint, GLsizei, const GLuint*))wglGetProcAddress("glUniform2uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform3uiv) = (void(GLApi*)(GLint, GLsizei, const GLuint*))wglGetProcAddress("glUniform3uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glUniform4uiv) = (void(GLApi*)(GLint, GLsizei, const GLuint*))wglGetProcAddress("glUniform4uiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glTexParameterIiv) = (void(GLApi*)(GLenum, GLenum, const GLint*))wglGetProcAddress("glTexParameterIiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glTexParameterIuiv) = (void(GLApi*)(GLenum, GLenum, const GLuint*))wglGetProcAddress("glTexParameterIuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetTexParameterIiv) = (void(GLApi*)(GLenum, GLenum, GLint*))wglGetProcAddress("glGetTexParameterIiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetTexParameterIuiv) = (void(GLApi*)(GLenum, GLenum, GLuint*))wglGetProcAddress("glGetTexParameterIuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glClearBufferiv) = (void(GLApi*)(GLenum, GLint, const GLint*))wglGetProcAddress("glClearBufferiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glClearBufferuiv) = (void(GLApi*)(GLenum, GLint, const GLuint*))wglGetProcAddress("glClearBufferuiv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glClearBufferfv) = (void(GLApi*)(GLenum, GLint, const GLfloat*))wglGetProcAddress("glClearBufferfv"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glClearBufferfi) = (void(GLApi*)(GLenum, GLint, GLfloat, GLint))wglGetProcAddress("glClearBufferfi"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetStringi) = (const GLubyte*(GLApi*)(GLenum, GLuint))wglGetProcAddress("glGetStringi"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_30__

	n_failed_function_num +=
		n_GetARBDepthBufferFloatFuncPointers() +
		n_GetARBFramebufferObjectFuncPointers() +
		n_GetARBFramebufferSRGBFuncPointers() +
		n_GetARBHalfFloatVertexFuncPointers() +
		n_GetARBMapBufferRangeFuncPointers() +
		n_GetARBTextureCompressionRGtcFuncPointers() +
		n_GetARBTextureRGFuncPointers() +
		n_GetARBVertexArrayObjectFuncPointers();

	GLEH_OPENGL_3_0 = !n_failed_function_num && b_Support_OpenGL(3, 0) &&
		b_SupportedExtension("GL_ARB_depth_buffer_float") &&
		b_SupportedExtension("GL_ARB_framebuffer_object") &&
		b_SupportedExtension("GL_ARB_framebuffer_sRGB") &&
		b_SupportedExtension("GL_ARB_half_float_vertex") &&
		b_SupportedExtension("GL_ARB_map_buffer_range") &&
		b_SupportedExtension("GL_ARB_texture_compression_rgtc") &&
		b_SupportedExtension("GL_ARB_texture_rg") &&
		b_SupportedExtension("GL_ARB_vertex_array_object");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 3.0 ---
 */

/*
 *								--- OpenGL 3.1 ---
 */

#ifdef __GENERATE_GL_31__

void (GLApi *GLFuncPtrName(glTexBuffer))(GLenum, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPrimitiveRestartIndex))(GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_31__

bool GLEH_OPENGL_3_1 = false;

int CGL4ExtensionHandler::n_GetGL31FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_31__

	if(!(GLFuncPtrName(glTexBuffer) = (void(GLApi*)(GLenum, GLenum, GLuint))wglGetProcAddress("glTexBuffer"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glPrimitiveRestartIndex) = (void(GLApi*)(GLuint))wglGetProcAddress("glPrimitiveRestartIndex"))) ++ n_failed_function_num;

#endif // __GENERATE_GL_31__

	n_failed_function_num += n_GetARBUniformBufferObjectFuncPointers() +
		n_GetARBDrawInstancedFuncPointers() +
		n_GetARBCopyBufferFuncPointers();

	GLEH_OPENGL_3_1 = !n_failed_function_num && b_Support_OpenGL(3, 1) &&
		b_SupportedExtension("GL_ARB_uniform_buffer_object") &&
		b_SupportedExtension("GL_ARB_draw_instanced") &&
		b_SupportedExtension("GL_ARB_copy_buffer");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 3.1 ---
 */

/*
 *								--- OpenGL 3.2 ---
 */

#ifdef __GENERATE_GL_32__

void (GLApi *GLFuncPtrName(glGetInteger64i_v))(GLenum, GLuint, GLint64 *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBufferParameteri64v))(GLenum, GLenum, GLint64 *) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramParameteri))(GLuint, GLenum, GLint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferTexture))(GLenum, GLenum, GLuint, GLint) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glFramebufferTextureFace))(GLenum, GLenum, GLuint, GLint, GLenum) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_32__

bool GLEH_OPENGL_3_2 = false;

int CGL4ExtensionHandler::n_GetGL32FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_32__

	if(!(GLFuncPtrName(glGetInteger64i_v) = (void(GLApi*)(GLenum, GLuint, GLint64 *))wglGetProcAddress("glGetInteger64i_v"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glGetBufferParameteri64v) = (void(GLApi*)(GLenum, GLenum, GLint64 *))wglGetProcAddress("glGetBufferParameteri64v"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glProgramParameteri) = (void(GLApi*)(GLuint, GLenum, GLint))wglGetProcAddress("glProgramParameteri"))) ++ n_failed_function_num;
	if(!(GLFuncPtrName(glFramebufferTexture) = (void(GLApi*)(GLenum, GLenum, GLuint, GLint))wglGetProcAddress("glFramebufferTexture"))) ++ n_failed_function_num;
	//if(!(GLFuncPtrName(glFramebufferTextureFace) = (void(GLApi*)(GLenum, GLenum, GLuint, GLint, GLenum))wglGetProcAddress("glFramebufferTextureFace"))) ++ n_failed_function_num; // this one doesn't exist (a bug in the documentation)

#endif // __GENERATE_GL_32__

	n_failed_function_num +=
		n_GetARBDepthClampFuncPointers() +
		n_GetARBDrawElementsBaseVertexFuncPointers() +
		n_GetARBFragmentCoordConventionsFuncPointers() +
		n_GetARBProvokingVertexFuncPointers() +
		n_GetARBSeamlessCubeMapFuncPointers() +
		n_GetARBSyncFuncPointers() +
		n_GetARBTextureMultisampleFuncPointers() +
		n_GetARBVertexArrayBGRAFuncPointers();

	GLEH_OPENGL_3_2 = !n_failed_function_num && b_Support_OpenGL(3, 2) &&
		b_SupportedExtension("GL_ARB_depth_clamp") &&
		b_SupportedExtension("GL_ARB_draw_elements_base_vertex") &&
		b_SupportedExtension("GL_ARB_fragment_coord_conventions") &&
		b_SupportedExtension("GL_ARB_provoking_vertex") &&
		b_SupportedExtension("GL_ARB_seamless_cube_map") &&
		b_SupportedExtension("GL_ARB_sync") &&
		b_SupportedExtension("GL_ARB_texture_multisample") &&
		b_SupportedExtension("GL_ARB_vertex_array_bgra");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 3.2 ---
 */

/*
 *								--- OpenGL 3.3 ---
 */

#ifdef __GENERATE_GL_33__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_33__

bool GLEH_OPENGL_3_3 = false;

int CGL4ExtensionHandler::n_GetGL33FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_33__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_33__

	n_failed_function_num +=
		n_GetARBShaderBitEncodingFuncPointers() +
		n_GetARBBlendFuncExtendedFuncPointers() +
		n_GetARBExplicitAttribLocationFuncPointers() +
		n_GetARBOcclusionQuery2FuncPointers() +
		n_GetARBSamplerObjectsFuncPointers() +
		n_GetARBTextureRGB10A2uiFuncPointers() +
		n_GetARBTextureSwizzleFuncPointers() +
		n_GetARBTimerQueryFuncPointers() +
		n_GetARBInstancedArraysFuncPointers() +
		n_GetARBVertexType2101010RevFuncPointers();

	GLEH_OPENGL_3_3 = !n_failed_function_num && b_Support_OpenGL(3, 2) &&
		b_SupportedExtension("GL_ARB_shader_bit_encoding") &&
		b_SupportedExtension("GL_ARB_blend_func_extended") &&
		b_SupportedExtension("GL_ARB_explicit_attrib_location") &&
		b_SupportedExtension("GL_ARB_occlusion_query2") &&
		b_SupportedExtension("GL_ARB_sampler_objects") &&
		b_SupportedExtension("GL_ARB_texture_rgb10_a2ui") &&
		b_SupportedExtension("GL_ARB_texture_swizzle") &&
		b_SupportedExtension("GL_ARB_timer_query") &&
		b_SupportedExtension("GL_ARB_instanced_arrays") &&
		b_SupportedExtension("GL_ARB_vertex_type_2_10_10_10_rev");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 3.3 ---
 */

/*
 *								--- OpenGL 4.0 ---
 */

#ifdef __GENERATE_GL_40__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_40__

bool GLEH_OPENGL_4_0 = false;

int CGL4ExtensionHandler::n_GetGL40FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_40__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_40__

	n_failed_function_num +=
		n_GetARBTextureQueryLodFuncPointers() +
		n_GetARBDrawBuffersBlendFuncPointers() +
		n_GetARBDrawIndirectFuncPointers() +
		n_GetARBGPUShader5FuncPointers() +
		n_GetARBGPUShaderFP64FuncPointers() +
		n_GetARBSampleShadingFuncPointers() +
		n_GetARBShaderSubroutineFuncPointers() +
		n_GetARBTessellationShaderFuncPointers() +
		n_GetARBTextureBufferObjectRGB32FuncPointers() +
		n_GetARBTextureGatherFuncPointers() +
		n_GetARBTextureCubeMapArrayFuncPointers() +
		n_GetARBTransformFeedback2FuncPointers() +
		n_GetARBTransformFeedback3FuncPointers();

	GLEH_OPENGL_4_0 = !n_failed_function_num && b_Support_OpenGL(4, 0) &&
		b_SupportedExtension("GL_ARB_texture_query_lod") &&
		b_SupportedExtension("GL_ARB_draw_buffers_blend") &&
		b_SupportedExtension("GL_ARB_draw_indirect") &&
		b_SupportedExtension("GL_ARB_gpu_shader5") &&
		b_SupportedExtension("GL_ARB_gpu_shader_fp64") &&
		b_SupportedExtension("GL_ARB_sample_shading") &&
		b_SupportedExtension("GL_ARB_shader_subroutine") &&
		b_SupportedExtension("GL_ARB_tessellation_shader") &&
		b_SupportedExtension("GL_ARB_texture_buffer_object_rgb32") &&
		b_SupportedExtension("GL_ARB_texture_cube_map_array") &&
		b_SupportedExtension("GL_ARB_texture_gather") &&
		b_SupportedExtension("GL_ARB_transform_feedback2") &&
		b_SupportedExtension("GL_ARB_transform_feedback3");

	return n_failed_function_num; 
}

/*
 *								--- ~OpenGL 4.0 ---
 */

/*
 *								--- OpenGL 4.1 ---
 */

#ifdef __GENERATE_GL_41__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_41__

bool GLEH_OPENGL_4_1 = false;

int CGL4ExtensionHandler::n_GetGL41FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_41__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_41__

	n_failed_function_num +=
		n_GetARBES2CompatibilityFuncPointers() +
		n_GetARBGetProgramBinaryFuncPointers() +
		n_GetARBSeparateShaderObjectsFuncPointers() +
		n_GetARBShaderPrecisionFuncPointers() +
		n_GetARBVertexAttrib64bitFuncPointers() +
		n_GetARBViewportArrayFuncPointers();
 
	GLEH_OPENGL_4_1 = !n_failed_function_num && b_Support_OpenGL(4, 1) &&
		b_SupportedExtension("GL_ARB_ES2_compatibility") &&
		b_SupportedExtension("GL_ARB_get_program_binary") &&
		b_SupportedExtension("GL_ARB_separate_shader_objects") &&
		b_SupportedExtension("GL_ARB_shader_precision") &&
		b_SupportedExtension("GL_ARB_vertex_attrib_64bit") &&
		b_SupportedExtension("GL_ARB_viewport_array");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 4.1 ---
 */

/*
 *								--- OpenGL 4.2 ---
 */

#ifdef __GENERATE_GL_42__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_42__

bool GLEH_OPENGL_4_2 = false;

int CGL4ExtensionHandler::n_GetGL42FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_42__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_42__

	n_failed_function_num +=
		n_GetARBTextureCompressionBptcFuncPointers() +
		n_GetARBCompressedTexturePixelStorageFuncPointers() +
		n_GetARBShaderAtomicCountersFuncPointers() +
		n_GetARBTextureStorageFuncPointers() +
		n_GetARBTransformFeedbackInstancedFuncPointers() +
		n_GetARBBaseInstanceFuncPointers() +
		n_GetARBShaderImageLoadStoreFuncPointers() +
		n_GetARBConservativeDepthFuncPointers() +
		n_GetARBShadingLanguage420packFuncPointers() +
		n_GetARBInternalformatQueryFuncPointers() +
		n_GetARBMapBufferAlignmentFuncPointers() +
		n_GetARBShadingLanguagePackingFuncPointers();

	GLEH_OPENGL_4_2 = !n_failed_function_num && b_Support_OpenGL(4, 2) &&
		b_SupportedExtension("GL_ARB_texture_compression_bptc") &&
		b_SupportedExtension("GL_ARB_compressed_texture_pixel_storage") &&
		b_SupportedExtension("GL_ARB_shader_atomic_counters") &&
		b_SupportedExtension("GL_ARB_texture_storage") &&
		b_SupportedExtension("GL_ARB_transform_feedback_instanced") &&
		b_SupportedExtension("GL_ARB_base_instance") &&
		b_SupportedExtension("GL_ARB_shader_image_load_store") &&
		b_SupportedExtension("GL_ARB_conservative_depth") &&
		b_SupportedExtension("GL_ARB_shading_language_420pack") &&
		b_SupportedExtension("GL_ARB_internalformat_query") &&
		b_SupportedExtension("GL_ARB_map_buffer_alignment") &&
		b_SupportedExtension("GL_ARB_shading_language_packing");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 4.2 ---
 */

/*
 *								--- OpenGL 4.3 ---
 */

#ifdef __GENERATE_GL_43__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_43__

bool GLEH_OPENGL_4_3 = false;

int CGL4ExtensionHandler::n_GetGL43FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_43__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_43__

	n_failed_function_num += n_GetARBArraysOfArraysFuncPointers() +
		n_GetARBES3CompatibilityFuncPointers() +
		n_GetARBClearBufferObjectFuncPointers() +
		n_GetARBComputeShaderFuncPointers() +
		n_GetARBCopyImageFuncPointers() +
		n_GetARBDebugGroupFuncPointers() +
		n_GetARBDebugLabelFuncPointers() +
		n_GetARBDebugOutput2FuncPointers() +
		n_GetARBDebugOutputFuncPointers() +
		n_GetARBExplicitUniformLocationFuncPointers() +
		n_GetARBFragmentLayerViewportFuncPointers() +
		n_GetARBFramebufferNoAttachmentsFuncPointers() +
		n_GetARBInternalformatQuery2FuncPointers() +
		n_GetARBInvalidateSubdataFuncPointers() +
		n_GetARBMultiDrawIndirectFuncPointers() +
		n_GetARBProgramInterfaceQueryFuncPointers() +
		n_GetARBRobustBufferAccessBehaviorFuncPointers() +
		n_GetARBShaderImageSizeFuncPointers() +
		n_GetARBShaderStorageBufferObjectFuncPointers() +
		n_GetARBStencilTexturingFuncPointers() +
		n_GetARBTextureBufferRangeFuncPointers() +
		n_GetARBTextureQueryLevelsFuncPointers() +
		n_GetARBTextureStorageMultisampleFuncPointers() +
		n_GetARBTextureViewFuncPointers() +
		n_GetARBVertexAttribBindingFuncPointers() +
		n_GetKHRTextureCompressionASTC_HDRFuncPointers() +
		n_GetARBRobustnessIsolationFuncPointers();

	GLEH_OPENGL_4_3 = !n_failed_function_num && b_Support_OpenGL(4, 3) &&
		b_SupportedExtension("GL_ARB_arrays_of_arrays") &&
		b_SupportedExtension("GL_ARB_ES3_compatibility") &&
		b_SupportedExtension("GL_ARB_clear_buffer_object") &&
		b_SupportedExtension("GL_ARB_compute_shader") &&
		b_SupportedExtension("GL_ARB_copy_image") &&
		//b_SupportedExtension("GL_ARB_debug_group") &&
		//b_SupportedExtension("GL_ARB_debug_label") &&
		//b_SupportedExtension("GL_ARB_debug_output2") &&
		b_SupportedExtension("GL_ARB_debug_output") &&
		b_SupportedExtension("GL_ARB_explicit_uniform_location") &&
		b_SupportedExtension("GL_ARB_fragment_layer_viewport") &&
		b_SupportedExtension("GL_ARB_framebuffer_no_attachments") &&
		b_SupportedExtension("GL_ARB_internalformat_query2") &&
		b_SupportedExtension("GL_ARB_invalidate_subdata") &&
		b_SupportedExtension("GL_ARB_multi_draw_indirect") &&
		b_SupportedExtension("GL_ARB_program_interface_query") &&
		b_SupportedExtension("GL_ARB_robust_buffer_access_behavior") &&
		b_SupportedExtension("GL_ARB_shader_image_size") &&
		b_SupportedExtension("GL_ARB_shader_storage_buffer_object") &&
		b_SupportedExtension("GL_ARB_stencil_texturing") &&
		b_SupportedExtension("GL_ARB_texture_buffer_range") &&
		b_SupportedExtension("GL_ARB_texture_query_levels") &&
		b_SupportedExtension("GL_ARB_texture_storage_multisample") &&
		b_SupportedExtension("GL_ARB_texture_view") &&
		b_SupportedExtension("GL_ARB_vertex_attrib_binding") /*&&
		b_SupportedExtension("GL_KHR_texture_compression_astc_ldr") && // this is not in the lists of extensions supported by NVidia's GL 4.4 but the entry points are there
		b_SupportedExtension("GL_ARB_robustness_isolation")*/;

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 4.3 ---
 */

/*
 *								--- OpenGL 4.4 ---
 */

#ifdef __GENERATE_GL_44__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_44__

bool GLEH_OPENGL_4_4 = false;

int CGL4ExtensionHandler::n_GetGL44FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_44__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_44__

	n_failed_function_num += n_GetARBBufferStorageFuncPointers() +
		n_GetARBClearTextureFuncPointers() +
		n_GetARBEnhancedLayoutsFuncPointers() +
		n_GetARBMultiBindFuncPointers() +
		n_GetARBQueryBufferObjectFuncPointers() +
		n_GetARBTextureMirrorClampToEdgeFuncPointers() +
		n_GetARBTextureStencil8FuncPointers() +
		n_GetARBVertexType10f11f11fRevFuncPointers();

	GLEH_OPENGL_4_4 = !n_failed_function_num && b_Support_OpenGL(4, 4) &&
		b_SupportedExtension("GL_ARB_buffer_storage") &&
		b_SupportedExtension("GL_ARB_clear_texture") &&
		b_SupportedExtension("GL_ARB_enhanced_layouts") &&
		b_SupportedExtension("GL_ARB_multi_bind") &&
		b_SupportedExtension("GL_ARB_query_buffer_object") &&
		b_SupportedExtension("GL_ARB_texture_mirror_clamp_to_edge") &&
		b_SupportedExtension("GL_ARB_texture_stencil8") &&
		b_SupportedExtension("GL_ARB_vertex_type_10f_11f_11f_rev");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 4.4 ---
 */

/*
 *								--- OpenGL 4.5 ---
 */

#ifdef __GENERATE_GL_45__

// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_45__

bool GLEH_OPENGL_4_5 = false;

int CGL4ExtensionHandler::n_GetGL45FuncPointers()
{
	int n_failed_function_num = 0;

#ifdef __GENERATE_GL_45__

	// f_ixme - no new tokens, except the exts?

#endif // __GENERATE_GL_45__

	n_failed_function_num += n_GetARBClipControlFuncPointers() +
		n_GetARBCullDistanceFuncPointers() +
		n_GetARBES31CompatibilityFuncPointers() +
		n_GetARBConditionalRenderInvertedFuncPointers() +
		n_GetKHRContextFlushControlFuncPointers() +
		n_GetARBDerivativeControlFuncPointers() +
		n_GetARBDirectStateAccessFuncPointers() +
		n_GetARBGetTextureSubImageFuncPointers() +
		n_GetKHRRobustnessFuncPointers() +
		n_GetARBShaderTextureImageSamplesFuncPointers() +
		n_GetARBTextureBarrierFuncPointers();

	GLEH_OPENGL_4_5 = !n_failed_function_num && b_Support_OpenGL(4, 5) &&
		b_SupportedExtension("GL_ARB_clip_control") &&
		b_SupportedExtension("GL_ARB_cull_distance") &&
		b_SupportedExtension("GL_ARB_ES3_1_compatibility") &&
		b_SupportedExtension("GL_ARB_conditional_render_inverted") &&
		b_SupportedExtension("GL_KHR_context_flush_control") &&
		b_SupportedExtension("GL_ARB_derivative_control") &&
		b_SupportedExtension("GL_ARB_direct_state_access") &&
		b_SupportedExtension("GL_ARB_get_texture_sub_image") &&
		b_SupportedExtension("GL_KHR_robustness") &&
		b_SupportedExtension("GL_ARB_shader_texture_image_samples") &&
		b_SupportedExtension("GL_ARB_texture_barrier");

	return n_failed_function_num;
}

/*
 *								--- ~OpenGL 4.5 ---
 */

/*
 *								--- GL_ARB_depth_buffer_float (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_depth_buffer_float = false;

int CGL4ExtensionHandler::n_GetARBDepthBufferFloatFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_depth_buffer_float = !n_failed_functions && b_SupportedExtension("GL_ARB_depth_buffer_float");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_depth_buffer_float ---
 */

/*
 *								--- GL_ARB_framebuffer_object (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_framebuffer_object = false;

#ifdef __GENERATE_GL_ARB_framebuffer_object

GLboolean (GLApi *GLFuncPtrName(glIsRenderbuffer))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindRenderbuffer))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteRenderbuffers))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenRenderbuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glRenderbufferStorage))(GLenum, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glRenderbufferStorageMultisample))(GLenum, GLsizei, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetRenderbufferParameteriv))(GLenum, GLenum, int*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsFramebuffer))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindFramebuffer))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteFramebuffers))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenFramebuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
GLenum (GLApi *GLFuncPtrName(glCheckFramebufferStatus))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferTexture1D))(GLenum, GLenum, GLenum, GLuint, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferTexture2D))(GLenum, GLenum, GLenum, GLuint, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferTexture3D))(GLenum, GLenum, GLenum, GLuint, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferTextureLayer))(GLenum, GLenum, GLuint, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFramebufferRenderbuffer))(GLenum, GLenum, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetFramebufferAttachmentParameteriv))(GLenum, GLenum, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlitFramebuffer))(int, int, int, int, int, int, int, int, GLbitfield, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenerateMipmap))(GLenum) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_framebuffer_object

int CGL4ExtensionHandler::n_GetARBFramebufferObjectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_framebuffer_object

	if(!(GLFuncPtrName(glIsRenderbuffer) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsRenderbuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindRenderbuffer) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glBindRenderbuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteRenderbuffers) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteRenderbuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenRenderbuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenRenderbuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glRenderbufferStorage) = (void(GLApi*)(GLenum, GLenum, GLsizei, GLsizei))wglGetProcAddress("glRenderbufferStorage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glRenderbufferStorageMultisample) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei, GLsizei))wglGetProcAddress("glRenderbufferStorageMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetRenderbufferParameteriv) = (void(GLApi*)(GLenum, GLenum, int*))wglGetProcAddress("glGetRenderbufferParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsFramebuffer) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsFramebuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindFramebuffer) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glBindFramebuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteFramebuffers) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteFramebuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenFramebuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenFramebuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCheckFramebufferStatus) = (GLenum(GLApi*)(GLenum))wglGetProcAddress("glCheckFramebufferStatus"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFramebufferTexture1D) = (void(GLApi*)(GLenum, GLenum, GLenum, GLuint, int))wglGetProcAddress("glFramebufferTexture1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFramebufferTexture2D) = (void(GLApi*)(GLenum, GLenum, GLenum, GLuint, int))wglGetProcAddress("glFramebufferTexture2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFramebufferTexture3D) = (void(GLApi*)(GLenum, GLenum, GLenum, GLuint, int, int))wglGetProcAddress("glFramebufferTexture3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFramebufferTextureLayer) = (void(GLApi*)(GLenum, GLenum, GLuint, int, int))wglGetProcAddress("glFramebufferTextureLayer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFramebufferRenderbuffer) = (void(GLApi*)(GLenum, GLenum, GLenum, GLuint))wglGetProcAddress("glFramebufferRenderbuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetFramebufferAttachmentParameteriv) = (void(GLApi*)(GLenum, GLenum, GLenum, int*))wglGetProcAddress("glGetFramebufferAttachmentParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBlitFramebuffer) = (void(GLApi*)(int, int, int, int, int, int, int, int, GLbitfield, GLenum))wglGetProcAddress("glBlitFramebuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenerateMipmap) = (void(GLApi*)(GLenum))wglGetProcAddress("glGenerateMipmap"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_framebuffer_object

	GLEH_ARB_framebuffer_object = !n_failed_functions && b_SupportedExtension("GL_ARB_framebuffer_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_framebuffer_object ---
 */

/*
 *								--- GL_ARB_framebuffer_sRGB (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_framebuffer_sRGB = false;

int CGL4ExtensionHandler::n_GetARBFramebufferSRGBFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_framebuffer_sRGB = !n_failed_functions && b_SupportedExtension("GL_ARB_framebuffer_sRGB");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_framebuffer_sRGB ---
 */

/*
 *								--- GL_ARB_half_float_vertex (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_half_float_vertex = false;

int CGL4ExtensionHandler::n_GetARBHalfFloatVertexFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_half_float_vertex = !n_failed_functions && b_SupportedExtension("GL_ARB_half_float_vertex");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_half_float_vertex ---
 */

/*
 *								--- GL_ARB_map_buffer_range (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_map_buffer_range = false;

#ifdef __GENERATE_GL_ARB_map_buffer_range

void* (GLApi *GLFuncPtrName(glMapBufferRange))(GLenum, GLintptr, GLsizeiptr, GLbitfield) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFlushMappedBufferRange))(GLenum, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_map_buffer_range

int CGL4ExtensionHandler::n_GetARBMapBufferRangeFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_map_buffer_range

	if(!(GLFuncPtrName(glMapBufferRange) = (void*(GLApi*)(GLenum, GLintptr, GLsizeiptr, GLbitfield))wglGetProcAddress("glMapBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFlushMappedBufferRange) = (void(GLApi*)(GLenum, GLintptr, GLsizeiptr))wglGetProcAddress("glFlushMappedBufferRange"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_map_buffer_range

	GLEH_ARB_map_buffer_range = !n_failed_functions && b_SupportedExtension("GL_ARB_map_buffer_range");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_map_buffer_range ---
 */

/*
 *								--- GL_ARB_texture_compression_rgtc (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_texture_compression_rgtc = false;

int CGL4ExtensionHandler::n_GetARBTextureCompressionRGtcFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_compression_rgtc = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_compression_rgtc");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_compression_rgtc ---
 */

/*
 *								--- GL_ARB_texture_rg (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_texture_rg = false;

int CGL4ExtensionHandler::n_GetARBTextureRGFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_rg = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_rg");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_rg ---
 */

/*
 *								--- GL_ARB_vertex_array_object (part of OpenGL 3.0 core) ---
 */

bool GLEH_ARB_vertex_array_object = false;

#ifdef __GENERATE_GL_ARB_vertex_array_object

void (GLApi *GLFuncPtrName(glBindVertexArray))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteVertexArrays))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenVertexArrays))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsVertexArray))(GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_vertex_array_object

int CGL4ExtensionHandler::n_GetARBVertexArrayObjectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_vertex_array_object

	if(!(GLFuncPtrName(glBindVertexArray) = (void(GLApi*)(GLuint))wglGetProcAddress("glBindVertexArray"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteVertexArrays) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteVertexArrays"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenVertexArrays) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenVertexArrays"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsVertexArray) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsVertexArray"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_vertex_array_object

	GLEH_ARB_vertex_array_object = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_array_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_array_object ---
 */

/*
 *								--- GL_ARB_uniform_buffer_object (part of OpenGL 3.1 core) ---
 */

bool GLEH_ARB_uniform_buffer_object = false;

#ifdef __GENERATE_GL_ARB_uniform_buffer_object

void (GLApi *GLFuncPtrName(glGetUniformIndices))(GLuint, GLsizei, const GLchar**, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveUniformsiv))(GLuint, GLsizei, const GLuint*, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveUniformName))(GLuint, GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glGetUniformBlockIndex))(GLuint, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveUniformBlockiv))(GLuint, GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveUniformBlockName))(GLuint, GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glBindBufferRange))(GLenum, GLuint, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glBindBufferBase))(GLenum, GLuint, GLuint) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glGetIntegeri_v))(GLenum, GLuint, int*) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glUniformBlockBinding))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_uniform_buffer_object

int CGL4ExtensionHandler::n_GetARBUniformBufferObjectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_uniform_buffer_object

	if(!(GLFuncPtrName(glGetUniformIndices) = (void(GLApi*)(GLuint, GLsizei, const GLchar**, GLuint*))wglGetProcAddress("glGetUniformIndices"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveUniformsiv) = (void(GLApi*)(GLuint, GLsizei, const GLuint*, GLenum, int*))wglGetProcAddress("glGetActiveUniformsiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveUniformName) = (void(GLApi*)(GLuint, GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetActiveUniformName"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetUniformBlockIndex) = (GLuint(GLApi*)(GLuint, const GLchar*))wglGetProcAddress("glGetUniformBlockIndex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveUniformBlockiv) = (void(GLApi*)(GLuint, GLuint, GLenum, int*))wglGetProcAddress("glGetActiveUniformBlockiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveUniformBlockName) = (void(GLApi*)(GLuint, GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetActiveUniformBlockName"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindBufferRange) = (void(GLApi*)(GLenum, GLuint, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glBindBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindBufferBase) = (void(GLApi*)(GLenum, GLuint, GLuint))wglGetProcAddress("glBindBufferBase"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetIntegeri_v) = (void(GLApi*)(GLenum, GLuint, int*))wglGetProcAddress("glGetIntegeri_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformBlockBinding) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glUniformBlockBinding"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_uniform_buffer_object

	GLEH_ARB_uniform_buffer_object = !n_failed_functions && b_SupportedExtension("GL_ARB_uniform_buffer_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_uniform_buffer_object ---
 */

/*
 *								--- GL_ARB_draw_instanced (part of OpenGL 3.1 core) ---
 */

bool GLEH_ARB_draw_instanced = false;

#ifdef __GENERATE_GL_ARB_draw_instanced

void (GLApi *GLFuncPtrName(glDrawArraysInstanced))(GLenum, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawElementsInstanced))(GLenum, GLsizei, GLenum, const void*, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_draw_instanced

int CGL4ExtensionHandler::n_GetARBDrawInstancedFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_draw_instanced

	if(!(GLFuncPtrName(glDrawArraysInstanced) = (void(GLApi*)(GLenum, int, GLsizei, GLsizei))wglGetProcAddress("glDrawArraysInstancedARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawElementsInstanced) = (void(GLApi*)(GLenum, GLsizei, GLenum, const void*, GLsizei))wglGetProcAddress("glDrawElementsInstancedARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_draw_instanced

	GLEH_ARB_draw_instanced = !n_failed_functions && b_SupportedExtension("GL_ARB_draw_instanced");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_draw_instanced ---
 */

/*
 *								--- GL_ARB_copy_buffer (part of OpenGL 3.1 core) ---
 */

bool GLEH_ARB_copy_buffer = false;

#ifdef __GENERATE_GL_ARB_copy_buffer

void (GLApi *GLFuncPtrName(glCopyBufferSubData))(GLenum, GLenum, GLintptr, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_copy_buffer

int CGL4ExtensionHandler::n_GetARBCopyBufferFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_copy_buffer

	if(!(GLFuncPtrName(glCopyBufferSubData) = (void(GLApi*)(GLenum, GLenum, GLintptr, GLintptr, GLsizeiptr))wglGetProcAddress("glCopyBufferSubData"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_copy_buffer

	GLEH_ARB_copy_buffer = !n_failed_functions && b_SupportedExtension("GL_ARB_copy_buffer");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_copy_buffer ---
 */

/*
 *								--- GL_ARB_depth_clamp (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_depth_clamp = false;

int CGL4ExtensionHandler::n_GetARBDepthClampFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_depth_clamp = !n_failed_functions && b_SupportedExtension("GL_ARB_depth_clamp");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_depth_clamp ---
 */

/*
 *								--- GL_ARB_draw_elements_base_vertex (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_draw_elements_base_vertex = false;

#ifdef __GENERATE_GL_ARB_draw_elements_base_vertex

void (GLApi *GLFuncPtrName(glDrawElementsBaseVertex))(GLenum, GLsizei, GLenum, void*, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawRangeElementsBaseVertex))(GLenum, GLuint, GLuint, GLsizei, GLenum, void*, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawElementsInstancedBaseVertex))(GLenum, GLsizei, GLenum, const void*, GLsizei, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiDrawElementsBaseVertex))(GLenum, GLsizei*, GLenum, void**, GLsizei, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_draw_elements_base_vertex

int CGL4ExtensionHandler::n_GetARBDrawElementsBaseVertexFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_draw_elements_base_vertex

	if(!(GLFuncPtrName(glDrawElementsBaseVertex) = (void(GLApi*)(GLenum, GLsizei, GLenum, void*, int))wglGetProcAddress("glDrawElementsBaseVertex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawRangeElementsBaseVertex) = (void(GLApi*)(GLenum, GLuint, GLuint, GLsizei, GLenum, void*, int))wglGetProcAddress("glDrawRangeElementsBaseVertex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawElementsInstancedBaseVertex) = (void(GLApi*)(GLenum, GLsizei, GLenum, const void*, GLsizei, int))wglGetProcAddress("glDrawElementsInstancedBaseVertex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMultiDrawElementsBaseVertex) = (void(GLApi*)(GLenum, GLsizei*, GLenum, void**, GLsizei, int*))wglGetProcAddress("glMultiDrawElementsBaseVertex"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_draw_elements_base_vertex

	GLEH_ARB_draw_elements_base_vertex = !n_failed_functions && b_SupportedExtension("GL_ARB_draw_elements_base_vertex");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_draw_elements_base_vertex ---
 */

/*
 *								--- GL_ARB_fragment_coord_conventions (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_fragment_coord_conventions = false;

int CGL4ExtensionHandler::n_GetARBFragmentCoordConventionsFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_fragment_coord_conventions = !n_failed_functions && b_SupportedExtension("GL_ARB_fragment_coord_conventions");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_fragment_coord_conventions ---
 */

/*
 *								--- GL_ARB_provoking_vertex (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_provoking_vertex = false;

#ifdef __GENERATE_GL_ARB_provoking_vertex

void (GLApi *GLFuncPtrName(glProvokingVertex))(GLenum) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_provoking_vertex

int CGL4ExtensionHandler::n_GetARBProvokingVertexFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_provoking_vertex

	if(!(GLFuncPtrName(glProvokingVertex) = (void(GLApi*)(GLenum))wglGetProcAddress("glProvokingVertex"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_provoking_vertex

	GLEH_ARB_provoking_vertex = !n_failed_functions && b_SupportedExtension("GL_ARB_provoking_vertex");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_provoking_vertex ---
 */

/*
 *								--- GL_ARB_seamless_cube_map (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_seamless_cube_map = false;

int CGL4ExtensionHandler::n_GetARBSeamlessCubeMapFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_seamless_cube_map = !n_failed_functions && b_SupportedExtension("GL_ARB_seamless_cube_map");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_seamless_cube_map ---
 */

/*
 *								--- GL_ARB_sync (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_sync = false;

#ifdef __GENERATE_GL_ARB_sync

GLsync (GLApi *GLFuncPtrName(glFenceSync))(GLenum, GLbitfield) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsSync))(GLsync) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteSync))(GLsync) INIT_FUNC_POINTERS;
GLenum (GLApi *GLFuncPtrName(glClientWaitSync))(GLsync, GLbitfield, GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glWaitSync))(GLsync, GLbitfield, GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetInteger64v))(GLenum, GLint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSynciv))(GLsync, GLenum, GLsizei, GLsizei*, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_sync

int CGL4ExtensionHandler::n_GetARBSyncFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_sync

	if(!(GLFuncPtrName(glFenceSync) = (GLsync(GLApi*)(GLenum, GLbitfield))wglGetProcAddress("glFenceSync"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsSync) = (GLboolean(GLApi*)(GLsync))wglGetProcAddress("glIsSync"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteSync) = (void(GLApi*)(GLsync))wglGetProcAddress("glDeleteSync"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClientWaitSync) = (GLenum(GLApi*)(GLsync, GLbitfield, GLuint64))wglGetProcAddress("glClientWaitSync"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glWaitSync) = (void(GLApi*)(GLsync, GLbitfield, GLuint64))wglGetProcAddress("glWaitSync"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetInteger64v) = (void(GLApi*)(GLenum, GLint64*))wglGetProcAddress("glGetInteger64v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSynciv) = (void(GLApi*)(GLsync, GLenum, GLsizei, GLsizei*, int*))wglGetProcAddress("glGetSynciv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_sync

	GLEH_ARB_sync = !n_failed_functions && b_SupportedExtension("GL_ARB_sync");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_sync ---
 */

/*
 *								--- GL_ARB_texture_multisample (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_texture_multisample = false;

#ifdef __GENERATE_GL_ARB_texture_multisample

void (GLApi *GLFuncPtrName(glTexImage2DMultisample))(GLenum, GLsizei, int, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexImage3DMultisample))(GLenum, GLsizei, int, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetMultisamplefv))(GLenum, GLuint, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSampleMaski))(GLuint, GLbitfield) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_texture_multisample

int CGL4ExtensionHandler::n_GetARBTextureMultisampleFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_multisample

	if(!(GLFuncPtrName(glTexImage2DMultisample) = (void(GLApi*)(GLenum, GLsizei, int, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexImage2DMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTexImage3DMultisample) = (void(GLApi*)(GLenum, GLsizei, int, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexImage3DMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetMultisamplefv) = (void(GLApi*)(GLenum, GLuint, float*))wglGetProcAddress("glGetMultisamplefv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSampleMaski) = (void(GLApi*)(GLuint, GLbitfield))wglGetProcAddress("glSampleMaski"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_multisample

	GLEH_ARB_texture_multisample = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_multisample");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_multisample ---
 */

/*
 *								--- GL_ARB_vertex_array_bgra (part of OpenGL 3.2 core) ---
 */

bool GLEH_ARB_vertex_array_bgra = false;

int CGL4ExtensionHandler::n_GetARBVertexArrayBGRAFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_vertex_array_bgra = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_array_bgra");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_array_bgra ---
 */

/*
 *								--- GL_ARB_shader_bit_encoding (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_shader_bit_encoding = false;

int CGL4ExtensionHandler::n_GetARBShaderBitEncodingFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_bit_encoding = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_bit_encoding");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_bit_encoding ---
 */

/*
 *								--- GL_ARB_blend_func_extended (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_blend_func_extended = false;

#ifdef __GENERATE_GL_ARB_blend_func_extended

void (GLApi *GLFuncPtrName(glBindFragDataLocationIndexed))(GLuint, GLuint, GLuint, const GLchar*) INIT_FUNC_POINTERS;
int (GLApi *GLFuncPtrName(glGetFragDataIndex))(GLuint, const GLchar*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_blend_func_extended

int CGL4ExtensionHandler::n_GetARBBlendFuncExtendedFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_blend_func_extended

	if(!(GLFuncPtrName(glBindFragDataLocationIndexed) = (void(GLApi*)(GLuint, GLuint, GLuint, const GLchar*))wglGetProcAddress("glBindFragDataLocationIndexed"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetFragDataIndex) = (int(GLApi*)(GLuint, const GLchar*))wglGetProcAddress("glGetFragDataIndex"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_blend_func_extended

	GLEH_ARB_blend_func_extended = !n_failed_functions && b_SupportedExtension("GL_ARB_blend_func_extended");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_blend_func_extended ---
 */

/*
 *								--- GL_ARB_explicit_attrib_location (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_explicit_attrib_location = false;

int CGL4ExtensionHandler::n_GetARBExplicitAttribLocationFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_explicit_attrib_location = !n_failed_functions && b_SupportedExtension("GL_ARB_explicit_attrib_location");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_explicit_attrib_location ---
 */

/*
 *								--- GL_ARB_occlusion_query2 (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_occlusion_query2 = false;

int CGL4ExtensionHandler::n_GetARBOcclusionQuery2FuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_occlusion_query2 = !n_failed_functions && b_SupportedExtension("GL_ARB_occlusion_query2");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_occlusion_query2 ---
 */

/*
 *								--- GL_ARB_sampler_objects (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_sampler_objects = false;

#ifdef __GENERATE_GL_ARB_sampler_objects

void (GLApi *GLFuncPtrName(glGenSamplers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteSamplers))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsSampler))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindSampler))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameteri))(GLuint, GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameterf))(GLuint, GLenum, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameteriv))(GLuint, GLenum, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameterfv))(GLuint, GLenum, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameterIiv))(GLuint, GLenum, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glSamplerParameterIuiv))(GLuint, GLenum, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSamplerParameteriv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSamplerParameterfv))(GLuint, GLenum, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSamplerParameterIiv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetSamplerParameterIuiv))(GLuint, GLenum, GLuint*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_sampler_objects

int CGL4ExtensionHandler::n_GetARBSamplerObjectsFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_sampler_objects

	if(!(GLFuncPtrName(glGenSamplers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenSamplers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteSamplers) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteSamplers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsSampler) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsSampler"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindSampler) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glBindSampler"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameteri) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glSamplerParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameterf) = (void(GLApi*)(GLuint, GLenum, float))wglGetProcAddress("glSamplerParameterf"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameteriv) = (void(GLApi*)(GLuint, GLenum, const int*))wglGetProcAddress("glSamplerParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameterfv) = (void(GLApi*)(GLuint, GLenum, const float*))wglGetProcAddress("glSamplerParameterfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameterIiv) = (void(GLApi*)(GLuint, GLenum, const int*))wglGetProcAddress("glSamplerParameterIiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glSamplerParameterIuiv) = (void(GLApi*)(GLuint, GLenum, const GLuint*))wglGetProcAddress("glSamplerParameterIuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSamplerParameteriv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetSamplerParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSamplerParameterfv) = (void(GLApi*)(GLuint, GLenum, float*))wglGetProcAddress("glGetSamplerParameterfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSamplerParameterIiv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetSamplerParameterIiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSamplerParameterIuiv) = (void(GLApi*)(GLuint, GLenum, GLuint*))wglGetProcAddress("glGetSamplerParameterIuiv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_sampler_objects

	GLEH_ARB_sampler_objects = !n_failed_functions && b_SupportedExtension("GL_ARB_sampler_objects");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_sampler_objects ---
 */

/*
 *								--- GL_ARB_texture_rgb10_a2ui (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_texture_rgb10_a2ui = false;

int CGL4ExtensionHandler::n_GetARBTextureRGB10A2uiFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_rgb10_a2ui = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_rgb10_a2ui");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_rgb10_a2ui ---
 */

/*
 *								--- GL_ARB_texture_swizzle (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_texture_swizzle = false;

int CGL4ExtensionHandler::n_GetARBTextureSwizzleFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_swizzle = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_swizzle");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_swizzle ---
 */

/*
 *								--- GL_ARB_timer_query (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_timer_query = false;

#ifdef __GENERATE_GL_ARB_timer_query

void (GLApi *GLFuncPtrName(glQueryCounter))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryObjecti64v))(GLuint, GLenum, GLint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryObjectui64v))(GLuint, GLenum, GLuint64*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_timer_query

int CGL4ExtensionHandler::n_GetARBTimerQueryFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_timer_query

	if(!(GLFuncPtrName(glQueryCounter) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glQueryCounter"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryObjecti64v) = (void(GLApi*)(GLuint, GLenum, GLint64*))wglGetProcAddress("glGetQueryObjecti64v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryObjectui64v) = (void(GLApi*)(GLuint, GLenum, GLuint64*))wglGetProcAddress("glGetQueryObjectui64v"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_timer_query

	GLEH_ARB_timer_query = !n_failed_functions && b_SupportedExtension("GL_ARB_timer_query");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_timer_query ---
 */

/*
 *								--- GL_ARB_instanced_arrays (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_instanced_arrays = false;

#ifdef __GENERATE_GL_ARB_instanced_arrays

void (GLApi *GLFuncPtrName(glVertexAttribDivisor))(GLuint, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_instanced_arrays

int CGL4ExtensionHandler::n_GetARBInstancedArraysFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_instanced_arrays

	if(!(GLFuncPtrName(glVertexAttribDivisor) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glVertexAttribDivisorARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_instanced_arrays

	GLEH_ARB_instanced_arrays = !n_failed_functions && b_SupportedExtension("GL_ARB_instanced_arrays");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_instanced_arrays ---
 */

/*
 *								--- GL_ARB_vertex_type_2_10_10_10_rev (part of OpenGL 3.3 core) ---
 */

bool GLEH_ARB_vertex_type_2_10_10_10_rev = false;

#ifdef __GENERATE_GL_ARB_vertex_type_2_10_10_10_rev

//void (GLApi *GLFuncPtrName(glVertexP2ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glVertexP3ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glVertexP4ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glVertexP2uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glVertexP3uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glVertexP4uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP1ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP2ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP3ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP4ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP1uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP2uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP3uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glTexCoordP4uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP1ui))(GLenum, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP2ui))(GLenum, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP3ui))(GLenum, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP4ui))(GLenum, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP1uiv))(GLenum, GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP2uiv))(GLenum, GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP3uiv))(GLenum, GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glMultiTexCoordP4uiv))(GLenum, GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glNormalP3ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glNormalP3uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glColorP3ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glColorP4ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glColorP3uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glColorP4uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glSecondaryColorP3ui))(GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
//void (GLApi *GLFuncPtrName(glSecondaryColorP3uiv))(GLenum, const GLuint*) INIT_FUNC_POINTERS; // deprecated as of OpenGL 3.0 (fixed-function vertex processing)
void (GLApi *GLFuncPtrName(glVertexAttribP1ui))(GLuint, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP2ui))(GLuint, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP3ui))(GLuint, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP4ui))(GLuint, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP1uiv))(GLuint, GLenum, GLboolean, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP2uiv))(GLuint, GLenum, GLboolean, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP3uiv))(GLuint, GLenum, GLboolean, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribP4uiv))(GLuint, GLenum, GLboolean, const GLuint*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_vertex_type_2_10_10_10_rev

int CGL4ExtensionHandler::n_GetARBVertexType2101010RevFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_vertex_type_2_10_10_10_rev

	//if(!(GLFuncPtrName(glVertexP2ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glVertexP2ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexP3ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glVertexP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexP4ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glVertexP4ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexP2uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glVertexP2uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexP3uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glVertexP3uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexP4uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glVertexP4uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP1ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glTexCoordP1ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP2ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glTexCoordP2ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP3ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glTexCoordP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP4ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glTexCoordP4ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP1uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glTexCoordP1uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP2uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glTexCoordP2uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP3uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glTexCoordP3uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTexCoordP4uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glTexCoordP4uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP1ui) = (void(GLApi*)(GLenum, GLenum, GLuint))wglGetProcAddress("glMultiTexCoordP1ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP2ui) = (void(GLApi*)(GLenum, GLenum, GLuint))wglGetProcAddress("glMultiTexCoordP2ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP3ui) = (void(GLApi*)(GLenum, GLenum, GLuint))wglGetProcAddress("glMultiTexCoordP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP4ui) = (void(GLApi*)(GLenum, GLenum, GLuint))wglGetProcAddress("glMultiTexCoordP4ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP1uiv) = (void(GLApi*)(GLenum, GLenum, const GLuint*))wglGetProcAddress("glMultiTexCoordP1uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP2uiv) = (void(GLApi*)(GLenum, GLenum, const GLuint*))wglGetProcAddress("glMultiTexCoordP2uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP3uiv) = (void(GLApi*)(GLenum, GLenum, const GLuint*))wglGetProcAddress("glMultiTexCoordP3uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glMultiTexCoordP4uiv) = (void(GLApi*)(GLenum, GLenum, const GLuint*))wglGetProcAddress("glMultiTexCoordP4uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glNormalP3ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glNormalP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glNormalP3uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glNormalP3uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glColorP3ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glColorP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glColorP4ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glColorP4ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glColorP3uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glColorP3uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glColorP4uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glColorP4uiv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glSecondaryColorP3ui) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glSecondaryColorP3ui"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glSecondaryColorP3uiv) = (void(GLApi*)(GLenum, const GLuint*))wglGetProcAddress("glSecondaryColorP3uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP1ui) = (void(GLApi*)(GLuint, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexAttribP1ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP2ui) = (void(GLApi*)(GLuint, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexAttribP2ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP3ui) = (void(GLApi*)(GLuint, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexAttribP3ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP4ui) = (void(GLApi*)(GLuint, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexAttribP4ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP1uiv) = (void(GLApi*)(GLuint, GLenum, GLboolean, const GLuint*))wglGetProcAddress("glVertexAttribP1uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP2uiv) = (void(GLApi*)(GLuint, GLenum, GLboolean, const GLuint*))wglGetProcAddress("glVertexAttribP2uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP3uiv) = (void(GLApi*)(GLuint, GLenum, GLboolean, const GLuint*))wglGetProcAddress("glVertexAttribP3uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribP4uiv) = (void(GLApi*)(GLuint, GLenum, GLboolean, const GLuint*))wglGetProcAddress("glVertexAttribP4uiv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_vertex_type_2_10_10_10_rev

	GLEH_ARB_vertex_type_2_10_10_10_rev = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_type_2_10_10_10_rev");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_type_2_10_10_10_rev ---
 */

/*
 *								--- GL_ARB_texture_query_lod (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_texture_query_lod = false;

int CGL4ExtensionHandler::n_GetARBTextureQueryLodFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_query_lod = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_query_lod");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_query_lod ---
 */

/*
 *								--- GL_ARB_draw_buffers_blend (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_draw_buffers_blend = false;

#ifdef __GENERATE_GL_ARB_draw_buffers_blend

void (GLApi *GLFuncPtrName(glBlendEquationi))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlendEquationSeparatei))(GLuint, GLenum, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlendFunci))(GLuint, GLenum, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlendFuncSeparatei))(GLuint, GLenum, GLenum, GLenum, GLenum) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_draw_buffers_blend

int CGL4ExtensionHandler::n_GetARBDrawBuffersBlendFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_draw_buffers_blend

	if(!(GLFuncPtrName(glBlendEquationi) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glBlendEquationiARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBlendEquationSeparatei) = (void(GLApi*)(GLuint, GLenum, GLenum))wglGetProcAddress("glBlendEquationSeparateiARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBlendFunci) = (void(GLApi*)(GLuint, GLenum, GLenum))wglGetProcAddress("glBlendFunciARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBlendFuncSeparatei) = (void(GLApi*)(GLuint, GLenum, GLenum, GLenum, GLenum))wglGetProcAddress("glBlendFuncSeparateiARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_draw_buffers_blend

	GLEH_ARB_draw_buffers_blend = !n_failed_functions && b_SupportedExtension("GL_ARB_draw_buffers_blend");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_draw_buffers_blend ---
 */

/*
 *								--- GL_ARB_draw_indirect (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_draw_indirect = false;

#ifdef __GENERATE_GL_ARB_draw_indirect

void (GLApi *GLFuncPtrName(glDrawArraysIndirect))(GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawElementsIndirect))(GLenum, GLenum, const void*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_draw_indirect

int CGL4ExtensionHandler::n_GetARBDrawIndirectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_draw_indirect

	if(!(GLFuncPtrName(glDrawArraysIndirect) = (void(GLApi*)(GLenum, const void*))wglGetProcAddress("glDrawArraysIndirect"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawElementsIndirect) = (void(GLApi*)(GLenum, GLenum, const void*))wglGetProcAddress("glDrawElementsIndirect"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_draw_indirect

	GLEH_ARB_draw_indirect = !n_failed_functions && b_SupportedExtension("GL_ARB_draw_indirect");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_draw_indirect ---
 */

/*
 *								--- GL_ARB_gpu_shader5 (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_gpu_shader5 = false;

int CGL4ExtensionHandler::n_GetARBGPUShader5FuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_gpu_shader5 = !n_failed_functions && b_SupportedExtension("GL_ARB_gpu_shader5");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_gpu_shader5 ---
 */

/*
 *								--- GL_ARB_gpu_shader_fp64 (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_gpu_shader_fp64 = false;

#ifdef __GENERATE_GL_ARB_gpu_shader_fp64

void (GLApi *GLFuncPtrName(glUniform1d))(int, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2d))(int, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3d))(int, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4d))(int, double, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform1dv))(int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform2dv))(int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform3dv))(int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniform4dv))(int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix2dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix2x3dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix2x4dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3x2dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix3x4dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4x2dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformMatrix4x3dv))(int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetUniformdv))(GLuint, int, double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1d))(GLuint, int, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2d))(GLuint, int, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3d))(GLuint, int, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4d))(GLuint, int, double, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_gpu_shader_fp64

int CGL4ExtensionHandler::n_GetARBGPUShaderFP64FuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_gpu_shader_fp64

	if(!(GLFuncPtrName(glUniform1d) = (void(GLApi*)(int, double))wglGetProcAddress("glUniform1d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform2d) = (void(GLApi*)(int, double, double))wglGetProcAddress("glUniform2d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform3d) = (void(GLApi*)(int, double, double, double))wglGetProcAddress("glUniform3d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform4d) = (void(GLApi*)(int, double, double, double, double))wglGetProcAddress("glUniform4d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform1dv) = (void(GLApi*)(int, GLsizei, const double*))wglGetProcAddress("glUniform1dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform2dv) = (void(GLApi*)(int, GLsizei, const double*))wglGetProcAddress("glUniform2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform3dv) = (void(GLApi*)(int, GLsizei, const double*))wglGetProcAddress("glUniform3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniform4dv) = (void(GLApi*)(int, GLsizei, const double*))wglGetProcAddress("glUniform4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix2dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix3dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix4dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix2x3dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix2x3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix2x4dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix2x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix3x2dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix3x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix3x4dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix3x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix4x2dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix4x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformMatrix4x3dv) = (void(GLApi*)(int, GLsizei, GLboolean, const double*))wglGetProcAddress("glUniformMatrix4x3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetUniformdv) = (void(GLApi*)(GLuint, int, double*))wglGetProcAddress("glGetUniformdv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1d) = (void(GLApi*)(GLuint, int, double))wglGetProcAddress("glProgramUniform1d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2d) = (void(GLApi*)(GLuint, int, double, double))wglGetProcAddress("glProgramUniform2d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3d) = (void(GLApi*)(GLuint, int, double, double, double))wglGetProcAddress("glProgramUniform3d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4d) = (void(GLApi*)(GLuint, int, double, double, double, double))wglGetProcAddress("glProgramUniform4d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform1dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2x3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4x3dv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_gpu_shader_fp64

	GLEH_ARB_gpu_shader_fp64 = !n_failed_functions && b_SupportedExtension("GL_ARB_gpu_shader_fp64");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_gpu_shader_fp64 ---
 */

/*
 *								--- GL_ARB_sample_shading (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_sample_shading = false;

#ifdef __GENERATE_GL_ARB_sample_shading

void (GLApi *GLFuncPtrName(glMinSampleShading))(GLclampf) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_sample_shading

int CGL4ExtensionHandler::n_GetARBSampleShadingFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_sample_shading

	if(!(GLFuncPtrName(glMinSampleShading) = (void(GLApi*)(GLclampf))wglGetProcAddress("glMinSampleShadingARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_sample_shading

	GLEH_ARB_sample_shading = !n_failed_functions && b_SupportedExtension("GL_ARB_sample_shading");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_sample_shading ---
 */

/*
 *								--- GL_ARB_shader_subroutine (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_shader_subroutine = false;

#ifdef __GENERATE_GL_ARB_shader_subroutine

int (GLApi *GLFuncPtrName(glGetSubroutineUniformLocation))(GLuint, GLenum, const GLchar*) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glGetSubroutineIndex))(GLuint, GLenum, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveSubroutineUniformiv))(GLuint, GLenum, GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveSubroutineUniformName))(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetActiveSubroutineName))(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformSubroutinesuiv))(GLenum, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetUniformSubroutineuiv))(GLenum, int, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramStageiv))(GLuint, GLenum, GLenum, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_shader_subroutine

int CGL4ExtensionHandler::n_GetARBShaderSubroutineFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_shader_subroutine

	if(!(GLFuncPtrName(glGetSubroutineUniformLocation) = (int(GLApi*)(GLuint, GLenum, const GLchar*))wglGetProcAddress("glGetSubroutineUniformLocation"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetSubroutineIndex) = (GLuint(GLApi*)(GLuint, GLenum, const GLchar*))wglGetProcAddress("glGetSubroutineIndex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveSubroutineUniformiv) = (void(GLApi*)(GLuint, GLenum, GLuint, GLenum, int*))wglGetProcAddress("glGetActiveSubroutineUniformiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveSubroutineUniformName) = (void(GLApi*)(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetActiveSubroutineUniformName"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetActiveSubroutineName) = (void(GLApi*)(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetActiveSubroutineName"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformSubroutinesuiv) = (void(GLApi*)(GLenum, GLsizei, const GLuint*))wglGetProcAddress("glUniformSubroutinesuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetUniformSubroutineuiv) = (void(GLApi*)(GLenum, int, GLuint*))wglGetProcAddress("glGetUniformSubroutineuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramStageiv) = (void(GLApi*)(GLuint, GLenum, GLenum, int*))wglGetProcAddress("glGetProgramStageiv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_shader_subroutine

	GLEH_ARB_shader_subroutine = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_subroutine");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_subroutine ---
 */

/*
 *								--- GL_ARB_tessellation_shader (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_tessellation_shader = false;

#ifdef __GENERATE_GL_ARB_tessellation_shader

void (GLApi *GLFuncPtrName(glPatchParameteri))(GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPatchParameterfv))(GLenum, const float*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_tessellation_shader

int CGL4ExtensionHandler::n_GetARBTessellationShaderFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_tessellation_shader

	if(!(GLFuncPtrName(glPatchParameteri) = (void(GLApi*)(GLenum, int))wglGetProcAddress("glPatchParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glPatchParameterfv) = (void(GLApi*)(GLenum, const float*))wglGetProcAddress("glPatchParameterfv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_tessellation_shader

	GLEH_ARB_tessellation_shader = !n_failed_functions && b_SupportedExtension("GL_ARB_tessellation_shader");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_tessellation_shader ---
 */

/*
 *								--- GL_ARB_texture_buffer_object_rgb32 (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_texture_buffer_object_rgb32 = false;

int CGL4ExtensionHandler::n_GetARBTextureBufferObjectRGB32FuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_buffer_object_rgb32 = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_buffer_object_rgb32");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_buffer_object_rgb32 ---
 */

/*
 *								--- GL_ARB_texture_cube_map_array (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_texture_cube_map_array = false;

int CGL4ExtensionHandler::n_GetARBTextureCubeMapArrayFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_cube_map_array = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_cube_map_array");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_cube_map_array ---
 */

/*
 *								--- GL_ARB_texture_gather (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_texture_gather = false;

int CGL4ExtensionHandler::n_GetARBTextureGatherFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_gather = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_gather");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_gather ---
 */

/*
 *								--- GL_ARB_transform_feedback2 (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_transform_feedback2 = false;

#ifdef __GENERATE_GL_ARB_transform_feedback2

void (GLApi *GLFuncPtrName(glBindTransformFeedback))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteTransformFeedbacks))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenTransformFeedbacks))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsTransformFeedback))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glPauseTransformFeedback))() INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glResumeTransformFeedback))() INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawTransformFeedback))(GLenum, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_transform_feedback2

int CGL4ExtensionHandler::n_GetARBTransformFeedback2FuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_transform_feedback2

	if(!(GLFuncPtrName(glBindTransformFeedback) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glBindTransformFeedback"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteTransformFeedbacks) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteTransformFeedbacks"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenTransformFeedbacks) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenTransformFeedbacks"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsTransformFeedback) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsTransformFeedback"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glPauseTransformFeedback) = (void(GLApi*)())wglGetProcAddress("glPauseTransformFeedback"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glResumeTransformFeedback) = (void(GLApi*)())wglGetProcAddress("glResumeTransformFeedback"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawTransformFeedback) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glDrawTransformFeedback"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_transform_feedback2

	GLEH_ARB_transform_feedback2 = !n_failed_functions && b_SupportedExtension("GL_ARB_transform_feedback2");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_transform_feedback2 ---
 */

/*
 *								--- GL_ARB_transform_feedback3 (part of OpenGL 4.0 core) ---
 */

bool GLEH_ARB_transform_feedback3 = false;

#ifdef __GENERATE_GL_ARB_transform_feedback3

void (GLApi *GLFuncPtrName(glDrawTransformFeedbackStream))(GLenum, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBeginQueryIndexed))(GLenum, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEndQueryIndexed))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryIndexediv))(GLenum, GLuint, GLenum, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_transform_feedback3

int CGL4ExtensionHandler::n_GetARBTransformFeedback3FuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_transform_feedback3

	if(!(GLFuncPtrName(glDrawTransformFeedbackStream) = (void(GLApi*)(GLenum, GLuint, GLuint))wglGetProcAddress("glDrawTransformFeedbackStream"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBeginQueryIndexed) = (void(GLApi*)(GLenum, GLuint, GLuint))wglGetProcAddress("glBeginQueryIndexed"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glEndQueryIndexed) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glEndQueryIndexed"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryIndexediv) = (void(GLApi*)(GLenum, GLuint, GLenum, int*))wglGetProcAddress("glGetQueryIndexediv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_transform_feedback3

	GLEH_ARB_transform_feedback3 = !n_failed_functions && b_SupportedExtension("GL_ARB_transform_feedback3");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_transform_feedback3 ---
 */

/*
 *								--- GL_ARB_ES2_compatibility (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_ES2_compatibility = false;

#ifdef __GENERATE_GL_ARB_ES2_compatibility

void (GLApi *GLFuncPtrName(glReleaseShaderCompiler))() INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glShaderBinary))(GLsizei, const GLuint*, GLenum, const void*, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetShaderPrecisionFormat))(GLenum, GLenum, int*, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDepthRangef))(GLclampf, GLclampf) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearDepthf))(GLclampf) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_ES2_compatibility

int CGL4ExtensionHandler::n_GetARBES2CompatibilityFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_ES2_compatibility

	if(!(GLFuncPtrName(glReleaseShaderCompiler) = (void(GLApi*)())wglGetProcAddress("glReleaseShaderCompiler"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glShaderBinary) = (void(GLApi*)(GLsizei, const GLuint*, GLenum, const void*, GLsizei))wglGetProcAddress("glShaderBinary"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetShaderPrecisionFormat) = (void(GLApi*)(GLenum, GLenum, int*, int*))wglGetProcAddress("glGetShaderPrecisionFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDepthRangef) = (void(GLApi*)(GLclampf, GLclampf))wglGetProcAddress("glDepthRangef"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearDepthf) = (void(GLApi*)(GLclampf))wglGetProcAddress("glClearDepthf"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_ES2_compatibility

	GLEH_ARB_ES2_compatibility = !n_failed_functions && b_SupportedExtension("GL_ARB_ES2_compatibility");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_ES2_compatibility ---
 */

/*
 *								--- GL_ARB_get_program_binary (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_get_program_binary = false;

#ifdef __GENERATE_GL_ARB_get_program_binary

void (GLApi *GLFuncPtrName(glGetProgramBinary))(GLuint, GLsizei, GLsizei*, GLenum*, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramBinary))(GLuint, GLenum, const void*, GLsizei) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramParameteri))(GLuint, GLenum, int) INIT_FUNC_POINTERS; // defined above

#endif // __GENERATE_GL_ARB_get_program_binary

int CGL4ExtensionHandler::n_GetARBGetProgramBinaryFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_get_program_binary

	if(!(GLFuncPtrName(glGetProgramBinary) = (void(GLApi*)(GLuint, GLsizei, GLsizei*, GLenum*, void*))wglGetProcAddress("glGetProgramBinary"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramBinary) = (void(GLApi*)(GLuint, GLenum, const void*, GLsizei))wglGetProcAddress("glProgramBinary"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramParameteri) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glProgramParameteri"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_get_program_binary

	GLEH_ARB_get_program_binary = !n_failed_functions && b_SupportedExtension("GL_ARB_get_program_binary");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_get_program_binary ---
 */

/*
 *								--- GL_ARB_separate_shader_objects (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_separate_shader_objects = false;

#ifdef __GENERATE_GL_ARB_separate_shader_objects

void (GLApi *GLFuncPtrName(glUseProgramStages))(GLuint, GLbitfield, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glActiveShaderProgram))(GLuint, GLuint) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glCreateShaderProgramv))(GLenum, GLsizei, const GLchar**) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindProgramPipeline))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteProgramPipelines))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenProgramPipelines))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsProgramPipeline))(GLuint) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramParameteri))(GLuint, GLenum, int) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glGetProgramPipelineiv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1i))(GLuint, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2i))(GLuint, int, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3i))(GLuint, int, int, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4i))(GLuint, int, int, int, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1ui))(GLuint, int, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2ui))(GLuint, int, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3ui))(GLuint, int, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4ui))(GLuint, int, GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1f))(GLuint, int, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2f))(GLuint, int, float, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3f))(GLuint, int, float, float, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4f))(GLuint, int, float, float, float, float) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramUniform1d))(GLuint, int, double) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform2d))(GLuint, int, double, double) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform3d))(GLuint, int, double, double, double) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform4d))(GLuint, int, double, double, double, double) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glProgramUniform1iv))(GLuint, int, GLsizei, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2iv))(GLuint, int, GLsizei, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3iv))(GLuint, int, GLsizei, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4iv))(GLuint, int, GLsizei, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1uiv))(GLuint, int, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2uiv))(GLuint, int, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3uiv))(GLuint, int, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4uiv))(GLuint, int, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform1fv))(GLuint, int, GLsizei, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform2fv))(GLuint, int, GLsizei, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform3fv))(GLuint, int, GLsizei, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniform4fv))(GLuint, int, GLsizei, const float*) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramUniform1dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform2dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform3dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniform4dv))(GLuint, int, GLsizei, const double*) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x3fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x2fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x4fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x2fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x4fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x3fv))(GLuint, int, GLsizei, GLboolean, const float*) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix2x4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x2dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix3x4dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
//void (GLApi *GLFuncPtrName(glProgramUniformMatrix4x3dv))(GLuint, int, GLsizei, GLboolean, const double*) INIT_FUNC_POINTERS; // defined above
void (GLApi *GLFuncPtrName(glValidateProgramPipeline))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramPipelineInfoLog))(GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_separate_shader_objects

int CGL4ExtensionHandler::n_GetARBSeparateShaderObjectsFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_separate_shader_objects

	if(!(GLFuncPtrName(glUseProgramStages) = (void(GLApi*)(GLuint, GLbitfield, GLuint))wglGetProcAddress("glUseProgramStages"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glActiveShaderProgram) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glActiveShaderProgram"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateShaderProgramv) = (GLuint(GLApi*)(GLenum, GLsizei, const GLchar**))wglGetProcAddress("glCreateShaderProgramv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindProgramPipeline) = (void(GLApi*)(GLuint))wglGetProcAddress("glBindProgramPipeline"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteProgramPipelines) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteProgramPipelines"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenProgramPipelines) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glGenProgramPipelines"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsProgramPipeline) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsProgramPipeline"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramParameteri) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glProgramParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramPipelineiv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetProgramPipelineiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1i) = (void(GLApi*)(GLuint, int, int))wglGetProcAddress("glProgramUniform1i"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2i) = (void(GLApi*)(GLuint, int, int, int))wglGetProcAddress("glProgramUniform2i"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3i) = (void(GLApi*)(GLuint, int, int, int, int))wglGetProcAddress("glProgramUniform3i"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4i) = (void(GLApi*)(GLuint, int, int, int, int, int))wglGetProcAddress("glProgramUniform4i"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1ui) = (void(GLApi*)(GLuint, int, GLuint))wglGetProcAddress("glProgramUniform1ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2ui) = (void(GLApi*)(GLuint, int, GLuint, GLuint))wglGetProcAddress("glProgramUniform2ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3ui) = (void(GLApi*)(GLuint, int, GLuint, GLuint, GLuint))wglGetProcAddress("glProgramUniform3ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4ui) = (void(GLApi*)(GLuint, int, GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glProgramUniform4ui"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1f) = (void(GLApi*)(GLuint, int, float))wglGetProcAddress("glProgramUniform1f"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2f) = (void(GLApi*)(GLuint, int, float, float))wglGetProcAddress("glProgramUniform2f"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3f) = (void(GLApi*)(GLuint, int, float, float, float))wglGetProcAddress("glProgramUniform3f"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4f) = (void(GLApi*)(GLuint, int, float, float, float, float))wglGetProcAddress("glProgramUniform4f"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1d) = (void(GLApi*)(GLuint, int, double))wglGetProcAddress("glProgramUniform1d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2d) = (void(GLApi*)(GLuint, int, double, double))wglGetProcAddress("glProgramUniform2d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3d) = (void(GLApi*)(GLuint, int, double, double, double))wglGetProcAddress("glProgramUniform3d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4d) = (void(GLApi*)(GLuint, int, double, double, double, double))wglGetProcAddress("glProgramUniform4d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1iv) = (void(GLApi*)(GLuint, int, GLsizei, const int*))wglGetProcAddress("glProgramUniform1iv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2iv) = (void(GLApi*)(GLuint, int, GLsizei, const int*))wglGetProcAddress("glProgramUniform2iv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3iv) = (void(GLApi*)(GLuint, int, GLsizei, const int*))wglGetProcAddress("glProgramUniform3iv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4iv) = (void(GLApi*)(GLuint, int, GLsizei, const int*))wglGetProcAddress("glProgramUniform4iv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1uiv) = (void(GLApi*)(GLuint, int, GLsizei, const GLuint*))wglGetProcAddress("glProgramUniform1uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2uiv) = (void(GLApi*)(GLuint, int, GLsizei, const GLuint*))wglGetProcAddress("glProgramUniform2uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3uiv) = (void(GLApi*)(GLuint, int, GLsizei, const GLuint*))wglGetProcAddress("glProgramUniform3uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4uiv) = (void(GLApi*)(GLuint, int, GLsizei, const GLuint*))wglGetProcAddress("glProgramUniform4uiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1fv) = (void(GLApi*)(GLuint, int, GLsizei, const float*))wglGetProcAddress("glProgramUniform1fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2fv) = (void(GLApi*)(GLuint, int, GLsizei, const float*))wglGetProcAddress("glProgramUniform2fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3fv) = (void(GLApi*)(GLuint, int, GLsizei, const float*))wglGetProcAddress("glProgramUniform3fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4fv) = (void(GLApi*)(GLuint, int, GLsizei, const float*))wglGetProcAddress("glProgramUniform4fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform1dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform1dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform2dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform3dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniform4dv) = (void(GLApi*)(GLuint, int, GLsizei, const double*))wglGetProcAddress("glProgramUniform4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix2fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix3fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix4fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x3fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix2x3fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x2fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix3x2fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x4fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix2x4fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x2fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix4x2fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x4fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix3x4fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x3fv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const float*))wglGetProcAddress("glProgramUniformMatrix4x3fv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2x3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix2x4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix2x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x2dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4x2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix3x4dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix3x4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformMatrix4x3dv) = (void(GLApi*)(GLuint, int, GLsizei, GLboolean, const double*))wglGetProcAddress("glProgramUniformMatrix4x3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glValidateProgramPipeline) = (void(GLApi*)(GLuint))wglGetProcAddress("glValidateProgramPipeline"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramPipelineInfoLog) = (void(GLApi*)(GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetProgramPipelineInfoLog"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_separate_shader_objects

	GLEH_ARB_separate_shader_objects = !n_failed_functions && b_SupportedExtension("GL_ARB_separate_shader_objects");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_separate_shader_objects ---
 */

/*
 *								--- GL_ARB_shader_precision (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_shader_precision = false;

int CGL4ExtensionHandler::n_GetARBShaderPrecisionFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_precision = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_precision");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_precision ---
 */

/*
 *								--- GL_ARB_vertex_attrib_64bit (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_vertex_attrib_64bit = false;

#ifdef __GENERATE_GL_ARB_vertex_attrib_64bit

void (GLApi *GLFuncPtrName(glVertexAttribL1d))(GLuint, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL2d))(GLuint, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL3d))(GLuint, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL4d))(GLuint, double, double, double, double) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL1dv))(GLuint, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL2dv))(GLuint, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL3dv))(GLuint, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL4dv))(GLuint, const double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribLPointer))(GLuint, int, GLenum, GLsizei, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribLdv))(GLuint, GLenum, double*) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glVertexArrayVertexAttribLOffsetEXT))(GLuint, GLuint, GLuint, int, GLenum, GLsizei, GLintptr) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)

#endif // __GENERATE_GL_ARB_vertex_attrib_64bit

int CGL4ExtensionHandler::n_GetARBVertexAttrib64bitFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_vertex_attrib_64bit

	if(!(GLFuncPtrName(glVertexAttribL1d) = (void(GLApi*)(GLuint, double))wglGetProcAddress("glVertexAttribL1d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL2d) = (void(GLApi*)(GLuint, double, double))wglGetProcAddress("glVertexAttribL2d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL3d) = (void(GLApi*)(GLuint, double, double, double))wglGetProcAddress("glVertexAttribL3d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL4d) = (void(GLApi*)(GLuint, double, double, double, double))wglGetProcAddress("glVertexAttribL4d"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL1dv) = (void(GLApi*)(GLuint, const double*))wglGetProcAddress("glVertexAttribL1dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL2dv) = (void(GLApi*)(GLuint, const double*))wglGetProcAddress("glVertexAttribL2dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL3dv) = (void(GLApi*)(GLuint, const double*))wglGetProcAddress("glVertexAttribL3dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL4dv) = (void(GLApi*)(GLuint, const double*))wglGetProcAddress("glVertexAttribL4dv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribLPointer) = (void(GLApi*)(GLuint, int, GLenum, GLsizei, const void*))wglGetProcAddress("glVertexAttribLPointer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetVertexAttribLdv) = (void(GLApi*)(GLuint, GLenum, double*))wglGetProcAddress("glGetVertexAttribLdv"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexAttribLOffsetEXT) = (void(GLApi*)(GLuint, GLuint, GLuint, int, GLenum, GLsizei, GLintptr))wglGetProcAddress("glVertexArrayVertexAttribLOffsetEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_vertex_attrib_64bit

	GLEH_ARB_vertex_attrib_64bit = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_attrib_64bit");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_attrib_64bit ---
 */

/*
 *								--- GL_ARB_viewport_array (part of OpenGL 4.1 core) ---
 */

bool GLEH_ARB_viewport_array = false;

#ifdef __GENERATE_GL_ARB_viewport_array

void (GLApi *GLFuncPtrName(glViewportArrayv))(GLuint, GLsizei, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glViewportIndexedf))(GLuint, float, float, float, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glViewportIndexedfv))(GLuint, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glScissorArrayv))(GLuint, GLsizei, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glScissorIndexed))(GLuint, int, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glScissorIndexedv))(GLuint, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDepthRangeArrayv))(GLuint, GLsizei, const GLclampd*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDepthRangeIndexed))(GLuint, GLclampd, GLclampd) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetFloati_v))(GLenum, GLuint, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetDoublei_v))(GLenum, GLuint, double*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetIntegerIndexediv))(GLenum, GLuint, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEnableIndexed))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDisableIndexed))(GLenum, GLuint) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsEnabledIndexed))(GLenum, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_viewport_array

int CGL4ExtensionHandler::n_GetARBViewportArrayFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_viewport_array

	if(!(GLFuncPtrName(glViewportArrayv) = (void(GLApi*)(GLuint, GLsizei, const float*))wglGetProcAddress("glViewportArrayv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glViewportIndexedf) = (void(GLApi*)(GLuint, float, float, float, float))wglGetProcAddress("glViewportIndexedf"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glViewportIndexedfv) = (void(GLApi*)(GLuint, const float*))wglGetProcAddress("glViewportIndexedfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glScissorArrayv) = (void(GLApi*)(GLuint, GLsizei, const int*))wglGetProcAddress("glScissorArrayv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glScissorIndexed) = (void(GLApi*)(GLuint, int, int, GLsizei, GLsizei))wglGetProcAddress("glScissorIndexed"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glScissorIndexedv) = (void(GLApi*)(GLuint, const int*))wglGetProcAddress("glScissorIndexedv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDepthRangeArrayv) = (void(GLApi*)(GLuint, GLsizei, const GLclampd*))wglGetProcAddress("glDepthRangeArrayv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDepthRangeIndexed) = (void(GLApi*)(GLuint, GLclampd, GLclampd))wglGetProcAddress("glDepthRangeIndexed"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetFloati_v) = (void(GLApi*)(GLenum, GLuint, float*))wglGetProcAddress("glGetFloati_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetDoublei_v) = (void(GLApi*)(GLenum, GLuint, double*))wglGetProcAddress("glGetDoublei_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetIntegerIndexediv) = (void(GLApi*)(GLenum, GLuint, int*))wglGetProcAddress("glGetIntegeri_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glEnableIndexed) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glEnablei"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDisableIndexed) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glDisablei"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsEnabledIndexed) = (GLboolean(GLApi*)(GLenum, GLuint))wglGetProcAddress("glIsEnabledi"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_viewport_array

	GLEH_ARB_viewport_array = !n_failed_functions && b_SupportedExtension("GL_ARB_viewport_array");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_viewport_array ---
 */

/*
 *								--- GL_ARB_texture_compression_bptc (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_texture_compression_bptc = false;

int CGL4ExtensionHandler::n_GetARBTextureCompressionBptcFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_compression_bptc = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_compression_bptc");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_compression_bptc ---
 */

/*
 *								--- GL_ARB_compressed_texture_pixel_storage (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_compressed_texture_pixel_storage = false;

int CGL4ExtensionHandler::n_GetARBCompressedTexturePixelStorageFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_compressed_texture_pixel_storage = !n_failed_functions && b_SupportedExtension("GL_ARB_compressed_texture_pixel_storage");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_compressed_texture_pixel_storage ---
 */

/*
 *								--- GL_ARB_shader_atomic_counters (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_shader_atomic_counters = false;

#ifdef __GENERATE_GL_ARB_shader_atomic_counters

void (GLApi *GLFuncPtrName(glGetActiveAtomicCounterBufferiv))(GLuint, GLuint, GLenum, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_shader_atomic_counters

int CGL4ExtensionHandler::n_GetARBShaderAtomicCountersFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_shader_atomic_counters

	if(!(GLFuncPtrName(glGetActiveAtomicCounterBufferiv) = (void(GLApi*)(GLuint, GLuint, GLenum, int*))wglGetProcAddress("glGetActiveAtomicCounterBufferiv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_shader_atomic_counters

	GLEH_ARB_shader_atomic_counters = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_atomic_counters");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_atomic_counters ---
 */

/*
 *								--- GL_ARB_texture_storage (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_texture_storage = false;

#ifdef __GENERATE_GL_ARB_texture_storage

void (GLApi *GLFuncPtrName(glTexStorage1D))(GLenum, GLsizei, GLenum, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexStorage2D))(GLenum, GLsizei, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexStorage3D))(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage1DEXT))(GLuint, GLenum, GLsizei, GLenum, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage2DEXT))(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage3DEXT))(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_texture_storage

int CGL4ExtensionHandler::n_GetARBTextureStorageFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_storage

	if(!(GLFuncPtrName(glTexStorage1D) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei))wglGetProcAddress("glTexStorage1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTexStorage2D) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei, GLsizei))wglGetProcAddress("glTexStorage2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTexStorage3D) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei))wglGetProcAddress("glTexStorage3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage1DEXT) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLenum, GLsizei))wglGetProcAddress("glTextureStorage1DEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage2DEXT) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei))wglGetProcAddress("glTextureStorage2DEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage3DEXT) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei))wglGetProcAddress("glTextureStorage3DEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_storage

	GLEH_ARB_texture_storage = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_storage");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_storage ---
 */

/*
 *								--- GL_ARB_transform_feedback_instanced (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_transform_feedback_instanced = false;

#ifdef __GENERATE_GL_ARB_transform_feedback_instanced

void (GLApi *GLFuncPtrName(glDrawTransformFeedbackInstanced))(GLenum, GLuint, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawTransformFeedbackStreamInstanced))(GLenum, GLuint, GLuint, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_transform_feedback_instanced

int CGL4ExtensionHandler::n_GetARBTransformFeedbackInstancedFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_transform_feedback_instanced

	if(!(GLFuncPtrName(glDrawTransformFeedbackInstanced) = (void(GLApi*)(GLenum, GLuint, GLsizei))wglGetProcAddress("glDrawTransformFeedbackInstanced"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawTransformFeedbackStreamInstanced) = (void(GLApi*)(GLenum, GLuint, GLuint, GLsizei))wglGetProcAddress("glDrawTransformFeedbackStreamInstanced"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_transform_feedback_instanced

	GLEH_ARB_transform_feedback_instanced = !n_failed_functions && b_SupportedExtension("GL_ARB_transform_feedback_instanced");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_transform_feedback_instanced ---
 */

/*
 *								--- GL_ARB_base_instance (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_base_instance = false;

#ifdef __GENERATE_GL_ARB_base_instance

void (GLApi *GLFuncPtrName(glDrawArraysInstancedBaseInstance))(GLenum, int, GLsizei, GLsizei, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawElementsInstancedBaseInstance))(GLenum, GLsizei, GLenum, const void*, GLsizei, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawElementsInstancedBaseVertexBaseInstance))(GLenum, GLsizei, GLenum, const void*, GLsizei, int, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_base_instance

int CGL4ExtensionHandler::n_GetARBBaseInstanceFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_base_instance

	if(!(GLFuncPtrName(glDrawArraysInstancedBaseInstance) = (void(GLApi*)(GLenum, int, GLsizei, GLsizei, GLuint))wglGetProcAddress("glDrawArraysInstancedBaseInstance"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawElementsInstancedBaseInstance) = (void(GLApi*)(GLenum, GLsizei, GLenum, const void*, GLsizei, GLuint))wglGetProcAddress("glDrawElementsInstancedBaseInstance"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawElementsInstancedBaseVertexBaseInstance) = (void(GLApi*)(GLenum, GLsizei, GLenum, const void*, GLsizei, int, GLuint))wglGetProcAddress("glDrawElementsInstancedBaseVertexBaseInstance"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_base_instance

	GLEH_ARB_base_instance = !n_failed_functions && b_SupportedExtension("GL_ARB_base_instance");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_base_instance ---
 */

/*
 *								--- GL_ARB_shader_image_load_store (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_shader_image_load_store = false;

#ifdef __GENERATE_GL_ARB_shader_image_load_store

void (GLApi *GLFuncPtrName(glBindImageTexture))(GLuint, GLuint, int, GLboolean, int, GLenum, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMemoryBarrier))(GLbitfield) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_shader_image_load_store

int CGL4ExtensionHandler::n_GetARBShaderImageLoadStoreFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_shader_image_load_store

	if(!(GLFuncPtrName(glBindImageTexture) = (void(GLApi*)(GLuint, GLuint, int, GLboolean, int, GLenum, GLenum))wglGetProcAddress("glBindImageTexture"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMemoryBarrier) = (void(GLApi*)(GLbitfield))wglGetProcAddress("glMemoryBarrier"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_shader_image_load_store

	GLEH_ARB_shader_image_load_store = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_image_load_store");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_image_load_store ---
 */

/*
 *								--- GL_ARB_conservative_depth (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_conservative_depth = false;

int CGL4ExtensionHandler::n_GetARBConservativeDepthFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_conservative_depth = !n_failed_functions && b_SupportedExtension("GL_ARB_conservative_depth");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_conservative_depth ---
 */

/*
 *								--- GL_ARB_shading_language_420pack (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_shading_language_420pack = false;

int CGL4ExtensionHandler::n_GetARBShadingLanguage420packFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shading_language_420pack = !n_failed_functions && b_SupportedExtension("GL_ARB_shading_language_420pack");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shading_language_420pack ---
 */

/*
 *								--- GL_ARB_internalformat_query (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_internalformat_query = false;

#ifdef __GENERATE_GL_ARB_internalformat_query

void (GLApi *GLFuncPtrName(glGetInternalformativ))(GLenum, GLenum, GLenum, GLsizei, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_internalformat_query

int CGL4ExtensionHandler::n_GetARBInternalformatQueryFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_internalformat_query

	if(!(GLFuncPtrName(glGetInternalformativ) = (void(GLApi*)(GLenum, GLenum, GLenum, GLsizei, int*))wglGetProcAddress("glGetInternalformativ"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_internalformat_query

	GLEH_ARB_internalformat_query = !n_failed_functions && b_SupportedExtension("GL_ARB_internalformat_query");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_internalformat_query ---
 */

/*
 *								--- GL_ARB_map_buffer_alignment (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_map_buffer_alignment = false;

int CGL4ExtensionHandler::n_GetARBMapBufferAlignmentFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_map_buffer_alignment = !n_failed_functions && b_SupportedExtension("GL_ARB_map_buffer_alignment");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_map_buffer_alignment ---
 */

/*
 *								--- GL_ARB_shading_language_packing (part of OpenGL 4.2 core) ---
 */

bool GLEH_ARB_shading_language_packing = false;

int CGL4ExtensionHandler::n_GetARBShadingLanguagePackingFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shading_language_packing = !n_failed_functions && b_SupportedExtension("GL_ARB_shading_language_packing");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shading_language_packing ---
 */

/*
 *								--- GL_ARB_arrays_of_arrays (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_arrays_of_arrays = false;

int CGL4ExtensionHandler::n_GetARBArraysOfArraysFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_arrays_of_arrays = !n_failed_functions && b_SupportedExtension("GL_ARB_arrays_of_arrays");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_arrays_of_arrays ---
 */

/*
 *								--- GL_ARB_ES3_compatibility (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_ES3_compatibility = false;

int CGL4ExtensionHandler::n_GetARBES3CompatibilityFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_ES3_compatibility = !n_failed_functions && b_SupportedExtension("GL_ARB_ES3_compatibility");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_ES3_compatibility ---
 */

/*
 *								--- GL_ARB_clear_buffer_object (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_clear_buffer_object = false;

#ifdef __GENERATE_GL_ARB_clear_buffer_object

void (GLApi *GLFuncPtrName(glClearBufferData))(GLenum, GLenum, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearBufferSubData))(GLenum, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedBufferDataEXT))(GLuint, GLenum, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedBufferSubDataEXT))(GLuint, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_clear_buffer_object

int CGL4ExtensionHandler::n_GetARBClearBufferObjectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_clear_buffer_object

	if(!(GLFuncPtrName(glClearBufferData) = (void(GLApi*)(GLenum, GLenum, GLenum, GLenum, const void*))wglGetProcAddress("glClearBufferData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearBufferSubData) = (void(GLApi*)(GLenum, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*))wglGetProcAddress("glClearBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedBufferDataEXT) = (void(GLApi*)(GLuint, GLenum, GLenum, GLenum, const void*))wglGetProcAddress("glClearNamedBufferDataEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedBufferSubDataEXT) = (void(GLApi*)(GLuint, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*))wglGetProcAddress("glClearNamedBufferSubDataEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_clear_buffer_object

	GLEH_ARB_clear_buffer_object = !n_failed_functions && b_SupportedExtension("GL_ARB_clear_buffer_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_clear_buffer_object ---
 */

/*
 *								--- GL_ARB_compute_shader (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_compute_shader = false;

#ifdef __GENERATE_GL_ARB_compute_shader

void (GLApi *GLFuncPtrName(glDispatchCompute))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDispatchComputeIndirect))(GLintptr) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_compute_shader

int CGL4ExtensionHandler::n_GetARBComputeShaderFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_compute_shader

	if(!(GLFuncPtrName(glDispatchCompute) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glDispatchCompute"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDispatchComputeIndirect) = (void(GLApi*)(GLintptr))wglGetProcAddress("glDispatchComputeIndirect"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_compute_shader

	GLEH_ARB_compute_shader = !n_failed_functions && b_SupportedExtension("GL_ARB_compute_shader");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_compute_shader ---
 */

/*
 *								--- GL_ARB_copy_image (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_copy_image = false;

#ifdef __GENERATE_GL_ARB_copy_image

void (GLApi *GLFuncPtrName(glCopyImageSubData))(GLuint, GLenum, int, int, int, int, GLuint, GLenum, int, int, int, int, GLsizei, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_copy_image

int CGL4ExtensionHandler::n_GetARBCopyImageFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_copy_image

	if(!(GLFuncPtrName(glCopyImageSubData) = (void(GLApi*)(GLuint, GLenum, int, int, int, int, GLuint, GLenum, int, int, int, int, GLsizei, GLsizei, GLsizei))wglGetProcAddress("glCopyImageSubData"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_copy_image

	GLEH_ARB_copy_image = !n_failed_functions && b_SupportedExtension("GL_ARB_copy_image");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_copy_image ---
 */

/*
 *								--- GL_ARB_debug_group (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_debug_group = false;

int CGL4ExtensionHandler::n_GetARBDebugGroupFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_debug_group = !n_failed_functions && b_SupportedExtension("GL_ARB_debug_group");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_debug_group ---
 */

/*
 *								--- GL_ARB_debug_label (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_debug_label = false;

int CGL4ExtensionHandler::n_GetARBDebugLabelFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_debug_label = !n_failed_functions && b_SupportedExtension("GL_ARB_debug_label");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_debug_label ---
 */

/*
 *								--- GL_ARB_debug_output2 (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_debug_output2 = false;

int CGL4ExtensionHandler::n_GetARBDebugOutput2FuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_debug_output2 = !n_failed_functions && b_SupportedExtension("GL_ARB_debug_output2");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_debug_output2 ---
 */

/*
 *								--- GL_ARB_debug_output (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_debug_output = false;

#ifdef __GENERATE_GL_ARB_debug_output

void (GLApi *GLFuncPtrName(glDebugMessageControl))(GLenum, GLenum, GLenum, GLsizei, const GLuint*, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDebugMessageInsert))(GLenum, GLenum, GLuint, GLenum, GLsizei, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDebugMessageCallback))(GLDEBUGPROCARB, void*) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glGetDebugMessageLog))(GLuint, GLsizei, GLenum*, GLenum*, GLuint*, GLenum*, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetPointerv))(GLenum, void**) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_debug_output

int CGL4ExtensionHandler::n_GetARBDebugOutputFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_debug_output

	if(!(GLFuncPtrName(glDebugMessageControl) = (void(GLApi*)(GLenum, GLenum, GLenum, GLsizei, const GLuint*, GLboolean))wglGetProcAddress("glDebugMessageControlARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDebugMessageInsert) = (void(GLApi*)(GLenum, GLenum, GLuint, GLenum, GLsizei, const GLchar*))wglGetProcAddress("glDebugMessageInsertARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDebugMessageCallback) = (void(GLApi*)(GLDEBUGPROCARB, void*))wglGetProcAddress("glDebugMessageCallbackARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetDebugMessageLog) = (GLuint(GLApi*)(GLuint, GLsizei, GLenum*, GLenum*, GLuint*, GLenum*, GLsizei*, GLchar*))wglGetProcAddress("glGetDebugMessageLogARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetPointerv) = (void(GLApi*)(GLenum, void**))wglGetProcAddress("glGetPointerv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_debug_output

	GLEH_ARB_debug_output = !n_failed_functions && b_SupportedExtension("GL_ARB_debug_output");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_debug_output ---
 */

/*
 *								--- GL_ARB_explicit_uniform_location (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_explicit_uniform_location = false;

int CGL4ExtensionHandler::n_GetARBExplicitUniformLocationFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_explicit_uniform_location = !n_failed_functions && b_SupportedExtension("GL_ARB_explicit_uniform_location");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_explicit_uniform_location ---
 */

/*
 *								--- GL_ARB_fragment_layer_viewport (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_fragment_layer_viewport = false;

int CGL4ExtensionHandler::n_GetARBFragmentLayerViewportFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_fragment_layer_viewport = !n_failed_functions && b_SupportedExtension("GL_ARB_fragment_layer_viewport");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_fragment_layer_viewport ---
 */

/*
 *								--- GL_ARB_framebuffer_no_attachments (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_framebuffer_no_attachments = false;

#ifdef __GENERATE_GL_ARB_framebuffer_no_attachments

void (GLApi *GLFuncPtrName(glFramebufferParameteri))(GLenum, GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetFramebufferParameteriv))(GLenum, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferParameteriEXT))(GLuint, GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedFramebufferParameterivEXT))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_framebuffer_no_attachments

int CGL4ExtensionHandler::n_GetARBFramebufferNoAttachmentsFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_framebuffer_no_attachments

	if(!(GLFuncPtrName(glFramebufferParameteri) = (void(GLApi*)(GLenum, GLenum, int))wglGetProcAddress("glFramebufferParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetFramebufferParameteriv) = (void(GLApi*)(GLenum, GLenum, int*))wglGetProcAddress("glGetFramebufferParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferParameteriEXT) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glNamedFramebufferParameteriEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedFramebufferParameterivEXT) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetNamedFramebufferParameterivEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_framebuffer_no_attachments

	GLEH_ARB_framebuffer_no_attachments = !n_failed_functions && b_SupportedExtension("GL_ARB_framebuffer_no_attachments");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_framebuffer_no_attachments ---
 */

/*
 *								--- GL_ARB_internalformat_query2 (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_internalformat_query2 = false;

#ifdef __GENERATE_GL_ARB_internalformat_query2

void (GLApi *GLFuncPtrName(glGetInternalformati64v))(GLenum, GLenum, GLenum, GLsizei, GLint64*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_internalformat_query2

int CGL4ExtensionHandler::n_GetARBInternalformatQuery2FuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_internalformat_query2

	if(!(GLFuncPtrName(glGetInternalformati64v) = (void(GLApi*)(GLenum, GLenum, GLenum, GLsizei, GLint64*))wglGetProcAddress("glGetInternalformati64v"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_internalformat_query2

	GLEH_ARB_internalformat_query2 = !n_failed_functions && b_SupportedExtension("GL_ARB_internalformat_query2");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_internalformat_query2 ---
 */

/*
 *								--- GL_ARB_invalidate_subdata (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_invalidate_subdata = false;

#ifdef __GENERATE_GL_ARB_invalidate_subdata

void (GLApi *GLFuncPtrName(glInvalidateTexSubImage))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateTexImage))(GLuint, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateBufferSubData))(GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateBufferData))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateFramebuffer))(GLenum, GLsizei, const GLenum*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateSubFramebuffer))(GLenum, GLsizei, const GLenum*, int, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_invalidate_subdata

int CGL4ExtensionHandler::n_GetARBInvalidateSubdataFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_invalidate_subdata

	if(!(GLFuncPtrName(glInvalidateTexSubImage) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei))wglGetProcAddress("glInvalidateTexSubImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateTexImage) = (void(GLApi*)(GLuint, int))wglGetProcAddress("glInvalidateTexImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateBufferSubData) = (void(GLApi*)(GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glInvalidateBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateBufferData) = (void(GLApi*)(GLuint))wglGetProcAddress("glInvalidateBufferData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateFramebuffer) = (void(GLApi*)(GLenum, GLsizei, const GLenum*))wglGetProcAddress("glInvalidateFramebuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateSubFramebuffer) = (void(GLApi*)(GLenum, GLsizei, const GLenum*, int, int, GLsizei, GLsizei))wglGetProcAddress("glInvalidateSubFramebuffer"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_invalidate_subdata

	GLEH_ARB_invalidate_subdata = !n_failed_functions && b_SupportedExtension("GL_ARB_invalidate_subdata");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_invalidate_subdata ---
 */

/*
 *								--- GL_ARB_multi_draw_indirect (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_multi_draw_indirect = false;

#ifdef __GENERATE_GL_ARB_multi_draw_indirect

void (GLApi *GLFuncPtrName(glMultiDrawArraysIndirect))(GLenum, const void*, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiDrawElementsIndirect))(GLenum, GLenum, const void*, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_multi_draw_indirect

int CGL4ExtensionHandler::n_GetARBMultiDrawIndirectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_multi_draw_indirect

	if(!(GLFuncPtrName(glMultiDrawArraysIndirect) = (void(GLApi*)(GLenum, const void*, GLsizei, GLsizei))wglGetProcAddress("glMultiDrawArraysIndirect"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMultiDrawElementsIndirect) = (void(GLApi*)(GLenum, GLenum, const void*, GLsizei, GLsizei))wglGetProcAddress("glMultiDrawElementsIndirect"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_multi_draw_indirect

	GLEH_ARB_multi_draw_indirect = !n_failed_functions && b_SupportedExtension("GL_ARB_multi_draw_indirect");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_multi_draw_indirect ---
 */

/*
 *								--- GL_ARB_program_interface_query (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_program_interface_query = false;

#ifdef __GENERATE_GL_ARB_program_interface_query

void (GLApi *GLFuncPtrName(glGetProgramInterfaceiv))(GLuint, GLenum, GLenum, int*) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glGetProgramResourceIndex))(GLuint, GLenum, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramResourceName))(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetProgramResourceiv))(GLuint, GLenum, GLuint, GLsizei, const GLenum*, GLsizei, GLsizei*, int*) INIT_FUNC_POINTERS;
int (GLApi *GLFuncPtrName(glGetProgramResourceLocation))(GLuint, GLenum, const GLchar*) INIT_FUNC_POINTERS;
int (GLApi *GLFuncPtrName(glGetProgramResourceLocationIndex))(GLuint, GLenum, const GLchar*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_program_interface_query

int CGL4ExtensionHandler::n_GetARBProgramInterfaceQueryFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_program_interface_query

	if(!(GLFuncPtrName(glGetProgramInterfaceiv) = (void(GLApi*)(GLuint, GLenum, GLenum, int*))wglGetProcAddress("glGetProgramInterfaceiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramResourceIndex) = (GLuint(GLApi*)(GLuint, GLenum, const GLchar*))wglGetProcAddress("glGetProgramResourceIndex"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramResourceName) = (void(GLApi*)(GLuint, GLenum, GLuint, GLsizei, GLsizei*, GLchar*))wglGetProcAddress("glGetProgramResourceName"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramResourceiv) = (void(GLApi*)(GLuint, GLenum, GLuint, GLsizei, const GLenum*, GLsizei, GLsizei*, int*))wglGetProcAddress("glGetProgramResourceiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramResourceLocation) = (int(GLApi*)(GLuint, GLenum, const GLchar*))wglGetProcAddress("glGetProgramResourceLocation"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetProgramResourceLocationIndex) = (int(GLApi*)(GLuint, GLenum, const GLchar*))wglGetProcAddress("glGetProgramResourceLocationIndex"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_program_interface_query

	GLEH_ARB_program_interface_query = !n_failed_functions && b_SupportedExtension("GL_ARB_program_interface_query");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_program_interface_query ---
 */

/*
 *								--- GL_ARB_robust_buffer_access_behavior (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_robust_buffer_access_behavior = false;

int CGL4ExtensionHandler::n_GetARBRobustBufferAccessBehaviorFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_robust_buffer_access_behavior = !n_failed_functions && b_SupportedExtension("GL_ARB_robust_buffer_access_behavior");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_robust_buffer_access_behavior ---
 */

/*
 *								--- GL_ARB_shader_image_size (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_shader_image_size = false;

int CGL4ExtensionHandler::n_GetARBShaderImageSizeFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_image_size = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_image_size");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_image_size ---
 */

/*
 *								--- GL_ARB_shader_storage_buffer_object (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_shader_storage_buffer_object = false;

#ifdef __GENERATE_GL_ARB_shader_storage_buffer_object

void (GLApi *GLFuncPtrName(glShaderStorageBlockBinding))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_shader_storage_buffer_object

int CGL4ExtensionHandler::n_GetARBShaderStorageBufferObjectFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_shader_storage_buffer_object

	if(!(GLFuncPtrName(glShaderStorageBlockBinding) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glShaderStorageBlockBinding"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_shader_storage_buffer_object

	GLEH_ARB_shader_storage_buffer_object = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_storage_buffer_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_storage_buffer_object ---
 */

/*
 *								--- GL_ARB_stencil_texturing (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_stencil_texturing = false;

int CGL4ExtensionHandler::n_GetARBStencilTexturingFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_stencil_texturing = !n_failed_functions && b_SupportedExtension("GL_ARB_stencil_texturing");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_stencil_texturing ---
 */

/*
 *								--- GL_ARB_texture_buffer_range (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_texture_buffer_range = false;

#ifdef __GENERATE_GL_ARB_texture_buffer_range

void (GLApi *GLFuncPtrName(glTexBufferRange))(GLenum, GLenum, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glTextureBufferRangeEXT))(GLuint, GLenum, GLenum, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)

#endif // __GENERATE_GL_ARB_texture_buffer_range

int CGL4ExtensionHandler::n_GetARBTextureBufferRangeFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_buffer_range

	if(!(GLFuncPtrName(glTexBufferRange) = (void(GLApi*)(GLenum, GLenum, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glTexBufferRange"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTextureBufferRangeEXT) = (void(GLApi*)(GLuint, GLenum, GLenum, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glTextureBufferRangeEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_buffer_range

	GLEH_ARB_texture_buffer_range = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_buffer_range");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_buffer_range ---
 */

/*
 *								--- GL_ARB_texture_query_levels (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_texture_query_levels = false;

int CGL4ExtensionHandler::n_GetARBTextureQueryLevelsFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_query_levels = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_query_levels");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_query_levels ---
 */

/*
 *								--- GL_ARB_texture_storage_multisample (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_texture_storage_multisample = false;

#ifdef __GENERATE_GL_ARB_texture_storage_multisample

void (GLApi *GLFuncPtrName(glTexStorage2DMultisample))(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexStorage3DMultisample))(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glTextureStorage2DMultisampleEXT))(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glTextureStorage3DMultisampleEXT))(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)

#endif // __GENERATE_GL_ARB_texture_storage_multisample

int CGL4ExtensionHandler::n_GetARBTextureStorageMultisampleFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_storage_multisample

	if(!(GLFuncPtrName(glTexStorage2DMultisample) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexStorage2DMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTexStorage3DMultisample) = (void(GLApi*)(GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexStorage3DMultisample"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTextureStorage2DMultisampleEXT) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTextureStorage2DMultisampleEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glTextureStorage3DMultisampleEXT) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTextureStorage3DMultisampleEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_storage_multisample

	GLEH_ARB_texture_storage_multisample = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_storage_multisample");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_storage_multisample ---
 */

/*
 *								--- GL_ARB_texture_view (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_texture_view = false;

#ifdef __GENERATE_GL_ARB_texture_view

void (GLApi *GLFuncPtrName(glTextureView))(GLuint, GLenum, GLuint, GLenum, GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_texture_view

int CGL4ExtensionHandler::n_GetARBTextureViewFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_view

	if(!(GLFuncPtrName(glTextureView) = (void(GLApi*)(GLuint, GLenum, GLuint, GLenum, GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glTextureView"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_view

	GLEH_ARB_texture_view = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_view");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_view ---
 */

/*
 *								--- GL_ARB_vertex_attrib_binding (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_vertex_attrib_binding = false;

#ifdef __GENERATE_GL_ARB_vertex_attrib_binding

void (GLApi *GLFuncPtrName(glBindVertexBuffer))(GLuint, GLuint, GLintptr, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribFormat))(GLuint, int, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribIFormat))(GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribLFormat))(GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribBinding))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexBindingDivisor))(GLuint, GLuint) INIT_FUNC_POINTERS;
//void (GLApi *GLFuncPtrName(glVertexArrayBindVertexBufferEXT))(GLuint, GLuint, GLuint, GLintptr, GLsizei) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glVertexArrayVertexAttribFormatEXT))(GLuint, GLuint, int, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glVertexArrayVertexAttribIFormatEXT))(GLuint, GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glVertexArrayVertexAttribLFormatEXT))(GLuint, GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glVertexArrayVertexAttribBindingEXT))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)
//void (GLApi *GLFuncPtrName(glVertexArrayVertexBindingDivisorEXT))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS; // deprecated as of OpenGL 4.5 (in favor of ARB_direct_state_access)

#endif // __GENERATE_GL_ARB_vertex_attrib_binding

int CGL4ExtensionHandler::n_GetARBVertexAttribBindingFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_vertex_attrib_binding

	if(!(GLFuncPtrName(glBindVertexBuffer) = (void(GLApi*)(GLuint, GLuint, GLintptr, GLsizei))wglGetProcAddress("glBindVertexBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribFormat) = (void(GLApi*)(GLuint, int, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexAttribFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribIFormat) = (void(GLApi*)(GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexAttribIFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribLFormat) = (void(GLApi*)(GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexAttribLFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribBinding) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glVertexAttribBinding"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexBindingDivisor) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glVertexBindingDivisor"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayBindVertexBufferEXT) = (void(GLApi*)(GLuint, GLuint, GLuint, GLintptr, GLsizei))wglGetProcAddress("glVertexArrayBindVertexBufferEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexAttribFormatEXT) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexArrayVertexAttribFormatEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexAttribIFormatEXT) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexArrayVertexAttribIFormatEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexAttribLFormatEXT) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexArrayVertexAttribLFormatEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexAttribBindingEXT) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glVertexArrayVertexAttribBindingEXT"))) ++ n_failed_functions;
	//if(!(GLFuncPtrName(glVertexArrayVertexBindingDivisorEXT) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glVertexArrayVertexBindingDivisorEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_vertex_attrib_binding

	GLEH_ARB_vertex_attrib_binding = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_attrib_binding");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_attrib_binding ---
 */

/*
 *								--- GL_KHR_texture_compression_astc_hdr (part of OpenGL 4.3 core) ---
 */

bool GLEH_KHR_texture_compression_astc_hdr = false;

int CGL4ExtensionHandler::n_GetKHRTextureCompressionASTC_HDRFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_KHR_texture_compression_astc_hdr = !n_failed_functions && b_SupportedExtension("GL_KHR_texture_compression_astc_hdr");

	return n_failed_functions;
}

/*
 *								--- ~GL_KHR_texture_compression_astc_hdr ---
 */

/*
 *								--- GL_ARB_robustness_isolation (part of OpenGL 4.3 core) ---
 */

bool GLEH_ARB_robustness_isolation = false;

int CGL4ExtensionHandler::n_GetARBRobustnessIsolationFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_robustness_isolation = !n_failed_functions && b_SupportedExtension("GL_ARB_robustness_isolation");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_robustness_isolation ---
 */

/*
 *								--- GL_ARB_buffer_storage (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_buffer_storage = false;

#ifdef __GENERATE_GL_ARB_buffer_storage

void (GLApi *GLFuncPtrName(glBufferStorage))(GLenum, GLsizeiptr, const void*, GLbitfield) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedBufferStorageEXT))(GLuint, GLsizeiptr, const void*, GLbitfield) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_buffer_storage

int CGL4ExtensionHandler::n_GetARBBufferStorageFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_buffer_storage

	if(!(GLFuncPtrName(glBufferStorage) = (void(GLApi*)(GLenum, GLsizeiptr, const void*, GLbitfield))wglGetProcAddress("glBufferStorage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedBufferStorageEXT) = (void(GLApi*)(GLuint, GLsizeiptr, const void*, GLbitfield))wglGetProcAddress("glNamedBufferStorageEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_buffer_storage

	GLEH_ARB_buffer_storage = !n_failed_functions && b_SupportedExtension("GL_ARB_buffer_storage");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_buffer_storage ---
 */

/*
 *								--- GL_ARB_clear_texture (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_clear_texture = false;

#ifdef __GENERATE_GL_ARB_clear_texture

void (GLApi *GLFuncPtrName(glClearTexImage))(GLuint, int, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearTexSubImage))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_clear_texture

int CGL4ExtensionHandler::n_GetARBClearTextureFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_clear_texture

	if(!(GLFuncPtrName(glClearTexImage) = (void(GLApi*)(GLuint, int, GLenum, GLenum, const void*))wglGetProcAddress("glClearTexImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearTexSubImage) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const void*))wglGetProcAddress("glClearTexSubImage"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_clear_texture

	GLEH_ARB_clear_texture = !n_failed_functions && b_SupportedExtension("GL_ARB_clear_texture");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_clear_texture ---
 */

/*
 *								--- GL_ARB_enhanced_layouts (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_enhanced_layouts = false;

int CGL4ExtensionHandler::n_GetARBEnhancedLayoutsFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_enhanced_layouts = !n_failed_functions && b_SupportedExtension("GL_ARB_enhanced_layouts");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_enhanced_layouts ---
 */

/*
 *								--- GL_ARB_multi_bind (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_multi_bind = false;

#ifdef __GENERATE_GL_ARB_multi_bind

void (GLApi *GLFuncPtrName(glBindBuffersBase))(GLenum, GLuint, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindBuffersRange))(GLenum, GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizeiptr*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindTextures))(GLuint, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindSamplers))(GLuint, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindImageTextures))(GLuint, GLsizei, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindVertexBuffers))(GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizei*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_multi_bind

int CGL4ExtensionHandler::n_GetARBMultiBindFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_multi_bind

	if(!(GLFuncPtrName(glBindBuffersBase) = (void(GLApi*)(GLenum, GLuint, GLsizei, const GLuint*))wglGetProcAddress("glBindBuffersBase"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindBuffersRange) = (void(GLApi*)(GLenum, GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizeiptr*))wglGetProcAddress("glBindBuffersRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindTextures) = (void(GLApi*)(GLuint, GLsizei, const GLuint*))wglGetProcAddress("glBindTextures"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindSamplers) = (void(GLApi*)(GLuint, GLsizei, const GLuint*))wglGetProcAddress("glBindSamplers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindImageTextures) = (void(GLApi*)(GLuint, GLsizei, const GLuint*))wglGetProcAddress("glBindImageTextures"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindVertexBuffers) = (void(GLApi*)(GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizei*))wglGetProcAddress("glBindVertexBuffers"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_multi_bind

	GLEH_ARB_multi_bind = !n_failed_functions && b_SupportedExtension("GL_ARB_multi_bind");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_multi_bind ---
 */

/*
 *								--- GL_ARB_query_buffer_object (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_query_buffer_object = false;

int CGL4ExtensionHandler::n_GetARBQueryBufferObjectFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_query_buffer_object = !n_failed_functions && b_SupportedExtension("GL_ARB_query_buffer_object");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_query_buffer_object ---
 */

/*
 *								--- GL_ARB_texture_mirror_clamp_to_edge (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_texture_mirror_clamp_to_edge = false;

int CGL4ExtensionHandler::n_GetARBTextureMirrorClampToEdgeFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_mirror_clamp_to_edge = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_mirror_clamp_to_edge");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_mirror_clamp_to_edge ---
 */

/*
 *								--- GL_ARB_texture_stencil8 (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_texture_stencil8 = false;

int CGL4ExtensionHandler::n_GetARBTextureStencil8FuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_texture_stencil8 = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_stencil8");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_stencil8 ---
 */

/*
 *								--- GL_ARB_vertex_type_10f_11f_11f_rev (part of OpenGL 4.4 core) ---
 */

bool GLEH_ARB_vertex_type_10f_11f_11f_rev = false;

int CGL4ExtensionHandler::n_GetARBVertexType10f11f11fRevFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_vertex_type_10f_11f_11f_rev = !n_failed_functions && b_SupportedExtension("GL_ARB_vertex_type_10f_11f_11f_rev");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_vertex_type_10f_11f_11f_rev ---
 */

/*
 *								--- GL_ARB_clip_control (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_clip_control = false;

#ifdef __GENERATE_GL_ARB_clip_control

void (GLApi *GLFuncPtrName(glClipControl))(GLenum, GLenum) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_clip_control

int CGL4ExtensionHandler::n_GetARBClipControlFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_clip_control

	if(!(GLFuncPtrName(glClipControl) = (void(GLApi*)(GLenum, GLenum))wglGetProcAddress("glClipControl"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_clip_control

	GLEH_ARB_clip_control = !n_failed_functions && b_SupportedExtension("GL_ARB_clip_control");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_clip_control ---
 */

/*
 *								--- GL_ARB_cull_distance (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_cull_distance = false;

int CGL4ExtensionHandler::n_GetARBCullDistanceFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_cull_distance = !n_failed_functions && b_SupportedExtension("GL_ARB_cull_distance");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_cull_distance ---
 */

/*
 *								--- GL_ARB_ES3_1_compatibility (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_ES3_1_compatibility = false;

#ifdef __GENERATE_GL_ARB_ES3_1_compatibility

void (GLApi *GLFuncPtrName(glMemoryBarrierByRegion))(GLbitfield) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_ES3_1_compatibility

int CGL4ExtensionHandler::n_GetARBES31CompatibilityFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_ES3_1_compatibility

	if(!(GLFuncPtrName(glMemoryBarrierByRegion) = (void(GLApi*)(GLbitfield))wglGetProcAddress("glMemoryBarrierByRegion"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_ES3_1_compatibility

	GLEH_ARB_ES3_1_compatibility = !n_failed_functions && b_SupportedExtension("GL_ARB_ES3_1_compatibility");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_ES3_1_compatibility ---
 */

/*
 *								--- GL_ARB_conditional_render_inverted (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_conditional_render_inverted = false;

int CGL4ExtensionHandler::n_GetARBConditionalRenderInvertedFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_conditional_render_inverted = !n_failed_functions && b_SupportedExtension("GL_ARB_conditional_render_inverted");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_conditional_render_inverted ---
 */

/*
 *								--- GL_KHR_context_flush_control (part of OpenGL 4.5 core) ---
 */

bool GLEH_KHR_context_flush_control = false;

int CGL4ExtensionHandler::n_GetKHRContextFlushControlFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_KHR_context_flush_control = !n_failed_functions && b_SupportedExtension("GL_KHR_context_flush_control");

	return n_failed_functions;
}

/*
 *								--- ~GL_KHR_context_flush_control ---
 */

/*
 *								--- GL_ARB_derivative_control (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_derivative_control = false;

int CGL4ExtensionHandler::n_GetARBDerivativeControlFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_derivative_control = !n_failed_functions && b_SupportedExtension("GL_ARB_derivative_control");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_derivative_control ---
 */

/*
 *								--- GL_ARB_direct_state_access (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_direct_state_access = false;

#ifdef __GENERATE_GL_ARB_direct_state_access

void (GLApi *GLFuncPtrName(glCreateTransformFeedbacks))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTransformFeedbackBufferBase))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTransformFeedbackBufferRange))(GLuint, GLuint, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTransformFeedbackiv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTransformFeedbacki_v))(GLuint, GLenum, GLuint, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTransformFeedbacki64_v))(GLuint, GLenum, GLuint, GLint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateBuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedBufferStorage))(GLuint, GLsizeiptr, const void*, GLbitfield) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedBufferData))(GLuint, GLsizeiptr, const void*, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedBufferSubData))(GLuint, GLintptr, GLsizeiptr, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyNamedBufferSubData))(GLuint, GLuint, GLintptr, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedBufferData))(GLuint, GLenum, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedBufferSubData))(GLuint, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void* (GLApi *GLFuncPtrName(glMapNamedBuffer))(GLuint, GLenum) INIT_FUNC_POINTERS;
void* (GLApi *GLFuncPtrName(glMapNamedBufferRange))(GLuint, GLintptr, GLsizeiptr, GLbitfield) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glUnmapNamedBuffer))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glFlushMappedNamedBufferRange))(GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedBufferParameteriv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedBufferParameteri64v))(GLuint, GLenum, GLint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedBufferPointerv))(GLuint, GLenum, void**) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedBufferSubData))(GLuint, GLintptr, GLsizeiptr, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateFramebuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferRenderbuffer))(GLuint, GLenum, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferParameteri))(GLuint, GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferTexture))(GLuint, GLenum, GLuint, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferTextureLayer))(GLuint, GLenum, GLuint, int, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferDrawBuffer))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferDrawBuffers))(GLuint, GLsizei, const GLenum*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedFramebufferReadBuffer))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateNamedFramebufferData))(GLuint, GLsizei, const GLenum*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glInvalidateNamedFramebufferSubData))(GLuint, GLsizei, const GLenum*, int, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedFramebufferiv))(GLuint, GLenum, int, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedFramebufferuiv))(GLuint, GLenum, int, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedFramebufferfv))(GLuint, GLenum, int, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glClearNamedFramebufferfi))(GLuint, GLenum, float, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBlitNamedFramebuffer))(GLuint, GLuint, int, int, int, int, int, int, int, int, GLbitfield, GLenum) INIT_FUNC_POINTERS;
GLenum (GLApi *GLFuncPtrName(glCheckNamedFramebufferStatus))(GLuint, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedFramebufferParameteriv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedFramebufferAttachmentParameteriv))(GLuint, GLenum, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateRenderbuffers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedRenderbufferStorage))(GLuint, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glNamedRenderbufferStorageMultisample))(GLuint, GLsizei, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedRenderbufferParameteriv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateTextures))(GLenum, GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureBuffer))(GLuint, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureBufferRange))(GLuint, GLenum, GLuint, GLintptr, GLsizeiptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage1D))(GLuint, GLsizei, GLenum, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage2D))(GLuint, GLsizei, GLenum, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage3D))(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage2DMultisample))(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureStorage3DMultisample))(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureSubImage1D))(GLuint, int, int, GLsizei, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureSubImage2D))(GLuint, int, int, int, GLsizei, GLsizei, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureSubImage3D))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTextureSubImage1D))(GLuint, int, int, GLsizei, GLenum, GLsizei, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTextureSubImage2D))(GLuint, int, int, int, GLsizei, GLsizei, GLenum, GLsizei, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompressedTextureSubImage3D))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLsizei, const void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyTextureSubImage1D))(GLuint, int, int, int, int, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyTextureSubImage2D))(GLuint, int, int, int, int, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCopyTextureSubImage3D))(GLuint, int, int, int, int, int, int, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameterf))(GLuint, GLenum, float) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameterfv))(GLuint, GLenum, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameteri))(GLuint, GLenum, int) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameterIiv))(GLuint, GLenum, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameterIuiv))(GLuint, GLenum, const GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTextureParameteriv))(GLuint, GLenum, const int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGenerateTextureMipmap))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glBindTextureUnit))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureImage))(GLuint, int, GLenum, GLenum, GLsizei, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetCompressedTextureImage))(GLuint, int, GLsizei, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureLevelParameterfv))(GLuint, int, GLenum, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureLevelParameteriv))(GLuint, int, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureParameterfv))(GLuint, GLenum, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureParameterIiv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureParameterIuiv))(GLuint, GLenum, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetTextureParameteriv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateVertexArrays))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDisableVertexArrayAttrib))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEnableVertexArrayAttrib))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayElementBuffer))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayVertexBuffer))(GLuint, GLuint, GLuint, GLintptr, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayVertexBuffers))(GLuint, GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizei*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayAttribFormat))(GLuint, GLuint, int, GLenum, GLboolean, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayAttribIFormat))(GLuint, GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayAttribLFormat))(GLuint, GLuint, int, GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayAttribBinding))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexArrayBindingDivisor))(GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexArrayiv))(GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexArrayIndexediv))(GLuint, GLuint, GLenum, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexArrayIndexed64iv))(GLuint, GLuint, GLenum, GLint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateSamplers))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateProgramPipelines))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateQueries))(GLenum, GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryBufferObjectiv))(GLuint, GLuint, GLenum, GLintptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryBufferObjectuiv))(GLuint, GLuint, GLenum, GLintptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryBufferObjecti64v))(GLuint, GLuint, GLenum, GLintptr) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetQueryBufferObjectui64v))(GLuint, GLuint, GLenum, GLintptr) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_direct_state_access

int CGL4ExtensionHandler::n_GetARBDirectStateAccessFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_direct_state_access

	if(!(GLFuncPtrName(glCreateTransformFeedbacks) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateTransformFeedbacks"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTransformFeedbackBufferBase) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glTransformFeedbackBufferBase"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTransformFeedbackBufferRange) = (void(GLApi*)(GLuint, GLuint, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glTransformFeedbackBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTransformFeedbackiv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetTransformFeedbackiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTransformFeedbacki_v) = (void(GLApi*)(GLuint, GLenum, GLuint, int*))wglGetProcAddress("glGetTransformFeedbacki_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTransformFeedbacki64_v) = (void(GLApi*)(GLuint, GLenum, GLuint, GLint64*))wglGetProcAddress("glGetTransformFeedbacki64_v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateBuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateBuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedBufferStorage) = (void(GLApi*)(GLuint, GLsizeiptr, const void*, GLbitfield))wglGetProcAddress("glNamedBufferStorage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedBufferData) = (void(GLApi*)(GLuint, GLsizeiptr, const void*, GLenum))wglGetProcAddress("glNamedBufferData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedBufferSubData) = (void(GLApi*)(GLuint, GLintptr, GLsizeiptr, const void*))wglGetProcAddress("glNamedBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCopyNamedBufferSubData) = (void(GLApi*)(GLuint, GLuint, GLintptr, GLintptr, GLsizeiptr))wglGetProcAddress("glCopyNamedBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedBufferData) = (void(GLApi*)(GLuint, GLenum, GLenum, GLenum, const void*))wglGetProcAddress("glClearNamedBufferData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedBufferSubData) = (void(GLApi*)(GLuint, GLenum, GLintptr, GLsizeiptr, GLenum, GLenum, const void*))wglGetProcAddress("glClearNamedBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMapNamedBuffer) = (void*(GLApi*)(GLuint, GLenum))wglGetProcAddress("glMapNamedBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMapNamedBufferRange) = (void*(GLApi*)(GLuint, GLintptr, GLsizeiptr, GLbitfield))wglGetProcAddress("glMapNamedBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUnmapNamedBuffer) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glUnmapNamedBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glFlushMappedNamedBufferRange) = (void(GLApi*)(GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glFlushMappedNamedBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedBufferParameteriv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetNamedBufferParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedBufferParameteri64v) = (void(GLApi*)(GLuint, GLenum, GLint64*))wglGetProcAddress("glGetNamedBufferParameteri64v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedBufferPointerv) = (void(GLApi*)(GLuint, GLenum, void**))wglGetProcAddress("glGetNamedBufferPointerv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedBufferSubData) = (void(GLApi*)(GLuint, GLintptr, GLsizeiptr, void*))wglGetProcAddress("glGetNamedBufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateFramebuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateFramebuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferRenderbuffer) = (void(GLApi*)(GLuint, GLenum, GLenum, GLuint))wglGetProcAddress("glNamedFramebufferRenderbuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferParameteri) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glNamedFramebufferParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferTexture) = (void(GLApi*)(GLuint, GLenum, GLuint, int))wglGetProcAddress("glNamedFramebufferTexture"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferTextureLayer) = (void(GLApi*)(GLuint, GLenum, GLuint, int, int))wglGetProcAddress("glNamedFramebufferTextureLayer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferDrawBuffer) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glNamedFramebufferDrawBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferDrawBuffers) = (void(GLApi*)(GLuint, GLsizei, const GLenum*))wglGetProcAddress("glNamedFramebufferDrawBuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedFramebufferReadBuffer) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glNamedFramebufferReadBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateNamedFramebufferData) = (void(GLApi*)(GLuint, GLsizei, const GLenum*))wglGetProcAddress("glInvalidateNamedFramebufferData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glInvalidateNamedFramebufferSubData) = (void(GLApi*)(GLuint, GLsizei, const GLenum*, int, int, GLsizei, GLsizei))wglGetProcAddress("glInvalidateNamedFramebufferSubData"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedFramebufferiv) = (void(GLApi*)(GLuint, GLenum, int, const int*))wglGetProcAddress("glClearNamedFramebufferiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedFramebufferuiv) = (void(GLApi*)(GLuint, GLenum, int, const GLuint*))wglGetProcAddress("glClearNamedFramebufferuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedFramebufferfv) = (void(GLApi*)(GLuint, GLenum, int, const float*))wglGetProcAddress("glClearNamedFramebufferfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glClearNamedFramebufferfi) = (void(GLApi*)(GLuint, GLenum, float, int))wglGetProcAddress("glClearNamedFramebufferfi"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBlitNamedFramebuffer) = (void(GLApi*)(GLuint, GLuint, int, int, int, int, int, int, int, int, GLbitfield, GLenum))wglGetProcAddress("glBlitNamedFramebuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCheckNamedFramebufferStatus) = (GLenum(GLApi*)(GLuint, GLenum))wglGetProcAddress("glCheckNamedFramebufferStatus"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedFramebufferParameteriv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetNamedFramebufferParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedFramebufferAttachmentParameteriv) = (void(GLApi*)(GLuint, GLenum, GLenum, int*))wglGetProcAddress("glGetNamedFramebufferAttachmentParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateRenderbuffers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateRenderbuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedRenderbufferStorage) = (void(GLApi*)(GLuint, GLenum, GLsizei, GLsizei))wglGetProcAddress("glNamedRenderbufferStorage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glNamedRenderbufferStorageMultisample) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei, GLsizei))wglGetProcAddress("glNamedRenderbufferStorageMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedRenderbufferParameteriv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetNamedRenderbufferParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateTextures) = (void(GLApi*)(GLenum, GLsizei, GLuint*))wglGetProcAddress("glCreateTextures"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureBuffer) = (void(GLApi*)(GLuint, GLenum, GLuint))wglGetProcAddress("glTextureBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureBufferRange) = (void(GLApi*)(GLuint, GLenum, GLuint, GLintptr, GLsizeiptr))wglGetProcAddress("glTextureBufferRange"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage1D) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei))wglGetProcAddress("glTextureStorage1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage2D) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei, GLsizei))wglGetProcAddress("glTextureStorage2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage3D) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLsizei))wglGetProcAddress("glTextureStorage3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage2DMultisample) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTextureStorage2DMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureStorage3DMultisample) = (void(GLApi*)(GLuint, GLsizei, GLenum, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTextureStorage3DMultisample"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureSubImage1D) = (void(GLApi*)(GLuint, int, int, GLsizei, GLenum, GLenum, const void*))wglGetProcAddress("glTextureSubImage1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureSubImage2D) = (void(GLApi*)(GLuint, int, int, int, GLsizei, GLsizei, GLenum, GLenum, const void*))wglGetProcAddress("glTextureSubImage2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureSubImage3D) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, const void*))wglGetProcAddress("glTextureSubImage3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCompressedTextureSubImage1D) = (void(GLApi*)(GLuint, int, int, GLsizei, GLenum, GLsizei, const void*))wglGetProcAddress("glCompressedTextureSubImage1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCompressedTextureSubImage2D) = (void(GLApi*)(GLuint, int, int, int, GLsizei, GLsizei, GLenum, GLsizei, const void*))wglGetProcAddress("glCompressedTextureSubImage2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCompressedTextureSubImage3D) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLsizei, const void*))wglGetProcAddress("glCompressedTextureSubImage3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCopyTextureSubImage1D) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei))wglGetProcAddress("glCopyTextureSubImage1D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCopyTextureSubImage2D) = (void(GLApi*)(GLuint, int, int, int, int, int, GLsizei, GLsizei))wglGetProcAddress("glCopyTextureSubImage2D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCopyTextureSubImage3D) = (void(GLApi*)(GLuint, int, int, int, int, int, int, GLsizei, GLsizei))wglGetProcAddress("glCopyTextureSubImage3D"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameterf) = (void(GLApi*)(GLuint, GLenum, float))wglGetProcAddress("glTextureParameterf"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameterfv) = (void(GLApi*)(GLuint, GLenum, const float*))wglGetProcAddress("glTextureParameterfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameteri) = (void(GLApi*)(GLuint, GLenum, int))wglGetProcAddress("glTextureParameteri"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameterIiv) = (void(GLApi*)(GLuint, GLenum, const int*))wglGetProcAddress("glTextureParameterIiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameterIuiv) = (void(GLApi*)(GLuint, GLenum, const GLuint*))wglGetProcAddress("glTextureParameterIuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTextureParameteriv) = (void(GLApi*)(GLuint, GLenum, const int*))wglGetProcAddress("glTextureParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGenerateTextureMipmap) = (void(GLApi*)(GLuint))wglGetProcAddress("glGenerateTextureMipmap"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glBindTextureUnit) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glBindTextureUnit"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureImage) = (void(GLApi*)(GLuint, int, GLenum, GLenum, GLsizei, void*))wglGetProcAddress("glGetTextureImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetCompressedTextureImage) = (void(GLApi*)(GLuint, int, GLsizei, void*))wglGetProcAddress("glGetCompressedTextureImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureLevelParameterfv) = (void(GLApi*)(GLuint, int, GLenum, float*))wglGetProcAddress("glGetTextureLevelParameterfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureLevelParameteriv) = (void(GLApi*)(GLuint, int, GLenum, int*))wglGetProcAddress("glGetTextureLevelParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureParameterfv) = (void(GLApi*)(GLuint, GLenum, float*))wglGetProcAddress("glGetTextureParameterfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureParameterIiv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetTextureParameterIiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureParameterIuiv) = (void(GLApi*)(GLuint, GLenum, GLuint*))wglGetProcAddress("glGetTextureParameterIuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureParameteriv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetTextureParameteriv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateVertexArrays) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateVertexArrays"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDisableVertexArrayAttrib) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glDisableVertexArrayAttrib"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glEnableVertexArrayAttrib) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glEnableVertexArrayAttrib"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayElementBuffer) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glVertexArrayElementBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayVertexBuffer) = (void(GLApi*)(GLuint, GLuint, GLuint, GLintptr, GLsizei))wglGetProcAddress("glVertexArrayVertexBuffer"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayVertexBuffers) = (void(GLApi*)(GLuint, GLuint, GLsizei, const GLuint*, const GLintptr*, const GLsizei*))wglGetProcAddress("glVertexArrayVertexBuffers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayAttribFormat) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLboolean, GLuint))wglGetProcAddress("glVertexArrayAttribFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayAttribIFormat) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexArrayAttribIFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayAttribLFormat) = (void(GLApi*)(GLuint, GLuint, int, GLenum, GLuint))wglGetProcAddress("glVertexArrayAttribLFormat"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayAttribBinding) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glVertexArrayAttribBinding"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexArrayBindingDivisor) = (void(GLApi*)(GLuint, GLuint, GLuint))wglGetProcAddress("glVertexArrayBindingDivisor"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetVertexArrayiv) = (void(GLApi*)(GLuint, GLenum, int*))wglGetProcAddress("glGetVertexArrayiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetVertexArrayIndexediv) = (void(GLApi*)(GLuint, GLuint, GLenum, int*))wglGetProcAddress("glGetVertexArrayIndexediv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetVertexArrayIndexed64iv) = (void(GLApi*)(GLuint, GLuint, GLenum, GLint64*))wglGetProcAddress("glGetVertexArrayIndexed64iv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateSamplers) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateSamplers"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateProgramPipelines) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateProgramPipelines"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateQueries) = (void(GLApi*)(GLenum, GLsizei, GLuint*))wglGetProcAddress("glCreateQueries"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryBufferObjectiv) = (void(GLApi*)(GLuint, GLuint, GLenum, GLintptr))wglGetProcAddress("glGetQueryBufferObjectiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryBufferObjectuiv) = (void(GLApi*)(GLuint, GLuint, GLenum, GLintptr))wglGetProcAddress("glGetQueryBufferObjectuiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryBufferObjecti64v) = (void(GLApi*)(GLuint, GLuint, GLenum, GLintptr))wglGetProcAddress("glGetQueryBufferObjecti64v"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetQueryBufferObjectui64v) = (void(GLApi*)(GLuint, GLuint, GLenum, GLintptr))wglGetProcAddress("glGetQueryBufferObjectui64v"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_direct_state_access

	GLEH_ARB_direct_state_access = !n_failed_functions && b_SupportedExtension("GL_ARB_direct_state_access");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_direct_state_access ---
 */

/*
 *								--- GL_ARB_get_texture_sub_image (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_get_texture_sub_image = false;

#ifdef __GENERATE_GL_ARB_get_texture_sub_image

void (GLApi *GLFuncPtrName(glGetTextureSubImage))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, GLsizei, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetCompressedTextureSubImage))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLsizei, void*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_get_texture_sub_image

int CGL4ExtensionHandler::n_GetARBGetTextureSubImageFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_get_texture_sub_image

	if(!(GLFuncPtrName(glGetTextureSubImage) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLenum, GLenum, GLsizei, void*))wglGetProcAddress("glGetTextureSubImage"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetCompressedTextureSubImage) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLsizei, void*))wglGetProcAddress("glGetCompressedTextureSubImage"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_get_texture_sub_image

	GLEH_ARB_get_texture_sub_image = !n_failed_functions && b_SupportedExtension("GL_ARB_get_texture_sub_image");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_get_texture_sub_image ---
 */

/*
 *								--- GL_KHR_robustness (part of OpenGL 4.5 core) ---
 */

bool GLEH_KHR_robustness = false;

#ifdef __GENERATE_GL_KHR_robustness

GLenum (GLApi *GLFuncPtrName(glGetGraphicsResetStatus))() INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glReadnPixels))(int, int, GLsizei, GLsizei, GLenum, GLenum, GLsizei, void*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetnUniformfv))(GLuint, int, GLsizei, float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetnUniformiv))(GLuint, int, GLsizei, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetnUniformuiv))(GLuint, int, GLsizei, GLuint*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_KHR_robustness

int CGL4ExtensionHandler::n_GetKHRRobustnessFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_KHR_robustness

	if(!(GLFuncPtrName(glGetGraphicsResetStatus) = (GLenum(GLApi*)())wglGetProcAddress("glGetGraphicsResetStatus"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glReadnPixels) = (void(GLApi*)(int, int, GLsizei, GLsizei, GLenum, GLenum, GLsizei, void*))wglGetProcAddress("glReadnPixels"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetnUniformfv) = (void(GLApi*)(GLuint, int, GLsizei, float*))wglGetProcAddress("glGetnUniformfv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetnUniformiv) = (void(GLApi*)(GLuint, int, GLsizei, int*))wglGetProcAddress("glGetnUniformiv"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetnUniformuiv) = (void(GLApi*)(GLuint, int, GLsizei, GLuint*))wglGetProcAddress("glGetnUniformuiv"))) ++ n_failed_functions;

#endif // __GENERATE_GL_KHR_robustness

	GLEH_KHR_robustness = !n_failed_functions && b_SupportedExtension("GL_KHR_robustness");

	return n_failed_functions;
}

/*
 *								--- ~GL_KHR_robustness ---
 */

/*
 *								--- GL_ARB_shader_texture_image_samples (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_shader_texture_image_samples = false;

int CGL4ExtensionHandler::n_GetARBShaderTextureImageSamplesFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_texture_image_samples = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_texture_image_samples");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_texture_image_samples ---
 */

/*
 *								--- GL_ARB_texture_barrier (part of OpenGL 4.5 core) ---
 */

bool GLEH_ARB_texture_barrier = false;

#ifdef __GENERATE_GL_ARB_texture_barrier

void (GLApi *GLFuncPtrName(glTextureBarrier))() INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_texture_barrier

int CGL4ExtensionHandler::n_GetARBTextureBarrierFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_texture_barrier

	if(!(GLFuncPtrName(glTextureBarrier) = (void(GLApi*)())wglGetProcAddress("glTextureBarrier"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_texture_barrier

	GLEH_ARB_texture_barrier = !n_failed_functions && b_SupportedExtension("GL_ARB_texture_barrier");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_texture_barrier ---
 */

/*
 *								--- GL_ARB_bindless_texture ---
 */

bool GLEH_ARB_bindless_texture = false;

#ifdef __GENERATE_GL_ARB_bindless_texture

GLuint64 (GLApi *GLFuncPtrName(glGetTextureHandleARB))(GLuint) INIT_FUNC_POINTERS;
GLuint64 (GLApi *GLFuncPtrName(glGetTextureSamplerHandleARB))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMakeTextureHandleResidentARB))(GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMakeTextureHandleNonResidentARB))(GLuint64) INIT_FUNC_POINTERS;
GLuint64 (GLApi *GLFuncPtrName(glGetImageHandleARB))(GLuint, int, GLboolean, int, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMakeImageHandleResidentARB))(GLuint64, GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMakeImageHandleNonResidentARB))(GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformHandleui64ARB))(int, GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glUniformHandleui64vARB))(int, GLsizei, const GLuint64*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformHandleui64ARB))(GLuint, int, GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramUniformHandleui64vARB))(GLuint, int, GLsizei, const GLuint64*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsTextureHandleResidentARB))(GLuint64) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsImageHandleResidentARB))(GLuint64) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL1ui64ARB))(GLuint, GLuint64EXT) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glVertexAttribL1ui64vARB))(GLuint, const GLuint64EXT*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetVertexAttribLui64vARB))(GLuint, GLenum, GLuint64EXT*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_bindless_texture

int CGL4ExtensionHandler::n_GetARBBindlessTextureFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_bindless_texture

	if(!(GLFuncPtrName(glGetTextureHandleARB) = (GLuint64(GLApi*)(GLuint))wglGetProcAddress("glGetTextureHandleARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetTextureSamplerHandleARB) = (GLuint64(GLApi*)(GLuint, GLuint))wglGetProcAddress("glGetTextureSamplerHandleARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMakeTextureHandleResidentARB) = (void(GLApi*)(GLuint64))wglGetProcAddress("glMakeTextureHandleResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMakeTextureHandleNonResidentARB) = (void(GLApi*)(GLuint64))wglGetProcAddress("glMakeTextureHandleNonResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetImageHandleARB) = (GLuint64(GLApi*)(GLuint, int, GLboolean, int, GLenum))wglGetProcAddress("glGetImageHandleARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMakeImageHandleResidentARB) = (void(GLApi*)(GLuint64, GLenum))wglGetProcAddress("glMakeImageHandleResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMakeImageHandleNonResidentARB) = (void(GLApi*)(GLuint64))wglGetProcAddress("glMakeImageHandleNonResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformHandleui64ARB) = (void(GLApi*)(int, GLuint64))wglGetProcAddress("glUniformHandleui64ARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glUniformHandleui64vARB) = (void(GLApi*)(int, GLsizei, const GLuint64*))wglGetProcAddress("glUniformHandleui64vARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformHandleui64ARB) = (void(GLApi*)(GLuint, int, GLuint64))wglGetProcAddress("glProgramUniformHandleui64ARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramUniformHandleui64vARB) = (void(GLApi*)(GLuint, int, GLsizei, const GLuint64*))wglGetProcAddress("glProgramUniformHandleui64vARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsTextureHandleResidentARB) = (GLboolean(GLApi*)(GLuint64))wglGetProcAddress("glIsTextureHandleResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsImageHandleResidentARB) = (GLboolean(GLApi*)(GLuint64))wglGetProcAddress("glIsImageHandleResidentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL1ui64ARB) = (void(GLApi*)(GLuint, GLuint64EXT))wglGetProcAddress("glVertexAttribL1ui64ARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glVertexAttribL1ui64vARB) = (void(GLApi*)(GLuint, const GLuint64EXT*))wglGetProcAddress("glVertexAttribL1ui64vARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetVertexAttribLui64vARB) = (void(GLApi*)(GLuint, GLenum, GLuint64EXT*))wglGetProcAddress("glGetVertexAttribLui64vARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_bindless_texture

	GLEH_ARB_bindless_texture = !n_failed_functions && b_SupportedExtension("GL_ARB_bindless_texture");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_bindless_texture ---
 */

/*
 *								--- GL_ARB_sparse_texture ---
 */

bool GLEH_ARB_sparse_texture = false;

#ifdef __GENERATE_GL_ARB_sparse_texture

void (GLApi *GLFuncPtrName(glTexPageCommitmentARB))(GLenum, int, int, int, int, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glTexturePageCommitmentEXTARB))(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLboolean) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_sparse_texture

int CGL4ExtensionHandler::n_GetARBSparseTextureFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_sparse_texture

	if(!(GLFuncPtrName(glTexPageCommitmentARB) = (void(GLApi*)(GLenum, int, int, int, int, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexPageCommitmentARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glTexturePageCommitmentEXTARB) = (void(GLApi*)(GLuint, int, int, int, int, GLsizei, GLsizei, GLsizei, GLboolean))wglGetProcAddress("glTexturePageCommitmentEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_sparse_texture

	GLEH_ARB_sparse_texture = !n_failed_functions && b_SupportedExtension("GL_ARB_sparse_texture");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_sparse_texture ---
 */

/*
 *								--- GL_ARB_compute_variable_group_size ---
 */

bool GLEH_ARB_compute_variable_group_size = false;

#ifdef __GENERATE_GL_ARB_compute_variable_group_size

void (GLApi *GLFuncPtrName(glDispatchComputeGroupSizeARB))(GLuint, GLuint, GLuint, GLuint, GLuint, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_compute_variable_group_size

int CGL4ExtensionHandler::n_GetARBComputeVariableGroupSizeFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_compute_variable_group_size

	if(!(GLFuncPtrName(glDispatchComputeGroupSizeARB) = (void(GLApi*)(GLuint, GLuint, GLuint, GLuint, GLuint, GLuint))wglGetProcAddress("glDispatchComputeGroupSizeARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_compute_variable_group_size

	GLEH_ARB_compute_variable_group_size = !n_failed_functions && b_SupportedExtension("GL_ARB_compute_variable_group_size");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_compute_variable_group_size ---
 */

/*
 *								--- GL_ARB_indirect_parameters ---
 */

bool GLEH_ARB_indirect_parameters = false;

#ifdef __GENERATE_GL_ARB_indirect_parameters

void (GLApi *GLFuncPtrName(glMultiDrawArraysIndirectCountARB))(GLenum, const void*, GLintptr, GLsizei, GLsizei) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glMultiDrawElementsIndirectCountARB))(GLenum, GLenum, const void*, GLintptr, GLsizei, GLsizei) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_indirect_parameters

int CGL4ExtensionHandler::n_GetARBIndirectParametersFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_indirect_parameters

	if(!(GLFuncPtrName(glMultiDrawArraysIndirectCountARB) = (void(GLApi*)(GLenum, const void*, GLintptr, GLsizei, GLsizei))wglGetProcAddress("glMultiDrawArraysIndirectCountARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glMultiDrawElementsIndirectCountARB) = (void(GLApi*)(GLenum, GLenum, const void*, GLintptr, GLsizei, GLsizei))wglGetProcAddress("glMultiDrawElementsIndirectCountARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_indirect_parameters

	GLEH_ARB_indirect_parameters = !n_failed_functions && b_SupportedExtension("GL_ARB_indirect_parameters");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_indirect_parameters ---
 */

/*
 *								--- GL_ARB_seamless_cubemap_per_texture ---
 */

bool GLEH_ARB_seamless_cubemap_per_texture = false;

int CGL4ExtensionHandler::n_GetARBSeamlessCubemapPerTextureFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_seamless_cubemap_per_texture = !n_failed_functions && b_SupportedExtension("GL_ARB_seamless_cubemap_per_texture");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_seamless_cubemap_per_texture ---
 */

/*
 *								--- GL_ARB_shader_draw_parameters ---
 */

bool GLEH_ARB_shader_draw_parameters = false;

int CGL4ExtensionHandler::n_GetARBShaderDrawParametersFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_draw_parameters = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_draw_parameters");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_draw_parameters ---
 */

/*
 *								--- GL_ARB_shader_group_vote ---
 */

bool GLEH_ARB_shader_group_vote = false;

int CGL4ExtensionHandler::n_GetARBShaderGroupVoteFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_group_vote = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_group_vote");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_group_vote ---
 */

/*
 *								--- GL_EXT_draw_buffers2 ---
 */

bool GLEH_EXT_draw_buffers2 = false;

#ifdef __GENERATE_GL_EXT_draw_buffers2

void (GLApi *GLFuncPtrName(glColorMaskIndexedEXT))(GLuint, GLboolean, GLboolean, GLboolean, GLboolean) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetBooleanIndexedvEXT))(GLenum, GLuint, GLboolean*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetIntegerIndexedvEXT))(GLenum, GLuint, int*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glEnableIndexedEXT))(GLenum, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDisableIndexedEXT))(GLenum, GLuint) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsEnabledIndexedEXT))(GLenum, GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_EXT_draw_buffers2

int CGL4ExtensionHandler::n_GetEXTDrawBuffers2FuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_EXT_draw_buffers2

	if(!(GLFuncPtrName(glColorMaskIndexedEXT) = (void(GLApi*)(GLuint, GLboolean, GLboolean, GLboolean, GLboolean))wglGetProcAddress("glColorMaskIndexedEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetBooleanIndexedvEXT) = (void(GLApi*)(GLenum, GLuint, GLboolean*))wglGetProcAddress("glGetBooleanIndexedvEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetIntegerIndexedvEXT) = (void(GLApi*)(GLenum, GLuint, int*))wglGetProcAddress("glGetIntegerIndexedvEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glEnableIndexedEXT) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glEnableIndexedEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDisableIndexedEXT) = (void(GLApi*)(GLenum, GLuint))wglGetProcAddress("glDisableIndexedEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsEnabledIndexedEXT) = (GLboolean(GLApi*)(GLenum, GLuint))wglGetProcAddress("glIsEnabledIndexedEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_EXT_draw_buffers2

	GLEH_EXT_draw_buffers2 = !n_failed_functions && b_SupportedExtension("GL_EXT_draw_buffers2");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_draw_buffers2 ---
 */

/*
 *								--- GL_EXT_gpu_program_parameters ---
 */

bool GLEH_EXT_gpu_program_parameters = false;

#ifdef __GENERATE_GL_EXT_gpu_program_parameters

void (GLApi *GLFuncPtrName(glProgramEnvParameters4fvEXT))(GLenum, GLuint, GLsizei, const float*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glProgramLocalParameters4fvEXT))(GLenum, GLuint, GLsizei, const float*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_EXT_gpu_program_parameters

int CGL4ExtensionHandler::n_GetEXTGPUProgramParametersFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_EXT_gpu_program_parameters

	if(!(GLFuncPtrName(glProgramEnvParameters4fvEXT) = (void(GLApi*)(GLenum, GLuint, GLsizei, const float*))wglGetProcAddress("glProgramEnvParameters4fvEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glProgramLocalParameters4fvEXT) = (void(GLApi*)(GLenum, GLuint, GLsizei, const float*))wglGetProcAddress("glProgramLocalParameters4fvEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_EXT_gpu_program_parameters

	GLEH_EXT_gpu_program_parameters = !n_failed_functions && b_SupportedExtension("GL_EXT_gpu_program_parameters");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_gpu_program_parameters ---
 */

/*
 *								--- GL_EXT_bindable_uniform ---
 */

bool GLEH_EXT_bindable_uniform = false;

#ifdef __GENERATE_GL_EXT_bindable_uniform

void (GLApi *GLFuncPtrName(glUniformBufferEXT))(GLuint, int, GLuint) INIT_FUNC_POINTERS;
int (GLApi *GLFuncPtrName(glGetUniformBufferSizeEXT))(GLuint, int) INIT_FUNC_POINTERS;
GLintptr (GLApi *GLFuncPtrName(glGetUniformOffsetEXT))(GLuint, int) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_EXT_bindable_uniform

int CGL4ExtensionHandler::n_GetEXTBindableUniformFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_EXT_bindable_uniform

	if(!(GLFuncPtrName(glUniformBufferEXT) = (void(GLApi*)(GLuint, int, GLuint))wglGetProcAddress("glUniformBufferEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetUniformBufferSizeEXT) = (int(GLApi*)(GLuint, int))wglGetProcAddress("glGetUniformBufferSizeEXT"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetUniformOffsetEXT) = (GLintptr(GLApi*)(GLuint, int))wglGetProcAddress("glGetUniformOffsetEXT"))) ++ n_failed_functions;

#endif // __GENERATE_GL_EXT_bindable_uniform

	GLEH_EXT_bindable_uniform = !n_failed_functions && b_SupportedExtension("GL_EXT_bindable_uniform");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_bindable_uniform ---
 */

/*
 *								--- GL_ARB_shader_texture_lod ---
 */

bool GLEH_ARB_shader_texture_lod = false;

int CGL4ExtensionHandler::n_GetARBShaderTextureLodFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_texture_lod = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_texture_lod");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_texture_lod ---
 */

/*
 *								--- GL_S3_s3tc ---
 */

bool GLEH_S3_s3tc = false;

int CGL4ExtensionHandler::n_GetS3_S3TCFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_S3_s3tc = !n_failed_functions && b_SupportedExtension("GL_S3_s3tc");

	return n_failed_functions;
}

/*
 *								--- ~GL_S3_s3tc ---
 */

/*
 *								--- GL_EXT_texture_compression_latc ---
 */

bool GLEH_EXT_texture_compression_latc = false;

int CGL4ExtensionHandler::n_GetEXTTextureCompressionLATCFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_EXT_texture_compression_latc = !n_failed_functions && b_SupportedExtension("GL_EXT_texture_compression_latc");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_texture_compression_latc ---
 */

/*
 *								--- GL_EXT_texture_compression_s3tc ---
 */

bool GLEH_EXT_texture_compression_s3tc = false;

int CGL4ExtensionHandler::n_GetEXTTextureCompressionS3TCFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_EXT_texture_compression_s3tc = !n_failed_functions && b_SupportedExtension("GL_EXT_texture_compression_s3tc");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_texture_compression_s3tc ---
 */

/*
 *								--- GL_EXT_texture_filter_anisotropic ---
 */

bool GLEH_EXT_texture_filter_anisotropic = false;

int CGL4ExtensionHandler::n_GetEXTTextureFilterAnisotropicFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_EXT_texture_filter_anisotropic = !n_failed_functions && b_SupportedExtension("GL_EXT_texture_filter_anisotropic");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_texture_filter_anisotropic ---
 */

/*
 *								--- GL_EXT_abgr ---
 */

bool GLEH_EXT_abgr = false;

int CGL4ExtensionHandler::n_GetEXTABGRFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_EXT_abgr = !n_failed_functions && b_SupportedExtension("GL_EXT_abgr");

	return n_failed_functions;
}

/*
 *								--- ~GL_EXT_abgr ---
 */

/*
 *								--- GL_ARB_shading_language_include ---
 */

bool GLEH_ARB_shading_language_include = false;

#ifdef __GENERATE_GL_ARB_shading_language_include

void (GLApi *GLFuncPtrName(glNamedStringARB))(GLenum, int, const GLchar*, int, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteNamedStringARB))(int, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompileShaderIncludeARB))(GLuint, GLsizei, const GLchar**, const int*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsNamedStringARB))(int, const GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedStringARB))(int, const GLchar*, GLsizei, int*, GLchar*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glGetNamedStringivARB))(int, const GLchar*, GLenum, int*) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_shading_language_include

int CGL4ExtensionHandler::n_GetARBShadingLanguageIncludeFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_shading_language_include

	if(!(GLFuncPtrName(glNamedStringARB) = (void(GLApi*)(GLenum, int, const GLchar*, int, const GLchar*))wglGetProcAddress("glNamedStringARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteNamedStringARB) = (void(GLApi*)(int, const GLchar*))wglGetProcAddress("glDeleteNamedStringARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCompileShaderIncludeARB) = (void(GLApi*)(GLuint, GLsizei, const GLchar**, const int*))wglGetProcAddress("glCompileShaderIncludeARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsNamedStringARB) = (GLboolean(GLApi*)(int, const GLchar*))wglGetProcAddress("glIsNamedStringARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedStringARB) = (void(GLApi*)(int, const GLchar*, GLsizei, int*, GLchar*))wglGetProcAddress("glGetNamedStringARB"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetNamedStringivARB) = (void(GLApi*)(int, const GLchar*, GLenum, int*))wglGetProcAddress("glGetNamedStringivARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_shading_language_include

	GLEH_ARB_shading_language_include = !n_failed_functions && b_SupportedExtension("GL_ARB_shading_language_include");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shading_language_include ---
 */

/*
 *								--- GL_ARB_cl_event ---
 */

bool GLEH_ARB_cl_event = false;

#ifdef __GENERATE_GL_ARB_cl_event

GLsync (GLApi *GLFuncPtrName(glCreateSyncFromCLeventARB))(cl_context, cl_event, GLbitfield) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_ARB_cl_event

int CGL4ExtensionHandler::n_GetARBCLEventFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_ARB_cl_event

	if(!(GLFuncPtrName(glCreateSyncFromCLeventARB) = (GLsync(GLApi*)(cl_context, cl_event, GLbitfield))wglGetProcAddress("glCreateSyncFromCLeventARB"))) ++ n_failed_functions;

#endif // __GENERATE_GL_ARB_cl_event

	GLEH_ARB_cl_event = !n_failed_functions && b_SupportedExtension("GL_ARB_cl_event");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_cl_event ---
 */

/*
 *								--- GL_ARB_shader_stencil_export ---
 */

bool GLEH_ARB_shader_stencil_export = false;

int CGL4ExtensionHandler::n_GetARBShaderStencilExportFuncPointers()
{
	int n_failed_functions = 0;

	// no new functions or procedures here

	GLEH_ARB_shader_stencil_export = !n_failed_functions && b_SupportedExtension("GL_ARB_shader_stencil_export");

	return n_failed_functions;
}

/*
 *								--- ~GL_ARB_shader_stencil_export ---
 */

/*
 *								--- GL_NV_command_list ---
 */

bool GLEH_NV_command_list = false;

#ifdef __GENERATE_GL_NV_command_list

void (GLApi *GLFuncPtrName(glCreateStatesNV))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteStatesNV))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsStateNV))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glStateCaptureNV))(GLuint, GLenum) INIT_FUNC_POINTERS;
GLuint (GLApi *GLFuncPtrName(glGetCommandHeaderNV))(GLenum, GLuint) INIT_FUNC_POINTERS;
GLushort (GLApi *GLFuncPtrName(glGetStageIndexNV))(GLenum) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawCommandsNV))(GLenum, GLuint, const GLintptr*, const GLsizei*, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawCommandsAddressNV))(GLenum, const GLuint64*, const GLsizei*, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawCommandsStatesNV))(GLuint, const GLintptr*, const GLsizei*, const GLuint*, const GLuint*, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDrawCommandsStatesAddressNV))(const GLuint64*, const GLsizei*, const GLuint*, const GLuint*, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCreateCommandListsNV))(GLsizei, GLuint*) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glDeleteCommandListsNV))(GLsizei, const GLuint*) INIT_FUNC_POINTERS;
GLboolean (GLApi *GLFuncPtrName(glIsCommandListNV))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glListDrawCommandsStatesClientNV))(GLuint, GLuint, const void**, const size_t*, const GLuint*, const GLuint*, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCommandListSegmentsNV))(GLuint, GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCompileCommandListNV))(GLuint) INIT_FUNC_POINTERS;
void (GLApi *GLFuncPtrName(glCallCommandListNV))(GLuint) INIT_FUNC_POINTERS;

#endif // __GENERATE_GL_NV_command_list

int CGL4ExtensionHandler::n_GetNVCommandListFuncPointers()
{
	int n_failed_functions = 0;

#ifdef __GENERATE_GL_NV_command_list

	if(!(GLFuncPtrName(glCreateStatesNV) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateStatesNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteStatesNV) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteStatesNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsStateNV) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsStateNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glStateCaptureNV) = (void(GLApi*)(GLuint, GLenum))wglGetProcAddress("glStateCaptureNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetCommandHeaderNV) = (GLuint(GLApi*)(GLenum, GLuint))wglGetProcAddress("glGetCommandHeaderNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glGetStageIndexNV) = (GLushort(GLApi*)(GLenum))wglGetProcAddress("glGetStageIndexNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawCommandsNV) = (void(GLApi*)(GLenum, GLuint, const GLintptr*, const GLsizei*, GLuint))wglGetProcAddress("glDrawCommandsNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawCommandsAddressNV) = (void(GLApi*)(GLenum, const GLuint64*, const GLsizei*, GLuint))wglGetProcAddress("glDrawCommandsAddressNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawCommandsStatesNV) = (void(GLApi*)(GLuint, const GLintptr*, const GLsizei*, const GLuint*, const GLuint*, GLuint))wglGetProcAddress("glDrawCommandsStatesNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDrawCommandsStatesAddressNV) = (void(GLApi*)(const GLuint64*, const GLsizei*, const GLuint*, const GLuint*, GLuint))wglGetProcAddress("glDrawCommandsStatesAddressNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCreateCommandListsNV) = (void(GLApi*)(GLsizei, GLuint*))wglGetProcAddress("glCreateCommandListsNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glDeleteCommandListsNV) = (void(GLApi*)(GLsizei, const GLuint*))wglGetProcAddress("glDeleteCommandListsNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glIsCommandListNV) = (GLboolean(GLApi*)(GLuint))wglGetProcAddress("glIsCommandListNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glListDrawCommandsStatesClientNV) = (void(GLApi*)(GLuint, GLuint, const void**, const size_t*, const GLuint*, const GLuint*, GLuint))wglGetProcAddress("glListDrawCommandsStatesClientNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCommandListSegmentsNV) = (void(GLApi*)(GLuint, GLuint))wglGetProcAddress("glCommandListSegmentsNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCompileCommandListNV) = (void(GLApi*)(GLuint))wglGetProcAddress("glCompileCommandListNV"))) ++ n_failed_functions;
	if(!(GLFuncPtrName(glCallCommandListNV) = (void(GLApi*)(GLuint))wglGetProcAddress("glCallCommandListNV"))) ++ n_failed_functions;

#endif // __GENERATE_GL_NV_command_list

	GLEH_NV_command_list = !n_failed_functions && b_SupportedExtension("GL_NV_command_list");

	return n_failed_functions;
}

/*
 *								--- ~GL_NV_command_list ---
 */

/*
 *								=== ~CGL4ExtensionHandler ===
 */
