//-----------------------------------------------------------------------------
// Demo program
// Realtime object detection in video captured from camera or AVI file.
//-----------------------------------------------------------------------------


// STL
#include <iostream>
#include <string>
#include <sstream>
#include <cmath>
#include <vector>
#include <ctime>
// OpenCV
#include <cv.h>
#include <highgui.h>
// ABRuntime
#include "abr.h"


using namespace std;


/// Return reference image to OpenCV IplImage.
/// Does not copy the image nor allocate any memory at all!
TImageUChar IplImageReference(IplImage * image, bool vflip)
{
	if (vflip)
	{
		return TImageUChar(
			TImageUChar::TImageDataPtr(image->imageData)+((image->height-1)*image->widthStep),
			TSize(image->width, image->height),
			-image->widthStep);
	}
	else
	{
		return TImageUChar(
			TImageUChar::TImageDataPtr(image->imageData),
			TSize(image->width, image->height),
			image->widthStep);
	}
}


/// Draw rectangle in image
void rectArea(TImageUChar & image, TRect & rect)
{
	unsigned char * a, * b;

	a = image.row(rect.y)+rect.x;
	b = image.row(rect.y+rect.h-1)+rect.x;
	for (int x = rect.x; x < rect.x+rect.w && x < image.size().w; ++x, ++a, ++b)
	{
		if (rect.y < image.size().h) *a = 255;
		if ((rect.y+rect.h-1) < image.size().h) *b = 255;
	}

	a = image.row(rect.y)+rect.x;
	b = image.row(rect.y)+rect.x+rect.w-1;
	for (int y = rect.y; y < rect.y+rect.h && y < image.size().h; ++y, a+=image.yOffset(), b+=image.yOffset())
	{
		if (rect.x < image.size().w) *a = 255;
		if ((rect.x+rect.w-1) < image.size().w) *b = 255;
	}
}


vector<TRect> detections(1000); // vector for detection results


/// Detection code
void detect(IplImage * frame, TImagePyramid * pyramid, TClassifier * classifier)
{
	TImageUChar image = IplImageReference(frame, false); // prepare image for detector
	
	//-DETECTION-----------------------------------------------------
	// Insert image to pyramid - rescale and calculate integrals	
	pyramid->insert(image);

	// Perform detection
	int dets = scanImagePyramid(pyramid, classifier, detections);
	//---------------------------------------------------------------
	
	// Draw detected areas to image
	if (dets > 0)
	{
		vector<TRect>::iterator r = detections.begin();
		for (int d = 0 ; d < dets; ++r, ++d)
		{
			rectArea(image, *r);
		}
	}

	// some statistics stuff
	static time_t t0, t1;
	static float fps = 0.0f;

	// FPS calculation
	static int n = 0;
	++n;
	if (n % 8 == 0)
	{
		t1 = clock() - t0;
		fps = float(0.5 * fps + 0.5 * (1.0f / (float(t1) / 8000))); // do some smoothing
        float rejectionRate = 1.0 - (float(dets) / getSamplesCount());
		cout <<  fps << " fps (" << getAverageFeaturesPerSample() << " eps), "
             << getSamplesCount() << " samples, "
             << rejectionRate*100.0 << " % rejection rate \r"
             << flush;
		t0 = clock();
	}
	
	// draw image
	cvShowImage("detection", frame);
}


/// Help information
void printHelp()
{
	cout << "Realtime object detection demo" << endl << endl;
	cout << "demo -c filename [options]" << endl << endl;
	cout << "Options:" << endl;
	cout << "-c filename\tClassifier filename (waldboost format)" << endl;
	cout << "-t x\tSet last stage threshold to x (default 0.0)" << endl;
	cout << "-s x\tSet classifier window shift to x (2)" << endl;
	cout << "-f x\tSet pyramid scale factor to x (1.33)" << endl;
	cout << "-l x\tSet number of pyramid levels to x (8)" << endl;
	cout << "-h\tThis help" << endl;
	cout << "--video filename\tUse video instead of camera input" << endl;
	cout << "--output filename\tSave video (XviD codec is used)" << endl;
	cout << "--size w h\tSet video frame size to w x h px (512 384)" << endl;
	cout << "--linear\tUse linear interpolation (off)" << endl;
	cout << "--smooth\tSmooth rescale in pyramid (off) [not implemented yet]" << endl << endl;
	cout << "demo -c test.xml -t 4.5 --size 320 240 --linear" << endl;
}


int main(int argc, char ** argv)
{

	string classifierFile = "";
	string videoFile = "";
	string outvideoFile = "";
	float finalStageThreshold = 0.0f;
	int windowShift = 2;
	float pyramidFactor = 1.33f;
	int pyramidLevels = 8;
	bool interpolation = false;
	int frameWidth = 512;
	int frameHeight = 384;
	
	// parse arguments
	for (int arg = 0; arg < argc; ++arg)
	{
		if (string(argv[arg]) == "--help" || string(argv[arg]) == "-h")
		{
			printHelp();
			return 0;
		}
		if (string(argv[arg]) == "--classifier" || string(argv[arg]) == "-c")
		{
			//if (arg+1 < argc) break;
			classifierFile = argv[++arg];
			continue;
		}
		if (string(argv[arg]) == "--video")
		{
			//if (arg+1 < argc) break;
			videoFile = argv[++arg];
			continue;
		}
		if (string(argv[arg]) == "--output")
		{
			//if (arg+1 < argc) break;
			outvideoFile = argv[++arg];
			continue;
		}
		if (string(argv[arg]) == "--threshold" || string(argv[arg]) == "-t")
		{
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> finalStageThreshold;
			continue;
		}
		if (string(argv[arg]) == "--shift" || string(argv[arg]) == "-s")
		{
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> windowShift;
			continue;
		}
		if (string(argv[arg]) == "--factor" || string(argv[arg]) == "-f")
		{
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> pyramidFactor;
			continue;
		}
		if (string(argv[arg]) == "--levels" || string(argv[arg]) == "-l")
		{
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> pyramidLevels;
			continue;
		}
		if (string(argv[arg]) == "--linear")
		{
			interpolation = true;
			continue;
		}
		if (string(argv[arg]) == "--size")
		{
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> frameWidth;
			//if (arg+1 < argc) break;
			istringstream(argv[++arg]) >> frameHeight;
			continue;
		}
	}

	
	if (classifierFile == "") // No classifier given
    {
        printHelp();
        return 0;
    }
    
    TClassifier * classifier = loadStrongClassifier(classifierFile);

	if (!classifier)
	{
		cerr << "Cannot load classifier!" << endl;
		return 0;
	}

	classifier->threshold() = finalStageThreshold;
	classifier->shiftx() = windowShift;
	classifier->shifty() = windowShift;

	
	// Open camera capture	
	CvCapture * capture = 0;
	if (videoFile != "")
	{
		capture = cvCaptureFromAVI(videoFile.c_str());
	}
	else
	{
		capture = cvCaptureFromCAM(0); // Camera 0 is used!
	}

	if (!capture)
	{
		cerr << "Cannot capture video!" << endl;
		return 0;
	}

	
	CvVideoWriter * videoWriter = 0;
	if (outvideoFile != "")
	{
		videoWriter = cvCreateVideoWriter(outvideoFile.c_str(), CV_FOURCC('X','V','I','D'), 5, cvSize(frameWidth, frameHeight));
		
		if (!videoWriter)
		{
			cerr << "Cannot initialize video writer - no output will be saved." << endl;
		}
	}


	// detection window
	cvNamedWindow("detection", 1);

	// Create pyramid
	TImagePyramid * pyramid = new TImagePyramid(TSize(frameWidth, frameHeight), pyramidFactor, pyramidLevels);
	
	IplImage * origFrame = 0;
	IplImage * grayFrame = 0;
	IplImage * smallFrame = 0;

	if (capture)
    {
		
        // Capture from the camera.
        for (;;)
        {
            // Capture the frame and load it in IplImage
            if (!cvGrabFrame(capture)) break;
            origFrame = cvRetrieveFrame(capture);

            // If the frame does not exist, quit the loop
            if (!origFrame) break;

			//
			if (!grayFrame)
				grayFrame = cvCreateImage(cvSize(origFrame->width, origFrame->height), IPL_DEPTH_8U, 1);
			
			if (!smallFrame)
				smallFrame = cvCreateImage(cvSize(frameWidth, frameHeight), IPL_DEPTH_8U, 1);
			
			// convert to grayscale
			cvCvtColor(origFrame, grayFrame, CV_RGB2GRAY);
			// resample to smaller resolution
			cvResize(grayFrame, smallFrame, (interpolation) ? CV_INTER_LINEAR : CV_INTER_NN);
			
			// Flip frame if necessary
			cvFlip(smallFrame);

			cvSmooth(smallFrame, smallFrame, CV_GAUSSIAN);

			// detect and draw
			detect(smallFrame, pyramid, classifier);

			if (videoWriter)
			{
				cvWriteFrame(videoWriter, smallFrame);
			}

			// Wait for a while before proceeding to the next frame
            if (cvWaitKey(10) >= 0) break;
		}
		cvReleaseCapture(&capture);
		cvReleaseImage(&grayFrame);
		cvReleaseImage(&smallFrame);
		if (videoWriter)
			cvReleaseVideoWriter(&videoWriter);
	}

	delete classifier;
	delete pyramid;
}
