/*
 *   lrd_engine.cpp
 *   Local Rank Difference evaluation
 *   $Id: lrd_engine.cpp 9 2008-05-02 08:14:34Z andyman $
 *   ----
 *   Department of Computer Graphics and Multimedia
 *   Faculty of Information Technology, Brno University of Technology
 *   ----
 *   Naive LRD evaluation engine.
 */

#include "lrd_engine.h"
#include <iostream>

using namespace std;


// Set ptr to alphas for each stage
void initClassifier(TClassifier * classifier)
{
	float * stgAlpha = classifier->alpha;
    for (unsigned s = 0; s < classifier->stageCount; ++s, stgAlpha += classifier->alphaCount)
    {
		TStage & stage = classifier->stage[s];
		stage.alpha = stgAlpha;
        stage.offset = 0;
    }
}


void setClassifierImageSize(TClassifier * classifier, unsigned widthStep)
{
    for (unsigned s = 0; s < classifier->stageCount; ++s)
    {
		TStage & stage = classifier->stage[s];
        stage.offset = stage.y * widthStep + stage.x;
    }
}


/// Sum of regions in 3x3 grid.
/// Sums values in regions and stores the results in a vector.
static void sumRegions3x3(unsigned char * data, int w, int h, unsigned widthStep, int * v)
{
	unsigned blockStep = h * widthStep;
    widthStep -= w;

	// Prepare pointer array
    unsigned char * base[9] = {
		data, data+w, data+2*w,
		data+blockStep,data+blockStep+w,data+blockStep+2*w,
		data+2*blockStep,data+2*blockStep+w,data+2*blockStep+2*w,
	};

    for (int y = 0; y < h; ++y)
    {
        // go through all pixels in row and accumulate
		int x = 0;
        while (x < w)
        {
			for (int i = 0; i < 9; ++i)
			{
				v[i] += *base[i];
				++base[i];
			}
			++x;
        }
        // set pointers to next line 
		for (int i = 0; i < 9; ++i)
			base[i] += widthStep;
    }
}


/// Simple version of LRD evaluation.
/// Sums the pixels in the feature grid and calculates ranks of selected pixels.
static float evalLRDStageSimple(IplImage * image, unsigned smpOffset, TStage * stg)
{
	int values[9] = {0,0,0,0,0,0,0,0,0};
    // Get absolute address of feature in image
	unsigned char * base = (unsigned char*)(image->imageData + smpOffset + stg->offset);
	// Get sums in the feature blocks
    // There is no need to calculate the means because we care only about
    // rang of blocks not actual values 
    sumRegions3x3(base, stg->w, stg->h, image->widthStep, values);
	
	int countA = 0;
	int countB = 0;
	int valA = values[stg->A];
	int valB = values[stg->B];

	// calculate ranks
    int * data = values;
	for (int i = 0; i < 9; ++i, ++data)
	{
		if (valA > *data) ++countA;
		if (valB > *data) ++countB;
	}

    int lrd = countA - countB + 8;

	// return weak hypothesis response
    return stg->alpha[lrd];
}


int evalLRDClassifier(IplImage * image, unsigned smpOffset, TClassifier * classifier, float * response)
{
    *response = 0.0;
    // go through all stages and accumulate response
	for (TStage * stage = classifier->stage; stage < classifier->stage + classifier->stageCount; ++stage)
    {
        *response += evalLRDStageSimple(image, smpOffset, stage);
        // Test the waldboost threshold
        if (*response < stage->theta_b)
        {
            return 0;
        }
    }

    // All stages have been evaluated - test final stage threshold
    return (*response > classifier->threshold) ? 1 : 0;
}


unsigned scanImage(IplImage * image, TClassifier * classifier, TDetectionList * results, unsigned first)
{
    // Set pointer to position for first detection
    TDetection * iter = results->detections + first;

    unsigned rowOffset = 0;
    for (unsigned y = 0; y < image->height-classifier->height; y+=2, rowOffset+=2*image->widthStep)
    {
        for (unsigned x = 0; x < image->width-classifier->width; x+=2)
        {
            // Evaluate classifier
            float response;
            int positive = evalLRDClassifier(image, rowOffset+x, classifier, &response);
            if (positive) // write position of the detection
            {
                iter->response = response;
                iter->area = cvRect(x, y, classifier->width, classifier->height);
                ++results->count;
                ++iter;
                if (results->count >= results->maxCount)
					return results->count - first;
            }
        }
    }

    return results->count - first;
}
